package com.biz.crm.dms.business.costpool.credit.local.service.register;

import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowItemEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditOccupyEntity;
import com.biz.crm.dms.business.costpool.credit.local.model.CreditCashModelDto;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditCashFlowItemRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditCashFlowRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditCashRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditOccupyRepository;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditCashVoService;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditCashFlowExtendDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditOccupyCashDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditType;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.OccupyTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.register.CreditCashFlowRegister;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditCashFlowExtendVo;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditCashVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 授信资金占用流水拆分注册器实现(授信资金占用时导致授信资金流水变动,将资金拆分绑定到具体授信信息)
 * <p>
 * 适用场景描述:
 * 主要用于客户购买下单时使用授信资金的业务场景
 *
 * dms_credit_cash 都是正数，表示剩余
 * dms_credit_cash_flow和dms_credit_cash_flow_item 正数是增加可用，负数是减少可用
 * dms_credit_occupy 占用存负数，释放存正数
 *
 * @author pengxi
 * @date 2022/6/27
 */
@Component
public class CreditOccupyCashFlowRegisterImpl implements CreditCashFlowRegister<CreditOccupyCashDto> {

  @Autowired(required = false)
  private CreditCashVoService creditCashVoService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private CreditCashRepository creditCashRepository;
  @Autowired(required = false)
  private CreditOccupyRepository creditOccupyRepository;
  @Autowired(required = false)
  private CreditCashFlowRepository creditCashFlowRepository;
  @Autowired(required = false)
  private CreditCashFlowItemRepository creditCashFlowItemRepository;

  @Override
  public Class<CreditOccupyCashDto> getClazz() {
    return CreditOccupyCashDto.class;
  }

  @Override
  @Transactional
  public List<CreditCashVo> onCreate(CreditOccupyCashDto dto) {
    Validate.notBlank(dto.getOccupyType(), "占用或释放类型识别错误！");
    OccupyTypeEnum occupyTypeEnum = OccupyTypeEnum.getByDictCode(dto.getOccupyType());
    switch (occupyTypeEnum) {
      case OCCUPY:
        return this.onOccupy(dto);
      case RELEASE:
        return this.onRelease(dto);
      default:
        break;
    }
    return null;
  }

  public List<CreditCashVo> onOccupy(CreditOccupyCashDto dto) {
    Validate.isTrue(dto.getOperateAmount().compareTo(BigDecimal.ZERO) < 0
        , String.format("非业务资金扣减,无法执行[%s]操作", dto.getAdjustOperateName()));
    CreditCashModelDto modelDto = new CreditCashModelDto();
    modelDto.setCustomerCode(dto.getCustomerCode());
    modelDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    modelDto.setInEffectiveTime(Boolean.TRUE);
    Map<String, CreditCashVo> cashModelMap = this.creditCashVoService.findByCreditCashModelDto(modelDto);
    Validate.isTrue(cashModelMap != null && !cashModelMap.isEmpty(), "授信资金异常!");
    //排序将临时授信排在前普通授信排在后,且临时授信根据失效时间倒序排列(用于后续操作的扣款优先级逻辑)
    List<CreditCashVo> cashModelList = cashModelMap.values().stream().sorted((o1, o2) -> {
      Long sortNum1 = CreditType.NORMAL_CREDIT.getDictCode().equals(o1.getCreditType()) ? 0 : o1.getCreditEndTime().getTime();
      Long sortNum2 = CreditType.NORMAL_CREDIT.getDictCode().equals(o2.getCreditType()) ? 0 : o2.getCreditEndTime().getTime();
      return sortNum2.compareTo(sortNum1);
    }).collect(Collectors.toList());
    Validate.isTrue(!CollectionUtils.isEmpty(cashModelList), String.format("可用额度不足,无法执行[%s]操作", dto.getAdjustOperateName()));
    List<CreditCashFlowItemEntity> itemEntities = this.buildCashFlowItemEntities(dto, cashModelList);
    List<CreditCashEntity> cashEntities = this.buildCashFlowItemEntities(cashModelMap, itemEntities);
    this.creditCashFlowItemRepository.saveBatch(itemEntities);
    this.creditCashRepository.updateBatchById(cashEntities);
    return (List<CreditCashVo>) this.nebulaToolkitService.copyCollectionByWhiteList(cashEntities, CreditCashEntity.class
        , CreditCashVo.class, HashSet.class, ArrayList.class);
  }

  public List<CreditCashVo> onRelease(CreditOccupyCashDto dto) {
    CreditOccupyEntity creditOccupyEntity = this.creditOccupyRepository.findOccupyByOccupyTypeAndBusinessOrderCode(TenantUtils.getTenantCode(), OccupyTypeEnum.OCCUPY, dto.getBusinessOrderCode());
    Validate.notNull(creditOccupyEntity, "未找到授信占用信息，无法执行释放逻辑！");

    List<CreditCashFlowEntity> creditCashFlowEntities = this.creditCashFlowRepository.findByCashSerialNumbers(Collections.singletonList(creditOccupyEntity.getCashSerialNumber()), TenantUtils.getTenantCode());
    Validate.notEmpty(creditCashFlowEntities, "未找到授信占用资金流水信息，无法执行释放逻辑！");

    CreditCashFlowEntity creditCashFlowEntity = creditCashFlowEntities.get(0);
    List<CreditCashFlowItemEntity> creditCashFlowItemEntities = this.creditCashFlowItemRepository.findByCashFlowIds(Collections.singletonList(creditCashFlowEntity.getId()));
    Validate.notEmpty(creditCashFlowItemEntities, "未找到授信占用资金流水明细，无法执行释放逻辑！");
    List<String> creditIds = creditCashFlowItemEntities.stream().map(CreditCashFlowItemEntity::getCreditId).collect(Collectors.toList());
    // 更新授信资金表中的金额
    List<CreditCashEntity> cashEntities = this.creditCashRepository.findByCreditIds(creditIds);
    Map<String, List<CreditCashFlowItemEntity>> creditCashFlowItemMap =
        creditCashFlowItemEntities.stream().collect(Collectors.groupingBy(CreditCashFlowItemEntity::getCreditId));
    cashEntities.forEach(cash -> {
      List<CreditCashFlowItemEntity> itemEntities = creditCashFlowItemMap.get(cash.getCreditId());
      if (CollectionUtils.isEmpty(itemEntities)) {
        return;
      }
      // 操作金额（负数）
      BigDecimal itemAmount = itemEntities.stream().map(CreditCashFlowItemEntity::getItemAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
      // 占用
      cash.setOccupyAmount(cash.getOccupyAmount().add(itemAmount));
      // 可用
      cash.setCanUseAmount(cash.getCanUseAmount().subtract(itemAmount));
    });
    this.creditCashRepository.updateBatchById(cashEntities);

    // 把占用流水复制，金额取反后保存为释放流水
    creditCashFlowItemEntities.forEach(i->{
      i.setId(null);
      i.setItemAmount(BigDecimal.ZERO.subtract(i.getItemAmount()));
    });
    this.creditCashFlowItemRepository.saveBatch(creditCashFlowItemEntities);

    return (List<CreditCashVo>) this.nebulaToolkitService.copyCollectionByWhiteList(cashEntities, CreditCashEntity.class
        , CreditCashVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<CreditCashFlowExtendVo> onRequestByCreditCashFlowExtendDto(CreditCashFlowExtendDto dto) {
    if (Objects.isNull(dto)) {
      return Lists.newLinkedList();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    List<CreditOccupyEntity> entities = this.creditOccupyRepository.findByCreditCashFlowExtendDto(dto);
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newLinkedList();
    }
    return entities.stream().map(entity -> {
      CreditCashFlowExtendVo extendVo = new CreditCashFlowExtendVo();
      extendVo.setCashSerialNumber(entity.getCashSerialNumber());
      extendVo.setOrderCode(entity.getBusinessOrderCode());
      extendVo.setRemark(entity.getRemark());
      extendVo.setOrderType(entity.getOrderType());
      return extendVo;
    }).collect(Collectors.toList());
  }

  /**
   * 根据授信资金流水拆分明细构建变动后的授信资金更新实体集合
   *
   * @param cashModelMap 可用资金信息映射
   * @param itemEntities 授信资金流水拆分明细
   * @return 变动后的授信资金更新实体集合
   */
  private List<CreditCashEntity> buildCashFlowItemEntities(Map<String, CreditCashVo> cashModelMap, List<CreditCashFlowItemEntity> itemEntities) {
    return itemEntities.stream().map(creditCashFlowItemEntity -> {
      CreditCashVo cashVo = cashModelMap.get(creditCashFlowItemEntity.getCreditId());
      CreditCashEntity cashEntity = this.nebulaToolkitService.copyObjectByWhiteList(cashVo, CreditCashEntity.class, HashSet.class, ArrayList.class);
      // 占用传入的是负数
      cashEntity.setCanUseAmount(cashEntity.getCanUseAmount().add(creditCashFlowItemEntity.getItemAmount()));
      cashEntity.setOccupyAmount(cashEntity.getOccupyAmount().subtract(creditCashFlowItemEntity.getItemAmount()));
      return cashEntity;
    }).collect(Collectors.toList());
  }

  /**
   * 获取资金流水拆分详情列表
   * 扣减逻辑:
   * 1.优先扣减临时授信且即将过期的授信额度,最后扣减普通授信额度(cashModelList保证此种排序即可)
   * 2.使用waitAllotAmount记录待扣减额度
   * 3.金额扣减不足时抛出额度不足信息
   *
   * @param dto           参数dto
   * @param cashModelList 授信资金模板信息
   * @return 资金流水拆分详情列表
   */
  private List<CreditCashFlowItemEntity> buildCashFlowItemEntities(CreditOccupyCashDto dto, List<CreditCashVo> cashModelList) {
    List<CreditCashFlowItemEntity> entities = Lists.newArrayList();
    //转换为正数
    BigDecimal waitAllotAmount = BigDecimal.ZERO.subtract(dto.getOperateAmount());
    for (CreditCashVo vo : cashModelList) {
      //1.如果可用额度小于等于0(虽然理论上不会小于0),则不使用此授信信息额度
      //2.如果待分配金额小于等于0,表示分配完成,无需继续分配
      if (vo.getCanUseAmount().compareTo(BigDecimal.ZERO) <= 0 || waitAllotAmount.compareTo(BigDecimal.ZERO) <= 0) {
        continue;
      }
      CreditCashFlowItemEntity entity = new CreditCashFlowItemEntity();
      entity.setTenantCode(dto.getTenantCode());
      entity.setCreditId(vo.getCreditId());
      entity.setCashFlowId(dto.getId());
      //如果待分配金额小于当前授信信息可使用金额,则直接使用待分配金额(转换为负数存储),否则使用前授信信息可使用金额(转换为负数存储)
      entity.setItemAmount(waitAllotAmount.compareTo(vo.getCanUseAmount()) > 0 ? BigDecimal.ZERO.subtract(vo.getCanUseAmount()) : BigDecimal.ZERO.subtract(waitAllotAmount));
      entities.add(entity);
      waitAllotAmount = waitAllotAmount.add(entity.getItemAmount());
    }
    Validate.isTrue(waitAllotAmount.compareTo(BigDecimal.ZERO) <= 0, String.format("可用额度不足,无法执行[%s]操作", dto.getAdjustOperateName()));
    return entities;
  }
}
