package com.biz.crm.dms.business.costpool.credit.local.service.register;

import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowItemEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditPayEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditRefundEntity;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditCashFlowItemRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditCashFlowRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditCashRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditPayRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditRefundRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditRepository;
import com.biz.crm.dms.business.costpool.credit.local.service.helper.CreditServiceHelper;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditCashFlowExtendDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditRefundCashDto;
import com.biz.crm.dms.business.costpool.credit.sdk.register.CreditCashFlowRegister;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditCashFlowExtendVo;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditCashVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 授信资金支付使用后因业务原因需要退还资金流水拆分注册器实现(授信资金支付使用后因业务原因需要退还资金扣减时导致授信资金流水变动,将资金拆分绑定到具体授信信息)
 * <p>
 * 适用场景描述:
 * 主要用于客户购买,兑换后需要取消或者退款等释放支付使用授信资金的业务场景
 *
 * @author ning.zhang
 * @date 2021/12/16
 */
@Component
public class CreditRefundCashFlowRegisterImpl implements CreditCashFlowRegister<CreditRefundCashDto> {

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private CreditPayRepository creditPayRepository;
  @Autowired(required = false)
  private CreditRefundRepository creditRefundRepository;
  @Autowired(required = false)
  private CreditCashRepository creditCashRepository;
  @Autowired(required = false)
  private CreditCashFlowRepository creditCashFlowRepository;
  @Autowired(required = false)
  private CreditCashFlowItemRepository creditCashFlowItemRepository;
  @Autowired(required = false)
  private CreditServiceHelper creditServiceHelper;
  @Autowired(required = false)
  private CreditRepository creditRepository;

  @Override
  public Class<CreditRefundCashDto> getClazz() {
    return CreditRefundCashDto.class;
  }

  /**
   * 当资金流水信息创建时,更新授信资金信息和生成资金流水拆分明细
   * <p>
   * 1.校验单据支付记录,支付信息流水记录,资金流水明细记录以及资金信息
   * 2.当前退款只支持一次性退款操作,不支持分批次退款
   * 3.更新相关授信资金信息,生成资金流水明细
   * 4.持久化资金和流水明细信息
   *
   * @param dto 参数dto
   * @return 授信资金信息
   */
  @Override
  @Transactional
  public List<CreditCashVo> onCreate(CreditRefundCashDto dto) {
    String tenantCode = TenantUtils.getTenantCode();
    CreditPayEntity payEntity = this.creditPayRepository.findByBusinessOrderCode(dto.getBusinessOrderCode(), tenantCode);
    Validate.notNull(payEntity, "单据对应的支付信息不存在!");
    //获取已退款金额信息
    Map<String, BigDecimal> refundedMap = this.buildRefundedMap(dto.getBusinessOrderCode(), tenantCode);
    BigDecimal totalRefundedAmount = refundedMap.values().stream().reduce(BigDecimal.ZERO, BigDecimal::add);
    Validate.isTrue(payEntity.getPayAmount().compareTo(dto.getOperateAmount().add(totalRefundedAmount)) >= 0
        , "退款金额不能大于单据支付金额!");
    //获取单据资金流水记录
    List<CreditCashFlowEntity> flowEntities = this.creditCashFlowRepository
        .findByCashSerialNumbers(Lists.newArrayList(payEntity.getCashSerialNumber()), tenantCode);
    Validate.isTrue(!CollectionUtils.isEmpty(flowEntities) && flowEntities.size() == 1, "单据支付流水获取异常!");
    //获取单据资金流水明细记录
    List<CreditCashFlowItemEntity> flowItemEntities = this.creditCashFlowItemRepository
        .findByCashFlowIds(flowEntities.stream().map(CreditCashFlowEntity::getId).collect(Collectors.toList()));
    Validate.isTrue(!CollectionUtils.isEmpty(flowItemEntities), "单据支付流水明细获取异常!");
    //获取资金流水明细对应的资金信息
    List<CreditCashEntity> cashEntities = this.creditCashRepository.findByCreditIds(flowItemEntities.stream().map(CreditCashFlowItemEntity::getCreditId).collect(Collectors.toList()));
    Validate.isTrue(!CollectionUtils.isEmpty(cashEntities), "单据支付授信资金信息获取异常!");
    //持久化资金和流水明细信息
    this.saveCreditInfo(dto, flowItemEntities, cashEntities, refundedMap);
    return (List<CreditCashVo>) this.nebulaToolkitService.copyCollectionByWhiteList(cashEntities, CreditCashEntity.class
        , CreditCashVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 保存授信相关信息
   *
   * @param dto              参数dto
   * @param flowItemEntities 资金流水明细记录
   * @param cashEntities     资金信息
   */
  private void saveCreditInfo(CreditRefundCashDto dto, List<CreditCashFlowItemEntity> flowItemEntities,
                              List<CreditCashEntity> cashEntities, Map<String, BigDecimal> refundedMap) {
    //更新资金可用金额,生成资金流水拆分信息
    Map<String, CreditCashEntity> cashEntityMap = cashEntities.stream().collect(Collectors.toMap(CreditCashEntity::getCreditId, t -> t));
    List<CreditCashFlowItemEntity> refundFlowItemEntities = Lists.newArrayList();
    BigDecimal remainRefundAmount = BigDecimal.valueOf(dto.getOperateAmount().doubleValue());
    for (CreditCashFlowItemEntity flowItemEntity : flowItemEntities) {
      //无待退款金额,退出循环
      if (remainRefundAmount.compareTo(BigDecimal.ZERO) <= 0) {
        break;
      }
      CreditCashEntity cashEntity = cashEntityMap.get(flowItemEntity.getCreditId());
      Validate.notNull(cashEntity, "单据支付授信资金信息获取异常!");
      //支付金额分摊到对应授信的明细金额
      BigDecimal payAmount = BigDecimal.ZERO.subtract(flowItemEntity.getItemAmount());
      //已退款金额分摊到对应授信的明细金额
      BigDecimal refundedAmount = refundedMap.getOrDefault(flowItemEntity.getCreditId(), BigDecimal.ZERO);
      //对应授信已使用金额
      BigDecimal haveUseAmount = this.creditServiceHelper.calculateHaveUseAmount(cashEntity);
      //支付金额分摊到对应授信的明细,还能退款的金额
      BigDecimal canRefundAmount = payAmount.subtract(refundedAmount);
      //支付金额分摊到对应授信的明细,具体退款的金额
      BigDecimal refundAmount = remainRefundAmount.compareTo(canRefundAmount) >= 0 ? canRefundAmount : remainRefundAmount;
      if (haveUseAmount.compareTo(refundAmount) < 0) {
        CreditEntity creditEntity = this.creditRepository.getById(flowItemEntity.getCreditId());
        throw new IllegalArgumentException(String.format("授信[%s]已使用金额少于退款金额!",creditEntity.getCreditCode()));
      }
      cashEntity.setCanUseAmount(cashEntity.getCanUseAmount().add(refundAmount));
      CreditCashFlowItemEntity cashFlowItemEntity = new CreditCashFlowItemEntity();
      cashFlowItemEntity.setTenantCode(dto.getTenantCode());
      cashFlowItemEntity.setCreditId(flowItemEntity.getCreditId());
      cashFlowItemEntity.setItemAmount(refundAmount);
      cashFlowItemEntity.setCashFlowId(dto.getId());
      refundFlowItemEntities.add(cashFlowItemEntity);
      remainRefundAmount = remainRefundAmount.subtract(refundAmount);
    }
    Validate.isTrue(remainRefundAmount.compareTo(BigDecimal.ZERO) == 0, "单据支付退款金额异常!");
    //持久化资金信息,资金流水拆分信息
    this.creditCashFlowItemRepository.saveBatch(refundFlowItemEntities);
    this.creditCashRepository.updateBatchById(cashEntities);
  }

  /**
   * 构建已退款授信金额映射(key:授信ID,value:退款总金额)
   *
   * @param businessOrderCode 业务单据号
   * @param tenantCode        租户编码
   * @return 已退款授信金额映射
   */
  private Map<String, BigDecimal> buildRefundedMap(String businessOrderCode, String tenantCode) {
    Map<String, BigDecimal> refundedMap = Maps.newHashMap();
    List<CreditRefundEntity> refundEntities = this.creditRefundRepository.findByBusinessOrderCode(businessOrderCode, tenantCode);
    if (CollectionUtils.isEmpty(refundEntities)) {
      return refundedMap;
    }
    List<String> cashSerialNumbers = refundEntities.stream().map(CreditRefundEntity::getCashSerialNumber).collect(Collectors.toList());
    //获取退款资金流水记录
    List<CreditCashFlowEntity> flowEntities = this.creditCashFlowRepository
        .findByCashSerialNumbers(cashSerialNumbers, tenantCode);
    //获取退款资金流水明细记录
    List<CreditCashFlowItemEntity> flowItemEntities = this.creditCashFlowItemRepository
        .findByCashFlowIds(flowEntities.stream().map(CreditCashFlowEntity::getId).collect(Collectors.toList()));
    Validate.isTrue(!CollectionUtils.isEmpty(flowItemEntities), "退款资金流水明细获取异常!");
    refundedMap = flowItemEntities.stream().collect(Collectors.toMap(CreditCashFlowItemEntity::getCreditId, CreditCashFlowItemEntity::getItemAmount, BigDecimal::add));
    return refundedMap;
  }

  @Override
  public List<CreditCashFlowExtendVo> onRequestByCreditCashFlowExtendDto(CreditCashFlowExtendDto dto) {
    if (Objects.isNull(dto)) {
      return Lists.newLinkedList();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    List<CreditRefundEntity> entities = this.creditRefundRepository.findByCreditCashFlowExtendDto(dto);
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newLinkedList();
    }
    return entities.stream().map(entity -> {
      CreditCashFlowExtendVo extendVo = new CreditCashFlowExtendVo();
      extendVo.setCashSerialNumber(entity.getCashSerialNumber());
      extendVo.setRemark(entity.getRemark());
      extendVo.setOrderCode(entity.getBusinessOrderCode());
      extendVo.setOrderType(entity.getOrderType());
      return extendVo;
    }).collect(Collectors.toList());
  }
}
