package com.biz.crm.dms.business.costpool.credit.local.service.internal;

import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowEntity;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditCashFlowRepository;
import com.biz.crm.dms.business.costpool.credit.local.model.CreditCashModelDto;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditCashFlowService;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditCashVoService;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.AbstractCreditCashFlowDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditType;
import com.biz.crm.dms.business.costpool.credit.sdk.register.CreditCashFlowRegister;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditCashVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;

/**
 * 授信资金流水表服务实现类
 *
 * @author ning.zhang
 * @date 2021-12-15 16:48:27
 */
@Slf4j
@Service("creditCashFlowService")
public class CreditCashFlowServiceImpl implements CreditCashFlowService {

  private Map<Class, CreditCashFlowRegister> registerMap;
  @Autowired(required = false)
  private CreditCashFlowRepository creditCashFlowRepository;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private CreditCashVoService creditCashVoService;

  public CreditCashFlowServiceImpl(List<CreditCashFlowRegister> itemRegisters) {
    registerMap = Maps.newHashMap();
    if (CollectionUtils.isEmpty(itemRegisters)) {
      return;
    }
    itemRegisters.forEach(register -> {
      CreditCashFlowRegister itemRegister = registerMap.get(register.getClazz());
      Validate.isTrue(Objects.isNull(itemRegister), String.format("存在相同的授信资金流水注册器[%s]", register.getClazz()));
      registerMap.put(register.getClazz(), register);
    });
  }

  @Override
  @Transactional
  public CreditCashFlowEntity create(AbstractCreditCashFlowDto dto) {
    this.createValidation(dto);
    CreditCashFlowEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, CreditCashFlowEntity.class, HashSet.class, ArrayList.class);
    entity.setCashSerialNumber(UUID.randomUUID().toString());
    //持久化
    this.creditCashFlowRepository.save(entity);
    //保存资金流水信息
    dto.setId(entity.getId());
    CreditCashFlowRegister register = this.registerMap.get(dto.getClass());
    Validate.notNull(register, String.format("找不到[%s]对应的授信资金流水拆分注册器", dto.getClass()));
    List<CreditCashVo> cashList = register.onCreate(dto);
    this.updateAmount(dto, cashList);
    return entity;
  }

  /**
   * 更新流水相关资金信息
   *
   * @param dto      参数dto
   * @param cashList 和流水相关的变动后的资金信息
   */
  private void updateAmount(AbstractCreditCashFlowDto dto, List<CreditCashVo> cashList) {
    CreditCashModelDto modelDto = new CreditCashModelDto();
    modelDto.setCustomerCode(dto.getCustomerCode());
    Map<String, CreditCashVo> cashModelMap = this.creditCashVoService.findByCreditCashModelDto(modelDto);
    //替换掉变动的资金信息(当前获取出来的是变动前的数据,此次需要变动后的数据参与计算)
    for (CreditCashVo creditCashVo : cashList) {
      cashModelMap.put(creditCashVo.getCreditId(), creditCashVo);
    }
    CreditCashVo normalCreditCashVo = null;
    for (CreditCashVo creditCashVo : cashModelMap.values()) {
      if (CreditType.NORMAL_CREDIT.getDictCode().equals(creditCashVo.getCreditType())) {
        normalCreditCashVo = creditCashVo;
        break;
      }
    }
    Validate.notNull(normalCreditCashVo, "授信资金异常!");
    //更新长期授信资金和长期授信可用资金
    CreditCashFlowEntity updateEntity = new CreditCashFlowEntity();
    updateEntity.setId(dto.getId());
    updateEntity.setLongAmount(normalCreditCashVo.getCreditAmount());
    updateEntity.setLongCanUseAmount(normalCreditCashVo.getCanUseAmount());
    this.creditCashFlowRepository.updateById(updateEntity);
  }

  /**
   * 在创建CreditCashFlow模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(AbstractCreditCashFlowDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getAdjustOperateCode(), "缺失调整操作编码");
    Validate.notNull(dto.getAdjustTypeCode(), "缺失调整类型编码");
    Validate.notNull(dto.getAdjustOperateName(), "缺失调整操作名称");
    Validate.notNull(dto.getAdjustTypeName(), "缺失调整类型名称");
    Validate.notNull(dto.getOperateAmount(), "缺失操作额度");
  }
}
