package com.biz.crm.dms.business.costpool.credit.local.service.internal;

import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditFreezeEntity;
import com.biz.crm.dms.business.costpool.credit.local.model.CreditCashModelDto;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditFreezeRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditRepository;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditCashFlowService;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditCashVoService;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditFreezeService;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditFreezeCashDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditFreezeDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustOperateEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditType;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditCashVo;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Map;

/**
 * 授信冻结表服务实现类
 *
 * @author ning.zhang
 * @date 2021-12-14 20:10:01
 */
@Slf4j
@Service("creditFreezeService")
public class CreditFreezeServiceImpl implements CreditFreezeService {

  @Autowired(required = false)
  private CreditFreezeRepository creditFreezeRepository;
  @Autowired(required = false)
  private CreditCashVoService creditCashVoService;
  @Autowired(required = false)
  private CreditRepository creditRepository;
  @Autowired(required = false)
  private CreditCashFlowService creditCashFlowService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  /**
   * 创建授信冻结信息
   * 1.普通授信根据冻结金额和已冻结金额判定是冻结还是解冻操作
   * 1.1 若当前传入冻结金额多于已冻结金额,此次操作为冻结操作,实际冻结金额为传入冻结金额-已冻结金额,并记录保存且生成资金变动明细
   * 1.2 若当前传入冻结金额少于已冻结金额,此次操作为解冻操作,实际解冻金额为传入冻结金额-已冻结金额,并记录保存且生成资金变动明细
   * 1.3 传入冻结金额的上限为授信额度,若传入冻结金额多于可用额度,其差额会在上账回款,订单退款等释放的金额,补全冻结金额.
   * 1.4 1.3中的补全实际上就是在资金变动明细里面因为此次冻结额度大于可用额度导致统计明细出现余额负数,后续的释放金额明细,就是补全负数.
   * <p>
   * 2.临时授信只针对授信额度全部冻结或者释放,
   * 2.1 临时授信冻结不关心可用余额,其逻辑与1.3,1.4类似
   * 2.2 其效果表现为只要冻结,整个临时授信不可用,可理解为额度小于等于0.
   * 2.3 并记录保存且生成资金变动明细
   *
   * @param dto 参数dto
   */
  @Override
  @Transactional
  public void create(CreditFreezeDto dto) {
    this.createValidation(dto);
    CashAdjustOperateEnum adjustOperate= CashAdjustOperateEnum.FREEZE_UNFREEZE;
    CashAdjustTypeEnum adjustType= CashAdjustTypeEnum.FREEZE_UNFREEZE;
    CreditFreezeCashDto cashFlowDto = new CreditFreezeCashDto();
    cashFlowDto.setCreditId(dto.getCreditId());
    cashFlowDto.setOperateAmount(BigDecimal.ZERO.subtract(dto.getFreezeAmount()));
    cashFlowDto.setCustomerCode(dto.getCustomerCode());
    cashFlowDto.setAdjustOperateCode(adjustOperate.getDictCode());
    cashFlowDto.setAdjustOperateName(adjustOperate.getValue());
    cashFlowDto.setAdjustTypeCode(adjustType.getDictCode());
    cashFlowDto.setAdjustTypeName(adjustType.getValue());
    CreditCashFlowEntity cashFlowEntity = this.creditCashFlowService.create(cashFlowDto);
    CreditFreezeEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, CreditFreezeEntity.class, HashSet.class, ArrayList.class);
    entity.setCashSerialNumber(cashFlowEntity.getCashSerialNumber());
    this.creditFreezeRepository.save(entity);
  }


  /**
   * 在创建CreditFreeze模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(CreditFreezeDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setId(null);
    Validate.notBlank(dto.getCreditId(), "缺失授信ID！");
    Validate.notBlank(dto.getCreditType(), "缺失授信类型");
    Validate.notNull(dto.getFreezeFlag(), "缺失冻结标志!");
    CreditEntity creditEntity = this.creditRepository.getById(dto.getCreditId());
    Validate.notNull(creditEntity, "授信信息不存在!");
    Validate.isTrue(creditEntity.getCreditType().equals(dto.getCreditType()), "授信类型不匹配");
    if (CreditType.TEMPORARY_CREDIT.getDictCode().equals(creditEntity.getCreditType())) {
      Validate.isTrue(ProcessStatusEnum.PASS.getDictCode().equals(creditEntity.getProcessStatus())
          , "临时授信未审批通过,无法冻结/解冻");
    }
    Date nowDate = new Date();
    Validate.isTrue(creditEntity.getCreditStartTime().before(nowDate)
        && creditEntity.getCreditEndTime().after(nowDate), "授信不在有效期内,无法冻结/解冻");
    dto.setCustomerCode(creditEntity.getCustomerCode());
    //金额校验
    CreditCashModelDto modelDto = new CreditCashModelDto();
    modelDto.setCreditIdList(Lists.newArrayList(dto.getCreditId()));
    Map<String, CreditCashVo> cashModelMap = this.creditCashVoService.findByCreditCashModelDto(modelDto);
    CreditCashVo modelVo = cashModelMap.get(dto.getCreditId());
    Validate.notNull(modelVo, "无可操作授信额度!");
    if (CreditType.NORMAL_CREDIT.getDictCode().equals(dto.getCreditType())) {
      Validate.notNull(dto.getFreezeAmount(), "缺失冻结额度");
      Validate.isTrue(dto.getFreezeAmount().compareTo(BigDecimal.ZERO) >= 0, "冻结额度必须大于0");
      Validate.isTrue(dto.getFreezeAmount().compareTo(creditEntity.getCreditAmount()) <= 0, "冻结额度必须小于授信金额");
      dto.setFreezeAmount(dto.getFreezeAmount().subtract(modelVo.getFreezeAmount()));
    } else {
      //临时授信冻结针对授信额度全部冻结,解冻则是全部解冻
      if (Boolean.TRUE.equals(dto.getFreezeFlag())) {
        Validate.isTrue(modelVo.getFreezeAmount().compareTo(BigDecimal.ZERO) == 0, "授信信息已冻结,无法继续冻结!");
        dto.setFreezeAmount(creditEntity.getCreditAmount());
      } else {
        Validate.isTrue(modelVo.getFreezeAmount().compareTo(BigDecimal.ZERO) > 0, "授信信息已解冻,无法继续解冻!");
        dto.setFreezeAmount(BigDecimal.ZERO.subtract(modelVo.getFreezeAmount()));
      }
    }
  }
}
