package com.biz.crm.dms.business.costpool.credit.local.service.internal;

import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditOccupyEntity;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditOccupyRepository;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditCashFlowService;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditOccupyService;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditOccupyCashDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditOccupyDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustOperateEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditOrderTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.OccupyTypeEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.Date;

/**
 * 授信占用实体类服务实现类
 *
 * @author ning.zhang
 * @date 2022-01-20 15:20:08
 */
@Slf4j
@Service
public class CreditOccupyServiceImpl implements CreditOccupyService {

  @Autowired(required = false)
  private CreditOccupyRepository creditOccupyRepository;
  @Autowired(required = false)
  private CreditCashFlowService creditCashFlowService;

  /**
   * 创建占用信息
   * 1.校验授信占用参数有效性
   * 2.并记录保存且生成资金变动明细
   *
   * @param dto 参数dto
   */
  @Override
  @Transactional
  public void occupy(CreditOccupyDto dto) {
    this.createValidation(dto);
    Validate.isTrue(CashAdjustTypeEnum.OCCUPY_USE.getDictCode().equals(dto.getAdjustTypeCode()), "不支持的授信调整类型！");
    Validate.isTrue(CashAdjustOperateEnum.OCCUPY_USE.getDictCode().equals(dto.getAdjustOperateCode()), "不支持的授信调整操作！");
    CashAdjustOperateEnum adjustOperate = CashAdjustOperateEnum.getByDictCode(dto.getAdjustOperateCode());
    CashAdjustTypeEnum adjustType = CashAdjustTypeEnum.getByDictCode(dto.getAdjustTypeCode());
    CreditOccupyEntity entity = new CreditOccupyEntity();
    entity.setBusinessOrderCode(dto.getFromCode());
    entity.setOccupyAmount(BigDecimal.ZERO.subtract(dto.getAmount()));
    entity.setTenantCode(TenantUtils.getTenantCode());
    entity.setOrderType(CreditOrderTypeEnum.SALE_ORDER.getDictCode());
    entity.setOccupyType(OccupyTypeEnum.OCCUPY.getDictCode());
    CreditOccupyCashDto cashFlowDto = new CreditOccupyCashDto();
    cashFlowDto.setOperateAmount(BigDecimal.ZERO.subtract(dto.getAmount()));
    cashFlowDto.setCustomerCode(dto.getCustomerCode());
    cashFlowDto.setBusinessOrderCode(dto.getFromCode());
    cashFlowDto.setAdjustOperateCode(adjustOperate.getDictCode());
    cashFlowDto.setAdjustOperateName(adjustOperate.getValue());
    cashFlowDto.setAdjustTypeCode(adjustType.getDictCode());
    cashFlowDto.setAdjustTypeName(adjustType.getValue());
    cashFlowDto.setOccupyType(OccupyTypeEnum.OCCUPY.getDictCode());
    //创建资金流水
    CreditCashFlowEntity cashFlowEntity = this.creditCashFlowService.create(cashFlowDto);
    entity.setCashSerialNumber(cashFlowEntity.getCashSerialNumber());
    //持久化占用记录
    this.creditOccupyRepository.save(entity);
  }

  @Override
  @Transactional
  public void release(CreditOccupyDto dto) {
    this.createValidation(dto);
    Validate.isTrue(CashAdjustTypeEnum.OCCUPY_RELEASE.getDictCode().equals(dto.getAdjustTypeCode()), "不支持的授信调整类型！");
    Validate.isTrue(CashAdjustOperateEnum.OCCUPY_RELEASE.getDictCode().equals(dto.getAdjustOperateCode()), "不支持的授信调整操作！");
    CashAdjustOperateEnum adjustOperate = CashAdjustOperateEnum.getByDictCode(dto.getAdjustOperateCode());
    CashAdjustTypeEnum adjustType = CashAdjustTypeEnum.getByDictCode(dto.getAdjustTypeCode());
    CreditOccupyEntity entity = new CreditOccupyEntity();
    entity.setBusinessOrderCode(dto.getFromCode());
    entity.setOccupyAmount(dto.getAmount());
    entity.setTenantCode(TenantUtils.getTenantCode());
    entity.setOrderType(CreditOrderTypeEnum.SALE_ORDER.getDictCode());
    entity.setOccupyType(OccupyTypeEnum.RELEASE.getDictCode());
    CreditOccupyCashDto cashFlowDto = new CreditOccupyCashDto();
    cashFlowDto.setOperateAmount(dto.getAmount());
    cashFlowDto.setCustomerCode(dto.getCustomerCode());
    cashFlowDto.setBusinessOrderCode(dto.getFromCode());
    cashFlowDto.setAdjustOperateCode(adjustOperate.getDictCode());
    cashFlowDto.setAdjustOperateName(adjustOperate.getValue());
    cashFlowDto.setAdjustTypeCode(adjustType.getDictCode());
    cashFlowDto.setAdjustTypeName(adjustType.getValue());
    cashFlowDto.setOccupyType(OccupyTypeEnum.RELEASE.getDictCode());
    //创建资金流水
    CreditCashFlowEntity cashFlowEntity = this.creditCashFlowService.create(cashFlowDto);
    entity.setCashSerialNumber(cashFlowEntity.getCashSerialNumber());
    //持久化释放记录
    this.creditOccupyRepository.save(entity);
  }

  /**
   * 在创建CreditOccupy模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(CreditOccupyDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    Validate.notBlank(dto.getAdjustOperateCode(), "缺失调整操作编码！");
    Validate.notBlank(dto.getAdjustTypeCode(), "缺失调整类型编码！");
    Validate.notBlank(dto.getCustomerCode(), "缺失客户编码！");
    Validate.notBlank(dto.getFromCode(), "缺失来源编码!");
    Validate.notNull(dto.getAmount(), "缺失金额!");
    Validate.isTrue(dto.getAmount().compareTo(BigDecimal.ZERO) > 0, "金额必须大于0");
  }
}
