package com.biz.crm.dms.business.costpool.credit.local.service.internal;

import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditPayEntity;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditPayRepository;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditCashFlowService;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditPayService;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditPayCashDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditPayDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustOperateEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditOrderTypeEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.Date;

/**
 * 授信支付实体类服务实现类
 *
 * @author ning.zhang
 * @date 2022-01-20 15:20:08
 */
@Slf4j
@Service("creditPayService")
public class CreditPayServiceImpl implements CreditPayService {

  @Autowired(required = false)
  private CreditPayRepository creditPayRepository;
  @Autowired(required = false)
  private CreditCashFlowService creditCashFlowService;

  /**
   * 创建支付信息
   * 1.校验授信支付参数有效性
   * 2.并记录保存且生成资金变动明细
   *
   * @param dto 参数dto
   */
  @Override
  @Transactional
  public void create(CreditPayDto dto) {
    this.createValidation(dto);
    CashAdjustOperateEnum adjustOperate = CashAdjustOperateEnum.getByDictCode(dto.getAdjustOperateCode());
    CashAdjustTypeEnum adjustType = CashAdjustTypeEnum.getByDictCode(dto.getAdjustTypeCode());
    CreditPayEntity entity = new CreditPayEntity();
    entity.setBusinessOrderCode(dto.getFromCode());
    entity.setPayAmount(dto.getAmount());
    entity.setTenantCode(TenantUtils.getTenantCode());
    entity.setOrderType(CreditOrderTypeEnum.SALE_ORDER.getDictCode());
    CreditPayCashDto cashFlowDto = new CreditPayCashDto();
    cashFlowDto.setOperateAmount(BigDecimal.ZERO.subtract(dto.getAmount()));
    cashFlowDto.setCustomerCode(dto.getCustomerCode());
    cashFlowDto.setFromCode(dto.getFromCode());
    cashFlowDto.setAdjustOperateCode(adjustOperate.getDictCode());
    cashFlowDto.setAdjustOperateName(adjustOperate.getValue());
    cashFlowDto.setAdjustTypeCode(adjustType.getDictCode());
    cashFlowDto.setAdjustTypeName(adjustType.getValue());
    //创建资金流水
    CreditCashFlowEntity cashFlowEntity = this.creditCashFlowService.create(cashFlowDto);
    entity.setCashSerialNumber(cashFlowEntity.getCashSerialNumber());
    //持久化支付记录
    this.creditPayRepository.save(entity);
  }

  /**
   * 在创建CreditPay模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(CreditPayDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    Validate.notBlank(dto.getAdjustOperateCode(), "缺失调整操作编码！");
    Validate.notBlank(dto.getAdjustTypeCode(), "缺失调整类型编码！");
    Validate.notBlank(dto.getCustomerCode(), "缺失客户编码！");
    Validate.notBlank(dto.getFromCode(), "缺失来源编码!");
    Validate.notNull(dto.getAmount(), "缺失金额!");
    Validate.isTrue(dto.getAmount().compareTo(BigDecimal.ZERO) > 0, "金额必须大于0");
    Validate.isTrue(CashAdjustTypeEnum.CREDIT_REDUCE.getDictCode().equals(dto.getAdjustTypeCode()), "不支持的授信调整类型！");
    Validate.isTrue(CashAdjustOperateEnum.ORDER.getDictCode().equals(dto.getAdjustOperateCode()), "不支持的授信调整操作！");
  }
}
