package com.biz.crm.dms.business.costpool.credit.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditEntity;
import com.biz.crm.dms.business.costpool.credit.local.model.CreditFileModelDto;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditWriteOffVoRepository;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditFileService;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditService;
import com.biz.crm.dms.business.costpool.credit.sdk.constant.CreditConstant;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditFileDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditRepayNoticeDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditFileType;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditType;
import com.biz.crm.dms.business.costpool.credit.sdk.event.CreditEventListener;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditEventVo;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditFileVo;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditWriteOffVo;
import com.biz.crm.dms.business.interaction.sdk.dto.base.ScopeDto;
import com.biz.crm.dms.business.interaction.sdk.dto.notice.NoticeDto;
import com.biz.crm.dms.business.interaction.sdk.enums.NoticePopupType;
import com.biz.crm.dms.business.interaction.sdk.enums.ScopeType;
import com.biz.crm.dms.business.interaction.sdk.service.notice.NoticeVoService;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 授信表服务实现类
 *
 * @author ning.zhang
 * @date 2021-12-14 20:08:47
 */
@Slf4j
@Service("creditService")
public class CreditServiceImpl implements CreditService {

  @Autowired(required = false)
  private CreditRepository creditRepository;
  @Autowired(required = false)
  private CustomerVoService customerVoService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private CreditFileService creditFileService;
  @Autowired(required = false)
  private List<CreditEventListener> listeners;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired(required = false)
  private CreditWriteOffVoRepository creditWriteOffVoRepository;
  @Autowired(required = false)
  private NoticeVoService noticeVoService;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Override
  @Transactional
  public void create(CreditDto dto) {
    this.createValidation(dto);

    Map<String, CustomerVo> customerMap = this.buildCustomerMap(dto.getCustomerCodeList());
    List<CreditEntity> entities = dto.getCustomerCodeList().stream().map(customerCode -> {
      CustomerVo customerVo = customerMap.get(customerCode);
      Validate.notNull(customerVo, String.format("客户[%s]信息不存在!", customerCode));
      CreditEntity creditEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, CreditEntity.class, HashSet.class, ArrayList.class);
      creditEntity.setCustomerCode(customerCode);
      creditEntity.setCustomerName(customerVo.getCustomerName());
      creditEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      creditEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      creditEntity.setCreditCode(this.generateCodeService.generateCode("SX", 1).get(0));
      creditEntity.setProcessStatus(CreditType.TEMPORARY_CREDIT.getDictCode().equals(creditEntity.getCreditType())
              ? ProcessStatusEnum.PREPARE.getDictCode() : ProcessStatusEnum.PASS.getDictCode());
      //持久化
      this.creditRepository.save(creditEntity);
      if (CreditType.TEMPORARY_CREDIT.getDictCode().equals(creditEntity.getCreditType()) && Boolean.TRUE.equals(dto.getSubmitProcess())) {
        dto.setCreditCode(creditEntity.getCreditCode());
        dto.setId(creditEntity.getId());
        creditEntity.setProcessNumber(this.commitProcess(dto));
        creditEntity.setProcessStatus(ProcessStatusEnum.COMMIT.getDictCode());
        this.creditRepository.updateById(creditEntity);
      }
      return creditEntity;
    }).collect(Collectors.toList());

    //创建扩展信息
    this.createExtInfo(dto, entities);
  }


  @Override
  @Transactional
  public void update(CreditDto dto) {
    this.updateValidation(dto);
    CreditEntity creditEntity = this.creditRepository.getById(dto.getId());
    Validate.notNull(creditEntity, "授信信息不存在!");
    CreditEntity updateEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, CreditEntity.class, HashSet.class, ArrayList.class);
    updateEntity.setProcessStatus(creditEntity.getProcessStatus());
    this.creditRepository.updateById(updateEntity);
    CreditFileModelDto fileModelDto = new CreditFileModelDto();
    fileModelDto.setBusinessId(creditEntity.getId());
    fileModelDto.setFileType(CreditFileType.CREDIT.getDictCode());
    fileModelDto.setFileList(dto.getFileList());
    //更新保存文件信息
    this.creditFileService.update(Lists.newArrayList(fileModelDto));
    if (CreditType.TEMPORARY_CREDIT.getDictCode().equals(creditEntity.getCreditType()) && Boolean.TRUE.equals(dto.getSubmitProcess())) {
      dto.setId(creditEntity.getId());
      dto.setCreditCode(creditEntity.getCreditCode());
      updateEntity.setProcessNumber(this.commitProcess(dto));
      updateEntity.setProcessStatus(ProcessStatusEnum.COMMIT.getDictCode());
      this.creditRepository.updateById(updateEntity);
    }
    //修改授信事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      CreditEventVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(creditEntity, CreditEventVo.class, HashSet.class, ArrayList.class);
      CreditEventVo newVo = this.nebulaToolkitService.copyObjectByBlankList(updateEntity, CreditEventVo.class, HashSet.class, ArrayList.class);
      newVo.setCustomerCode(oldVo.getCustomerCode());
      listeners.forEach(listener -> {
        listener.onUpdate(oldVo, newVo);
      });
    }
  }

  @Override
  @Transactional
  public void enableBatch(List<String> customerCodes) {
    Validate.isTrue(!CollectionUtils.isEmpty(customerCodes), "授信客户编码集合");
    List<CreditEntity> entities = this.creditRepository.findByCustomerCodes(customerCodes, null
            , EnableStatusEnum.DISABLE.getCode(), TenantUtils.getTenantCode());
    Validate.isTrue(!CollectionUtils.isEmpty(entities), "无可启用授信信息");
    List<String> ids = entities.stream().map(CreditEntity::getId).collect(Collectors.toList());
    this.creditRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    //启用授信事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<CreditEventVo> voList = (List<CreditEventVo>) nebulaToolkitService.copyCollectionByWhiteList(entities, CreditEntity.class
              , CreditEventVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onEnable(voList);
      });
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> customerCodes) {
    Validate.isTrue(!CollectionUtils.isEmpty(customerCodes), "缺失授信客户编码集合");
    List<CreditEntity> entities = this.creditRepository.findByCustomerCodes(customerCodes, null
            , EnableStatusEnum.ENABLE.getCode(), TenantUtils.getTenantCode());
    Validate.isTrue(!CollectionUtils.isEmpty(entities), "无可禁用授信信息");
    List<String> ids = entities.stream().map(CreditEntity::getId).collect(Collectors.toList());
    this.creditRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    //禁用授信事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<CreditEventVo> voList = (List<CreditEventVo>) nebulaToolkitService.copyCollectionByWhiteList(entities, CreditEntity.class
              , CreditEventVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onDisable(voList);
      });
    }
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失授信id");
    List<CreditEntity> entities = this.creditRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(ids) && entities.size() == ids.size(), "数据删除个数不匹配");
    entities.forEach(creditEntity -> {
      Validate.isTrue(CreditType.TEMPORARY_CREDIT.getDictCode().equals(creditEntity.getCreditType())
              , String.format("当前授信[%s],非临时授信无法删除", creditEntity.getCreditCode()));
      Validate.isTrue(ProcessStatusEnum.PREPARE.getDictCode().equals(creditEntity.getProcessStatus())
              , String.format("当前临时授信[%s],非待提交状态无法删除", creditEntity.getCreditCode()));
    });
    this.creditRepository.updateDelFlagByIds(ids);
  }

  @Override
  public void sendNotice(CreditRepayNoticeDto dto) {
    Validate.notNull(dto, "信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    //获取待核销的客户授信信息
    List<CreditWriteOffVo> creditWriteOffList = this.creditWriteOffVoRepository.findByCreditRepayNoticeDto(dto);
    Validate.isTrue(!CollectionUtils.isEmpty(creditWriteOffList), "没有需要还款的客户");
    //构建授信待还款公告
    List<NoticeDto> noticeDtoList = creditWriteOffList.stream().map(creditWriteOffVo -> {
      NoticeDto noticeDto = this.nebulaToolkitService.copyObjectByWhiteList(dto, NoticeDto.class, HashSet.class, ArrayList.class);
      noticeDto.setContent(String.format(CreditConstant.CREDIT_REPLAY_NOTICE_CONTENT, creditWriteOffVo.getWaitWriteOffAmount()));
      if (Boolean.TRUE.equals(dto.getIndexPopup())) {
        noticeDto.setPopupType(NoticePopupType.READ_NOT_POPUP.getDictCode());
      }
      ScopeDto scopeDto = new ScopeDto();
      scopeDto.setContainFlag(Boolean.TRUE);
      scopeDto.setScopeCode(creditWriteOffVo.getCustomerCode());
      scopeDto.setScopeType(ScopeType.CUSTOMER.name());
      noticeDto.setScopeList(Lists.newArrayList(scopeDto));
      return noticeDto;
    }).collect(Collectors.toList());
    //保存公告信息
    this.noticeVoService.create(noticeDtoList);
  }

  /**
   * 创建扩展信息
   *
   * @param dto      参数dto
   * @param entities 授信信息实体列表
   */
  private void createExtInfo(CreditDto dto, List<CreditEntity> entities) {
    List<CreditFileModelDto> fileModelDtoList = Lists.newArrayList();
    List<CreditEventVo> eventVoList = Lists.newArrayList();
    entities.forEach(creditEntity -> {
      CreditFileModelDto fileModelDto = new CreditFileModelDto();
      fileModelDto.setBusinessId(creditEntity.getId());
      fileModelDto.setFileType(CreditFileType.CREDIT.getDictCode());
      fileModelDto.setFileList(dto.getFileList());
      fileModelDtoList.add(fileModelDto);
      CreditEventVo creditEventVo = this.nebulaToolkitService.copyObjectByWhiteList(creditEntity, CreditEventVo.class, HashSet.class, ArrayList.class);
      if (!CollectionUtils.isEmpty(dto.getFileList())) {
        List<CreditFileVo> list = (List<CreditFileVo>) this.nebulaToolkitService.copyCollectionByWhiteList(dto.getFileList()
                , CreditFileDto.class, CreditFileVo.class, HashSet.class, ArrayList.class);
        creditEventVo.setFileList(list);
      }
      eventVoList.add(creditEventVo);
    });
    //更新保存文件信息
    this.creditFileService.update(fileModelDtoList);
    //新增授信事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      listeners.forEach(listener -> {
        listener.onCreate(eventVoList);
      });
    }
  }

  /**
   * 构建客户信息映射(key:客户编码,value:客户信息)
   *
   * @param customerCodes 客户编码集合
   * @return 客户信息映射
   */
  private Map<String, CustomerVo> buildCustomerMap(List<String> customerCodes) {
    Map<String, CustomerVo> resultMap = Maps.newHashMap();
    if (CollectionUtils.isEmpty(customerCodes)) {
      return resultMap;
    }
    List<CustomerVo> customerList = this.customerVoService.findByCustomerCodes(customerCodes);
    if (CollectionUtils.isEmpty(customerList)) {
      return resultMap;
    }
    return customerList.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, t -> t, (key1, key2) -> key2));
  }

  /**
   * 在创建Credit模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(CreditDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getCustomerCodeList()), "缺失客户编码");
    Validate.notBlank(dto.getCreditType(), "缺失授信类型");
    Validate.notNull(dto.getCreditStartTime(), "缺失授信开始时间");
    Validate.notNull(dto.getCreditEndTime(), "缺失授信结束时间");
    Validate.notNull(dto.getRepayEndTime(), "缺失还款截止时间");
    Validate.isTrue(dto.getCreditStartTime().before(dto.getCreditEndTime()), "授信开始时间必须小于授信结束时间");
    Validate.isTrue(dto.getCreditEndTime().getTime() <= dto.getRepayEndTime().getTime(), "授信还款截止时间不能小于授信结束时间");
    Validate.notNull(dto.getCreditAmount(), "缺失授信额度");
    Validate.isTrue(dto.getCreditAmount().compareTo(BigDecimal.ZERO) > 0, "授信额度需大于0");
    List<String> scopeTypeList = Arrays.stream(CreditType.values()).map(CreditType::getDictCode).collect(Collectors.toList());
    Validate.isTrue(scopeTypeList.contains(dto.getCreditType()), "不支持的授信类型!");
    List<CreditEntity> creditEntities = this.creditRepository
            .findByCustomerCodes(dto.getCustomerCodeList(), CreditType.NORMAL_CREDIT.getDictCode(), null, dto.getTenantCode());
    if (CreditType.NORMAL_CREDIT.getDictCode().equals(dto.getCreditType())) {
      Validate.isTrue(CollectionUtils.isEmpty(creditEntities)
              , String.format("客户[%s]已开通授信,无法继续开通!", CollectionUtils.isEmpty(creditEntities) ? "" : creditEntities.get(0).getCustomerName()));
    } else {
      Map<String, CreditEntity> entityMap = creditEntities.stream().collect(Collectors.toMap(CreditEntity::getCustomerCode, t -> t));
      Date nowDate = new Date();
      dto.getCustomerCodeList().forEach(customerCode -> {
        CreditEntity creditEntity = entityMap.get(customerCode);
        Validate.notNull(creditEntity, String.format("客户[%s]未开通普通授信,请先开通普通授信", customerCode));
        Validate.isTrue(creditEntity.getCreditStartTime().getTime() <= dto.getCreditStartTime().getTime(), "临时授信开始时间不能小于普通授信开始时间");
        Validate.isTrue(creditEntity.getCreditEndTime().getTime() >= dto.getCreditEndTime().getTime(), "临时授信结束时间不能大于普通授信结束时间");
        Validate.isTrue(creditEntity.getRepayEndTime().getTime() >= dto.getRepayEndTime().getTime(), "临时授信还款时间不能大于普通授信还款时间");
        Validate.isTrue(creditEntity.getCreditEndTime().getTime() >= nowDate.getTime(), "普通授信已过期,无法开通临时授信");
      });
    }
  }

  /**
   * 在修改Credit模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(CreditDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getCreditType(), "缺失授信类型");
    Validate.notNull(dto.getCreditStartTime(), "缺失授信开始时间");
    Validate.notNull(dto.getCreditEndTime(), "缺失授信结束时间");
    Validate.notNull(dto.getRepayEndTime(), "缺失还款截止时间");
    Validate.isTrue(dto.getCreditStartTime().before(dto.getCreditEndTime()), "授信开始时间必须小于授信结束时间");
    Validate.isTrue(dto.getCreditEndTime().getTime() <= dto.getRepayEndTime().getTime(), "授信还款截止时间不能小于授信结束时间");
    Validate.notNull(dto.getCreditAmount(), "缺失授信额度");
    Validate.isTrue(dto.getCreditAmount().compareTo(BigDecimal.ZERO) > 0, "授信额度需大于0");
    if (CreditType.NORMAL_CREDIT.getDictCode().equals(dto.getCreditType())) {
      List<CreditEntity> temporaryCreditEntities = this.creditRepository
              .findByCustomerCodes(Lists.newArrayList(dto.getCustomerCode()), CreditType.TEMPORARY_CREDIT.getDictCode(), null, dto.getTenantCode());
      if (!CollectionUtils.isEmpty(temporaryCreditEntities)) {
        temporaryCreditEntities.forEach(creditEntity -> {
          Validate.isTrue(dto.getCreditStartTime().getTime() <= (creditEntity.getCreditStartTime().getTime())
                  , "存在临时授信开始时间小于当前普通授信开始时间");
          Validate.isTrue(dto.getCreditEndTime().getTime() >= (creditEntity.getCreditEndTime().getTime())
                  , "存在临时授信结束时间大于当前普通授信结束时间");
          Validate.isTrue(dto.getCreditEndTime().getTime() >= (creditEntity.getCreditEndTime().getTime())
                  , "存在临时授信还款时间大于当前普通授信还款时间");
        });
      }
    } else {
      List<CreditEntity> normalCreditEntities = this.creditRepository
              .findByCustomerCodes(Lists.newArrayList(dto.getCustomerCode()), CreditType.NORMAL_CREDIT.getDictCode(), null, dto.getTenantCode());
      Validate.isTrue(!CollectionUtils.isEmpty(normalCreditEntities) && normalCreditEntities.size() == 1
              , "临时授信对应的普通授信异常");
      CreditEntity creditEntity = normalCreditEntities.get(0);
      Validate.isTrue(creditEntity.getCreditStartTime().getTime() <= dto.getCreditStartTime().getTime(), "临时授信开始时间不能小于普通授信开始时间");
      Validate.isTrue(creditEntity.getCreditEndTime().getTime() >= dto.getCreditEndTime().getTime(), "临时授信结束时间不能大于普通授信结束时间");
      Validate.isTrue(creditEntity.getRepayEndTime().getTime() >= dto.getRepayEndTime().getTime(), "临时授信还款时间不能大于普通授信还款时间");
    }
  }


  @Autowired(required = false)
  private ProcessBusinessService processBusinessService;

  /**
   * 临时授信提交工作流进行审批，提交成功返回流程实例ID，提交失败则抛出异常
   *
   * @param dto 授信请求DTO
   */
  private String commitProcess(CreditDto dto) {
    ProcessBusinessDto businessProcessDto = dto.getProcessBusiness();
    businessProcessDto.setBusinessNo(dto.getCreditCode());
    businessProcessDto.setBusinessFormJson(JsonUtils.obj2JsonString(dto));
    businessProcessDto.setBusinessCode(CreditConstant.CREDIT_PROCESS_NAME);
    ProcessBusinessVo processBusinessVo = processBusinessService.processStart(businessProcessDto);
    return processBusinessVo.getProcessNo();
  }

  /**
   * 商城客户申请临时授信
   *
   * @param dto
   */
  @Override
  @Transactional
  public void handleTemporaryCreditApply(CreditDto dto) {
    String customerCode = dto.getCustomerCode();
    String customerName = dto.getCustomerName();
    if (StringUtils.isBlank(customerCode)) {
      return;
    }
    List<String> customerList = Lists.newArrayList();
    customerList.add(customerCode);
    dto.setCustomerCodeList(customerList);
    dto.setCreditType(CreditType.TEMPORARY_CREDIT.getDictCode());
    //校验
    this.createValidation(dto);
    CreditEntity creditEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, CreditEntity.class, HashSet.class, ArrayList.class);
    creditEntity.setCustomerCode(customerCode);
    creditEntity.setCustomerName(customerName);
    creditEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    creditEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    creditEntity.setCreditCode(this.generateCodeService.generateCode("SX", 1).get(0));
    creditEntity.setProcessStatus(ProcessStatusEnum.COMMIT.getDictCode());
    //持久化
    this.creditRepository.save(creditEntity);
    List<CreditEntity> entities = Lists.newArrayList();
    entities.add(creditEntity);
    //创建扩展信息
    this.createExtInfo(dto, entities);
    //提交审批
    dto.setId(creditEntity.getId());
    dto.setCreditCode(creditEntity.getCreditCode());
    ProcessBusinessDto processBusiness = dto.getProcessBusiness();
    processBusiness.setProcessKey(defaultProcessKey);
    processBusiness.setProcessTitle(customerName + "商城发起临时授信审批流程");
    creditEntity.setProcessNumber(this.commitProcess(dto));
    this.creditRepository.updateById(creditEntity);
  }

  @Value("${crm.business.credit.process-key:}")
  private String defaultProcessKey;
}
