package com.biz.crm.dms.business.costpool.credit.local.service.internal;

import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditWriteOffEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditWriteOffItemEntity;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditCashRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditWriteOffItemRepository;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditCashFlowService;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditWriteOffService;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditEntity;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditWriteOffRepository;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditWriteOffCashDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditWriteOffDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditWriteOffItemCashDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditWriteOffOrderDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustOperateEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustTypeEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 授信核销表服务实现类
 *
 * @author ning.zhang
 * @date 2021-12-21 15:27:04
 */
@Slf4j
@Service("creditWriteOffService")
public class CreditWriteOffServiceImpl implements CreditWriteOffService {

  @Autowired(required = false)
  private CreditWriteOffRepository creditWriteOffRepository;
  @Autowired(required = false)
  private CreditCashRepository creditCashRepository;
  @Autowired(required = false)
  private CreditRepository creditRepository;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private CreditWriteOffItemRepository creditWriteOffItemRepository;
  @Autowired(required = false)
  private CreditCashFlowService creditCashFlowService;

  /**
   * 创建授信核销
   * <p>
   * 1.校验参数有效性
   * 2.获取授信资金信息映射(key:授信ID,value:授信资金信息)
   * 3.获取授信信息映射(key:授信ID,value:授信信息)
   * 4.拆分单据金额到具体授信信息,生成授信核销记录明细
   * 5.生成资金流水并持久化核销信息和核销记录明细信息
   *
   * @param dto 参数dto
   */
  @Override
  @Transactional
  public void create(CreditWriteOffDto dto) {
    this.createValidation(dto);
    List<String> creditIds = Lists.newArrayList();
    dto.getWriteOffInfoList().forEach(infoDto -> creditIds.addAll(infoDto.getCreditIds()));
    List<CreditCashEntity> cashEntities = this.creditCashRepository.findByCreditIds(creditIds);
    Validate.isTrue(!CollectionUtils.isEmpty(cashEntities), "授信资金信息不存在!");
    List<CreditEntity> creditEntities = this.creditRepository.listByIds(creditIds);
    Validate.isTrue(!CollectionUtils.isEmpty(creditEntities), "授信信息不存在!");
    Map<String, CreditCashEntity> cashEntityMap = cashEntities.stream().collect(Collectors.toMap(CreditCashEntity::getCreditId, t -> t));
    Map<String, CreditEntity> creditEntityMap = creditEntities.stream().collect(Collectors.toMap(CreditEntity::getId, t -> t));

    //拆分单据金额到具体授信信息,生成授信核销记录明细
    List<CreditWriteOffItemEntity> writeOffItemEntities = Lists.newArrayList();
    dto.getWriteOffInfoList().forEach(infoDto -> {
      //将待核销的授信按还款时间倒序排列,依次核销
      List<CreditEntity> list = infoDto.getCreditIds().stream().map(creditId -> {
        CreditEntity creditEntity = creditEntityMap.get(creditId);
        Validate.notNull(creditEntity, "授信信息不存在!");
        Validate.isTrue(creditEntity.getCustomerCode().equals(dto.getCustomerCode()), "授信信息与客户信息不匹配!");
        Validate.isTrue(creditEntity.getCreditEndTime().before(new Date()), String.format("授信信息[%s],正在使用中,无法核销!", creditEntity.getCreditCode()));
        return creditEntity;
      }).sorted((o1, o2) -> o2.getRepayEndTime().compareTo(o1.getRepayEndTime())).collect(Collectors.toList());
      //拆分单据分摊核销金额到具体的授信信息上
      writeOffItemEntities.addAll(this.buildCreditWriteOffItem(cashEntityMap, infoDto.getOrderList(), list));
    });
    this.save(dto, writeOffItemEntities);
  }

  /**
   * 生成资金流水并持久化核销信息和核销记录明细信息
   * 1.统计金额,生成资金流水请求dto
   * 2.创建资金流水
   * 3.持久化核销记录和核销明细信息
   *
   * @param dto                  参数dto
   * @param writeOffItemEntities 核销记录明细信息
   */
  private void save(CreditWriteOffDto dto, List<CreditWriteOffItemEntity> writeOffItemEntities) {
    CreditWriteOffEntity writeOffEntity = new CreditWriteOffEntity();
    writeOffEntity.setCustomerCode(dto.getCustomerCode());
    writeOffEntity.setTenantCode(dto.getTenantCode());
    writeOffEntity.setWriteOffAmount(BigDecimal.ZERO);
    Map<String, CreditWriteOffItemCashDto> itemCashMap = Maps.newHashMap();
    writeOffItemEntities.forEach(itemEntity -> {
      writeOffEntity.setWriteOffAmount(writeOffEntity.getWriteOffAmount().add(itemEntity.getWriteOffItemAmount()));
      CreditWriteOffItemCashDto itemCashDto = itemCashMap.get(itemEntity.getCreditId());
      if (Objects.nonNull(itemCashDto)) {
        itemCashDto.setWriteOffAmount(itemCashDto.getWriteOffAmount().add(itemEntity.getWriteOffItemAmount()));
      } else {
        itemCashDto = new CreditWriteOffItemCashDto();
        itemCashDto.setWriteOffAmount(itemEntity.getWriteOffItemAmount());
        itemCashDto.setCreditId(itemEntity.getCreditId());
        itemCashMap.put(itemEntity.getCreditId(), itemCashDto);
      }
    });
    CashAdjustOperateEnum adjustOperate = CashAdjustOperateEnum.CREDIT_WRITE_OFF;
    CashAdjustTypeEnum adjustType = CashAdjustTypeEnum.CREDIT_WRITE_OFF;
    CreditWriteOffCashDto cashFlowDto = new CreditWriteOffCashDto();
    cashFlowDto.setWriteOffItemList(Lists.newArrayList(itemCashMap.values()));
    cashFlowDto.setCustomerCode(dto.getCustomerCode());
    cashFlowDto.setOperateAmount(writeOffEntity.getWriteOffAmount());
    cashFlowDto.setAdjustOperateCode(adjustOperate.getDictCode());
    cashFlowDto.setAdjustOperateName(adjustOperate.getValue());
    cashFlowDto.setAdjustTypeCode(adjustType.getDictCode());
    cashFlowDto.setAdjustTypeName(adjustType.getValue());
    CreditCashFlowEntity cashFlowEntity = this.creditCashFlowService.create(cashFlowDto);
    writeOffEntity.setCashSerialNumber(cashFlowEntity.getCashSerialNumber());
    this.creditWriteOffRepository.save(writeOffEntity);
    writeOffItemEntities.forEach(creditWriteOffItemEntity -> creditWriteOffItemEntity.setWriteOffId(writeOffEntity.getId()));
    this.creditWriteOffItemRepository.saveBatch(writeOffItemEntities);
  }

  /**
   * 构建核销记录明细
   * 1.根据排序后的授信信息,分配核销单据中的金额
   * 2.依次循环授信信息,获取到授信信息的待核销金额
   * 3.循环单据信息,分配核销金额
   * 3.1 如果单据剩余分配金额大于待核销金额,单据剩余分配金额减去待核销金额作为单据剩余金额,参与下一个授信信息核销.跳出单据循坏,继续下一个授信核销
   * 3.2 如果单据剩余分配金额小于等于待核销金额,则单据剩余金额消耗完毕移除单据(此单据不在参与后续分配),继续下一个单据循环
   * 3.3 如果待核销金额为0,分配结束,跳出单据循环,继续下一个授信信息循环
   * 3.4 生成授信核销记录明细
   *
   * @param cashEntityMap 授信资金映射信息(key:授信ID,value:授信资金信息)
   * @param orderDtoList  核销单据信息
   * @param list          待核销授信信息
   * @return 核销记录明细
   */
  private List<CreditWriteOffItemEntity> buildCreditWriteOffItem(Map<String, CreditCashEntity> cashEntityMap
      , List<CreditWriteOffOrderDto> orderDtoList, List<CreditEntity> list) {
    List<CreditWriteOffItemEntity> writeOffItemEntities = Lists.newArrayList();
    for (CreditEntity creditEntity : list) {
      CreditCashEntity cashEntity = cashEntityMap.get(creditEntity.getId());
      Validate.notNull(cashEntity, "授信资金信息不存在!");
      BigDecimal waitWriteOffAmount = cashEntity.getCreditAmount().subtract(cashEntity.getCanUseAmount()).subtract(cashEntity.getFreezeAmount());
      Validate.isTrue(waitWriteOffAmount.compareTo(BigDecimal.ZERO) > 0, String.format("授信信息[%s],待核销金额为0不需要核销!", creditEntity.getCreditCode()));
      Iterator<CreditWriteOffOrderDto> iterator = orderDtoList.iterator();
      while (iterator.hasNext()) {
        CreditWriteOffOrderDto orderDto = iterator.next();
        CreditWriteOffItemEntity writeOffItemEntity = this.nebulaToolkitService
            .copyObjectByWhiteList(orderDto, CreditWriteOffItemEntity.class, HashSet.class, ArrayList.class);
        writeOffItemEntity.setId(null);
        writeOffItemEntity.setCreditId(creditEntity.getId());
        writeOffItemEntity.setOrderAmount(orderDto.getAmount());
        if (orderDto.getRemainAmount().compareTo(waitWriteOffAmount) > 0) {
          orderDto.setRemainAmount(orderDto.getRemainAmount().subtract(waitWriteOffAmount));
          writeOffItemEntity.setWriteOffItemAmount(waitWriteOffAmount);
          writeOffItemEntities.add(writeOffItemEntity);
          //分配完毕,设置为0,结束分配
          waitWriteOffAmount = BigDecimal.ZERO;
        } else {
          waitWriteOffAmount = waitWriteOffAmount.subtract(orderDto.getRemainAmount());
          writeOffItemEntity.setWriteOffItemAmount(orderDto.getRemainAmount());
          writeOffItemEntities.add(writeOffItemEntity);
          iterator.remove();
        }
        //当待核销金额等于0时,跳出单据金额分配循坏
        if (waitWriteOffAmount.compareTo(BigDecimal.ZERO) == 0) {
          break;
        }
      }
    }
    Validate.isTrue(CollectionUtils.isEmpty(orderDtoList), "单据总金额超出待核销总金额,无法核销");
    return writeOffItemEntities;
  }

  /**
   * 在创建CreditWriteOff模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(CreditWriteOffDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getCustomerCode(), "缺失客户编码");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getWriteOffInfoList()), "缺失核销详细信息!");
    List<String> orderCodes = Lists.newArrayList();
    dto.getWriteOffInfoList().forEach(infoDto -> {
      Validate.isTrue(!CollectionUtils.isEmpty(infoDto.getCreditIds()), "缺失授信ID!");
      Validate.isTrue(!CollectionUtils.isEmpty(infoDto.getOrderList()), "缺失核销单据信息");
      infoDto.getOrderList().forEach(orderDto -> {
        Validate.notBlank(orderDto.getOrderType(), "缺失单据类型");
        Validate.notBlank(orderDto.getOrderCode(), "缺失单据编码");
        Validate.notNull(orderDto.getAmount(), "缺失金额");
        Validate.isTrue(orderDto.getAmount().compareTo(BigDecimal.ZERO) > 0, "金额需大于0");
        orderDto.setRemainAmount(orderDto.getAmount());
        orderDto.setTenantCode(dto.getTenantCode());
        orderCodes.add(orderDto.getOrderCode());
      });
    });
    List<CreditWriteOffItemEntity> itemEntities = creditWriteOffItemRepository.findByOrderCodes(orderCodes, dto.getTenantCode());
    Validate.isTrue(CollectionUtils.isEmpty(itemEntities), String.format("单据[%s]已经参与过核销,无法继续参与核销"
        , CollectionUtils.isEmpty(itemEntities) ? "" : itemEntities.get(0).getOrderCode()));
  }
}
