package com.biz.crm.dms.business.costpool.credit.local.service.notifier;

import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditRecordEntity;
import com.biz.crm.dms.business.costpool.credit.local.model.CreditFileModelDto;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditRecordRepository;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditCashFlowService;
import com.biz.crm.dms.business.costpool.credit.local.service.CreditFileService;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditCreateCashDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditFileDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditUpdateCashDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustOperateEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditFileType;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditType;
import com.biz.crm.dms.business.costpool.credit.sdk.event.CreditEventListener;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditEventVo;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditFileVo;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;

/**
 * 授信记录实现授信事件监听
 *
 * @author ning.zhang
 * @date 2021/12/16
 */
@Component
public class CreditRecordEventListenerImpl implements CreditEventListener {

  @Autowired(required = false)
  private CreditRecordRepository creditRecordRepository;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private CreditCashFlowService creditCashFlowService;
  @Autowired(required = false)
  private CreditFileService creditFileService;

  /**
   * 监听授信信息创建事件,保存相关授信记录
   * 1.遍历授信创建事件VO
   * 2.创建授信资金流水
   * 3.转换授信记录信息文件信息
   * 4.持久化授信记录和相关文件信息
   *
   * @param voList 新增的授信信息集合
   */
  @Override
  @Transactional
  public void onCreate(List<CreditEventVo> voList) {
    Validate.isTrue(!CollectionUtils.isEmpty(voList), "新增的授信信息不能为空!");
    List<CreditFileModelDto> fileModelDtoList = Lists.newArrayList();
    voList.forEach(creditEventVo -> {
      //需要审批通过之后才能加入明细
      if (!ProcessStatusEnum.PASS.getDictCode().equals(creditEventVo.getProcessStatus())) {
        return;
      }
      CashAdjustOperateEnum adjustOperate = CreditType.NORMAL_CREDIT.getDictCode().equals(creditEventVo.getCreditType())
          ? CashAdjustOperateEnum.OPEN_CREDIT : CashAdjustOperateEnum.TEMPORARY_CREDIT;
      CashAdjustTypeEnum adjustType = CreditType.NORMAL_CREDIT.getDictCode().equals(creditEventVo.getCreditType())
          ? CashAdjustTypeEnum.OPEN_CREDIT : CashAdjustTypeEnum.TEMPORARY_CREDIT;
      CreditRecordEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(creditEventVo, CreditRecordEntity.class, HashSet.class, ArrayList.class);
      entity.setCreditId(creditEventVo.getId());
      entity.setId(null);
      CreditCreateCashDto cashFlowDto = new CreditCreateCashDto();
      cashFlowDto.setCreditId(creditEventVo.getId());
      cashFlowDto.setCreditType(creditEventVo.getCreditType());
      cashFlowDto.setOperateAmount(creditEventVo.getCreditAmount());
      cashFlowDto.setCustomerCode(creditEventVo.getCustomerCode());
      cashFlowDto.setAdjustOperateCode(adjustOperate.getDictCode());
      cashFlowDto.setAdjustOperateName(adjustOperate.getValue());
      cashFlowDto.setAdjustTypeCode(adjustType.getDictCode());
      cashFlowDto.setAdjustTypeName(adjustType.getValue());
      CreditCashFlowEntity cashFlowEntity = this.creditCashFlowService.create(cashFlowDto);
      entity.setCashSerialNumber(cashFlowEntity.getCashSerialNumber());
      //持久化授信记录
      this.creditRecordRepository.save(entity);
      CreditFileModelDto fileModelDto = new CreditFileModelDto();
      fileModelDto.setFileType(CreditFileType.CREDIT_RECORD.getDictCode());
      fileModelDto.setBusinessId(entity.getId());
      if (!CollectionUtils.isEmpty(creditEventVo.getFileList())) {
        List<CreditFileDto> list = (List<CreditFileDto>) this.nebulaToolkitService.copyCollectionByWhiteList(creditEventVo.getFileList()
            , CreditFileVo.class, CreditFileDto.class, HashSet.class, ArrayList.class);
        fileModelDto.setFileList(list);
      }
      fileModelDtoList.add(fileModelDto);
    });
    //更新保存文件信息
    if (!CollectionUtils.isEmpty(fileModelDtoList)) {
      this.creditFileService.update(fileModelDtoList);
    }
  }

  /**
   * 监听授信信息更新事件,保存相关授信记录
   * 1.创建授信资金流水
   * 2.转换授信记录信息文件信息
   * 3.持久化授信记录和相关文件信息
   *
   * @param oldVo 修改之前的vo
   * @param newVo 修改之后的vo
   */
  @Override
  @Transactional
  public void onUpdate(CreditEventVo oldVo, CreditEventVo newVo) {
    Validate.isTrue(Objects.nonNull(oldVo) && Objects.nonNull(newVo), "更新前和更新后授信信息不能为空!");
    //需要审批通过之后才能加入明细
    if (!ProcessStatusEnum.PASS.getDictCode().equals(newVo.getProcessStatus())) {
      return;
    }
    CashAdjustOperateEnum adjustOperate = CashAdjustOperateEnum.CHANGE_CREDIT;
    CashAdjustTypeEnum adjustType = CashAdjustTypeEnum.CHANGE_CREDIT;
    CreditRecordEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(newVo, CreditRecordEntity.class, HashSet.class, ArrayList.class);
    entity.setCreditId(newVo.getId());
    entity.setId(null);
    CreditUpdateCashDto cashFlowDto = new CreditUpdateCashDto();
    cashFlowDto.setCreditId(newVo.getId());
    cashFlowDto.setOperateAmount(newVo.getCreditAmount().subtract(oldVo.getCreditAmount()));
    cashFlowDto.setCustomerCode(newVo.getCustomerCode());
    cashFlowDto.setAdjustOperateCode(adjustOperate.getDictCode());
    cashFlowDto.setAdjustOperateName(adjustOperate.getValue());
    cashFlowDto.setAdjustTypeCode(adjustType.getDictCode());
    cashFlowDto.setAdjustTypeName(adjustType.getValue());
    CreditCashFlowEntity cashFlowEntity = this.creditCashFlowService.create(cashFlowDto);
    entity.setCashSerialNumber(cashFlowEntity.getCashSerialNumber());
    //持久化授信记录
    this.creditRecordRepository.save(entity);
    CreditFileModelDto fileModelDto = new CreditFileModelDto();
    fileModelDto.setFileType(CreditFileType.CREDIT_RECORD.getDictCode());
    fileModelDto.setBusinessId(entity.getId());
    if (!CollectionUtils.isEmpty(newVo.getFileList())) {
      List<CreditFileDto> list = (List<CreditFileDto>) this.nebulaToolkitService.copyCollectionByWhiteList(newVo.getFileList()
          , CreditFileVo.class, CreditFileDto.class, HashSet.class, ArrayList.class);
      fileModelDto.setFileList(list);
    }
    //更新保存文件信息
    this.creditFileService.update(Lists.newArrayList(fileModelDto));
  }

  @Override
  public void onDisable(List<CreditEventVo> voList) {
    //do nothing
  }

  @Override
  public void onEnable(List<CreditEventVo> voList) {
    //do nothing
  }

  /**
   * 监听授信信息审批事件,保存相关授信记录
   * 1.遍历授信审批事件VO
   * 2.创建授信资金流水
   * 3.转换授信记录信息文件信息
   * 4.持久化授信记录和相关文件信息
   *
   * @param voList 审批的授信信息集合
   */
  @Override
  @Transactional
  public void onApprove(List<CreditEventVo> voList) {
    Validate.isTrue(!CollectionUtils.isEmpty(voList), "审批的授信信息不能为空!");
    List<CreditFileModelDto> fileModelDtoList = Lists.newArrayList();
    voList.forEach(creditEventVo -> {
      //需要审批通过之后才能加入明细
      if (!ProcessStatusEnum.PASS.getDictCode().equals(creditEventVo.getProcessStatus())) {
        return;
      }
      CashAdjustOperateEnum adjustOperate = CreditType.NORMAL_CREDIT.getDictCode().equals(creditEventVo.getCreditType())
          ? CashAdjustOperateEnum.OPEN_CREDIT : CashAdjustOperateEnum.TEMPORARY_CREDIT;
      CashAdjustTypeEnum adjustType = CreditType.NORMAL_CREDIT.getDictCode().equals(creditEventVo.getCreditType())
          ? CashAdjustTypeEnum.OPEN_CREDIT : CashAdjustTypeEnum.TEMPORARY_CREDIT;
      CreditRecordEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(creditEventVo, CreditRecordEntity.class, HashSet.class, ArrayList.class);
      entity.setId(null);
      entity.setCreditId(creditEventVo.getId());
      CreditCreateCashDto cashFlowDto = new CreditCreateCashDto();
      cashFlowDto.setCreditId(creditEventVo.getId());
      cashFlowDto.setCreditType(creditEventVo.getCreditType());
      cashFlowDto.setOperateAmount(creditEventVo.getCreditAmount());
      cashFlowDto.setCustomerCode(creditEventVo.getCustomerCode());
      cashFlowDto.setAdjustOperateCode(adjustOperate.getDictCode());
      cashFlowDto.setAdjustOperateName(adjustOperate.getValue());
      cashFlowDto.setAdjustTypeCode(adjustType.getDictCode());
      cashFlowDto.setAdjustTypeName(adjustType.getValue());
      CreditCashFlowEntity cashFlowEntity = this.creditCashFlowService.create(cashFlowDto);
      entity.setCashSerialNumber(cashFlowEntity.getCashSerialNumber());
      //持久化授信记录
      this.creditRecordRepository.save(entity);
      CreditFileModelDto fileModelDto = new CreditFileModelDto();
      fileModelDto.setFileType(CreditFileType.CREDIT_RECORD.getDictCode());
      fileModelDto.setBusinessId(entity.getId());
      if (!CollectionUtils.isEmpty(creditEventVo.getFileList())) {
        List<CreditFileDto> list = (List<CreditFileDto>) this.nebulaToolkitService.copyCollectionByWhiteList(creditEventVo.getFileList()
            , CreditFileVo.class, CreditFileDto.class, HashSet.class, ArrayList.class);
        fileModelDto.setFileList(list);
      }
      fileModelDtoList.add(fileModelDto);
    });
    //更新保存文件信息
    if (!CollectionUtils.isEmpty(fileModelDtoList)) {
      this.creditFileService.update(fileModelDtoList);
    }
  }
}
