package com.biz.crm.dms.business.costpool.credit.local.service.register;

import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowItemEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditDownUpAccountEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditFileEntity;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditCashFlowItemRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditCashRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditDownUpAccountRepository;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditCashFlowExtendDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditDownUpAccountCashDto;
import com.biz.crm.dms.business.costpool.credit.sdk.register.CreditCashFlowRegister;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditCashFlowExtendVo;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditCashVo;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditFileVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 授信回款上账-资金流水注册器实现
 * <p>
 * <p>
 * 适用场景描述:
 * 主要用于回款上账,更新授信资金(可用金额)保存流水拆分明细
 *
 * @author ning.zhang
 * @date 2021/12/16
 */
@Component
public class CreditDownUpAccountCashFlowRegisterImpl implements CreditCashFlowRegister<CreditDownUpAccountCashDto> {

  @Autowired(required = false)
  private CreditCashFlowItemRepository creditCashFlowItemRepository;
  @Autowired(required = false)
  private CreditCashRepository creditCashRepository;
  @Autowired(required = false)
  private CreditDownUpAccountRepository creditDownUpAccountRepository;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Class<CreditDownUpAccountCashDto> getClazz() {
    return CreditDownUpAccountCashDto.class;
  }

  @Override
  @Transactional
  public List<CreditCashVo> onCreate(CreditDownUpAccountCashDto dto) {
    Validate.notBlank(dto.getCreditId(), "缺失授信ID");
    CreditCashEntity nowCashEntity = creditCashRepository.findByCreditId(dto.getCreditId());
    Validate.notNull(nowCashEntity,"授信资金异常!");
    //负数表示占用可用余额,正数表示释放使用金额
    if (dto.getOperateAmount().compareTo(BigDecimal.ZERO) > 0) {
      //上账金额+可用余额(因为超可用余额冻结可能为负)+冻结金额不能大于授信总额度
      Validate.isTrue(nowCashEntity.getFreezeAmount().add(nowCashEntity.getOccupyAmount()).add(dto.getOperateAmount()).add(nowCashEntity.getCanUseAmount())
              .compareTo(nowCashEntity.getCreditAmount()) <= 0, String.format("回款/上账金额超过可释放最大额度,无法执行[%s]操作", dto.getAdjustOperateName()));
    } else {
      Validate.isTrue(dto.getOperateAmount().add(nowCashEntity.getCanUseAmount()).compareTo(BigDecimal.ZERO) >= 0
          , String.format("回款/上账金额超过可使用最大额度,无法执行[%s]操作", dto.getAdjustOperateName()));
    }
    //更新资金可用额度和冻结额度
    nowCashEntity.setCanUseAmount(nowCashEntity.getCanUseAmount().add(dto.getOperateAmount()));
    this.creditCashRepository.updateById(nowCashEntity);
    //持久化资金流水拆分信息
    CreditCashFlowItemEntity cashFlowItemEntity = new CreditCashFlowItemEntity();
    cashFlowItemEntity.setTenantCode(dto.getTenantCode());
    cashFlowItemEntity.setCreditId(dto.getCreditId());
    cashFlowItemEntity.setItemAmount(dto.getOperateAmount());
    cashFlowItemEntity.setCashFlowId(dto.getId());
    this.creditCashFlowItemRepository.save(cashFlowItemEntity);
    return Lists.newArrayList(this.nebulaToolkitService.copyObjectByWhiteList(nowCashEntity, CreditCashVo.class, HashSet.class, ArrayList.class));
  }

  @Override
  public List<CreditCashFlowExtendVo> onRequestByCreditCashFlowExtendDto(CreditCashFlowExtendDto dto) {
    if (Objects.isNull(dto)) {
      return Lists.newLinkedList();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    List<CreditDownUpAccountEntity> entities = this.creditDownUpAccountRepository.findByCreditCashFlowExtendDto(dto);
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newLinkedList();
    }
    return entities.stream().map(entity -> {
      CreditCashFlowExtendVo extendVo = new CreditCashFlowExtendVo();
      extendVo.setCashSerialNumber(entity.getCashSerialNumber());
      extendVo.setRemark(entity.getRemark());
      extendVo.setOrderCode(entity.getOrderCode());
      extendVo.setOrderType(entity.getOrderType());
      if (!CollectionUtils.isEmpty(entity.getFileList())) {
        List<CreditFileVo> voList = (List<CreditFileVo>) nebulaToolkitService.copyCollectionByWhiteList(entity.getFileList(), CreditFileEntity.class
            , CreditFileVo.class, HashSet.class, ArrayList.class);
        extendVo.setFileList(voList);
      }
      return extendVo;
    }).collect(Collectors.toList());
  }
}
