package com.biz.crm.dms.business.costpool.credit.local.service.register;

import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashFlowItemEntity;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditCashFlowItemRepository;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditCashEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditFileEntity;
import com.biz.crm.dms.business.costpool.credit.local.entity.CreditRecordEntity;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditCashRepository;
import com.biz.crm.dms.business.costpool.credit.local.repository.CreditRecordRepository;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditCashFlowExtendDto;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditUpdateCashDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditType;
import com.biz.crm.dms.business.costpool.credit.sdk.register.CreditCashFlowRegister;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditCashFlowExtendVo;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditCashVo;
import com.biz.crm.dms.business.costpool.credit.sdk.vo.CreditFileVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 授信变更-资金流水注册器实现
 * <p>
 * <p>
 * 适用场景描述:
 * 主要用于授信变更,更新授信资金(授信金额,可用金额)保存流水拆分明细
 *
 * @author ning.zhang
 * @date 2021/12/16
 */
@Component
public class CreditUpdateCashFlowRegisterImpl implements CreditCashFlowRegister<CreditUpdateCashDto> {

  @Autowired(required = false)
  private CreditCashFlowItemRepository creditCashFlowItemRepository;
  @Autowired(required = false)
  private CreditCashRepository creditCashRepository;
  @Autowired(required = false)
  private CreditRecordRepository creditRecordRepository;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Class<CreditUpdateCashDto> getClazz() {
    return CreditUpdateCashDto.class;
  }

  @Override
  @Transactional
  public List<CreditCashVo> onCreate(CreditUpdateCashDto dto) {
    Validate.notBlank(dto.getCreditId(), "缺失授信ID");
    CreditCashEntity nowCashEntity = creditCashRepository.findByCreditId(dto.getCreditId());
    if (dto.getOperateAmount().compareTo(BigDecimal.ZERO) < 0) {
      Validate.isTrue(Objects.nonNull(nowCashEntity) && nowCashEntity.getCanUseAmount().add(dto.getOperateAmount()).compareTo(BigDecimal.ZERO) >= 0
          , String.format("可用额度不足,无法执行[%s]操作", dto.getAdjustOperateName()));
    }
    //更新资金授信额度和可用额度
    nowCashEntity.setCanUseAmount(nowCashEntity.getCanUseAmount().add(dto.getOperateAmount()));
    nowCashEntity.setCreditAmount(nowCashEntity.getCreditAmount().add(dto.getOperateAmount()));
    this.creditCashRepository.updateById(nowCashEntity);
    //持久化资金流水拆分信息
    CreditCashFlowItemEntity cashFlowItemEntity = new CreditCashFlowItemEntity();
    cashFlowItemEntity.setTenantCode(dto.getTenantCode());
    cashFlowItemEntity.setCreditId(dto.getCreditId());
    cashFlowItemEntity.setItemAmount(dto.getOperateAmount());
    cashFlowItemEntity.setCashFlowId(dto.getId());
    this.creditCashFlowItemRepository.save(cashFlowItemEntity);
    return Lists.newArrayList(this.nebulaToolkitService.copyObjectByWhiteList(nowCashEntity, CreditCashVo.class, HashSet.class, ArrayList.class));
  }

  @Override
  public List<CreditCashFlowExtendVo> onRequestByCreditCashFlowExtendDto(CreditCashFlowExtendDto dto) {
    //如果是单据查询,直接返回空,因为无关联单据信息
    if (Objects.isNull(dto) || !CollectionUtils.isEmpty(dto.getOrderCodes())) {
      return Lists.newLinkedList();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    List<CreditRecordEntity> entities = this.creditRecordRepository.findByCreditCashFlowExtendDto(dto);
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newLinkedList();
    }
    return entities.stream().map(entity -> {
      CreditCashFlowExtendVo extendVo = new CreditCashFlowExtendVo();
      extendVo.setCashSerialNumber(entity.getCashSerialNumber());
      extendVo.setRemark(entity.getRemark());
      extendVo.setTemporaryAmount(CreditType.TEMPORARY_CREDIT.getDictCode().equals(entity.getCreditType()) ? entity.getCreditAmount() : null);
      if (!CollectionUtils.isEmpty(entity.getFileList())) {
        List<CreditFileVo> voList = (List<CreditFileVo>) nebulaToolkitService.copyCollectionByWhiteList(entity.getFileList(), CreditFileEntity.class
            , CreditFileVo.class, HashSet.class, ArrayList.class);
        extendVo.setFileList(voList);
      }
      return extendVo;
    }).collect(Collectors.toList());
  }
}
