package com.biz.crm.dms.business.costpool.discount.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.costpool.discount.local.entity.CostPoolDiscount;
import com.biz.crm.dms.business.costpool.discount.local.repository.CostPoolDiscountRepository;
import com.biz.crm.dms.business.costpool.discount.local.service.CostPoolDiscountService;
import com.biz.crm.dms.business.costpool.discount.sdk.constant.PoolDiscountConstant;
import com.biz.crm.dms.business.costpool.discount.sdk.dto.CostPoolDiscountDto;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolGroupEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolOperationTypeEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolPayTypeEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolTypeEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolUseTypeEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.strategy.OperationTypeStrategy;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.security.sdk.vo.LoginDetails;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * 折扣费用池，按照租户进行隔离(CostPoolDiscount)表服务实现类
 *
 * @author songjingen
 * @since 2021-12-20 19:45:20
 */
@Service("CostPoolDiscountService")
public class CostPoolDiscountServiceImpl implements CostPoolDiscountService {

  @Autowired(required = false)
  private CostPoolDiscountRepository costPoolDiscountRepository;
  @Autowired(required = false)
  private CustomerVoService customerVoService;
  @Autowired(required = false)
  private List<OperationTypeStrategy> operationTypeStrategies;
  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  /**
   * 分页查询数据
   *
   * @param pageable         分页对象
   * @param costPoolDiscount 实体对象
   * @return
   */
  @Override
  public Page<CostPoolDiscount> findByConditions(Pageable pageable, CostPoolDiscount costPoolDiscount) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(costPoolDiscount)) {
      costPoolDiscount = new CostPoolDiscount();
    }
    return this.costPoolDiscountRepository.findByConditions(pageable, costPoolDiscount);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public CostPoolDiscount findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.costPoolDiscountRepository.getById(id);
  }

  /**
   * 新增数据(上账)
   *
   * @param costPoolDiscount 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public void create(CostPoolDiscount costPoolDiscount) {
    this.createForm(costPoolDiscount);
    this.costPoolDiscountRepository.save(costPoolDiscount);
  }

  /**
   * 调整折扣池
   *
   * @param costPoolDiscountDto
   */
  @Override
  @Transactional
  public void handleAdjust(CostPoolDiscountDto costPoolDiscountDto) {
    this.handleAdjustValidate(costPoolDiscountDto);
    //调用策略类，根据操作类型确认执行哪个策略实现
    PoolOperationTypeEnum operationTypeEnum = PoolOperationTypeEnum.getByKey(costPoolDiscountDto.getOperationType());
    Validate.notNull(operationTypeEnum, "不支持此操作类型进行调整费用池信息");
    String operationTypeGroup = operationTypeEnum.getGroup();
    Validate.isTrue(!CollectionUtils.isEmpty(operationTypeStrategies), "未查询到操作类型对应的策略实现");
    for (OperationTypeStrategy operationTypeStrategy : operationTypeStrategies) {
      String group = operationTypeStrategy.getOperationTypeGroup();
      if (StringUtils.equals(group, operationTypeGroup)) {
        operationTypeStrategy.onSaveDiscountInfos(costPoolDiscountDto);
        break;
      }
    }
  }

  /**
   * 根据类型和客户编码查看数据信息
   *
   * @param useType
   * @param customerCode
   * @return
   */
  @Override
  public CostPoolDiscount findByTypeAndCustomerCode(String useType, String customerCode) {
    if (StringUtils.isAnyBlank(useType, customerCode)) {
      return null;
    }
    return this.costPoolDiscountRepository.findByTypeAndCustomerCode(useType, customerCode);
  }

  /**
   * 修改
   *
   * @param poolDiscount
   */
  @Override
  @Transactional
  public void update(CostPoolDiscount poolDiscount) {
    Validate.notNull(poolDiscount, "修改时，对象不能为空");
    Validate.notBlank(poolDiscount.getId(), "修改时，主键不能为空");
    LoginDetails loginUser = this.loginUserService.getLoginUser();
    poolDiscount.setModifyTime(new Date());
    poolDiscount.setModifyAccount(loginUser.getAccount());
    poolDiscount.setModifyName(loginUser.getUsername());
    this.costPoolDiscountRepository.updateById(poolDiscount);
  }

  /**
   * 根据折扣池编码查询对象
   *
   * @param poolCode
   * @return
   */
  @Override
  public CostPoolDiscount findByPoolCode(String poolCode) {
    if (StringUtils.isBlank(poolCode)) {
      return null;
    }
    return this.costPoolDiscountRepository.findByPoolCode(poolCode);
  }

  /**
   * 根据客户编码查询
   *
   * @param customerCode
   * @return
   */
  @Override
  public List<CostPoolDiscount> findByCustomerCode(String customerCode) {
    if (StringUtils.isBlank(customerCode)) {
      return null;
    }
    return this.costPoolDiscountRepository.findByCustomer(customerCode);
  }

  /**
   * 调整验证
   *
   * @param costPoolDiscount
   */
  private void handleAdjustValidate(CostPoolDiscountDto costPoolDiscount) {
    Validate.notNull(costPoolDiscount, "新增时，对象信息不能为空！");
    Validate.notBlank(costPoolDiscount.getCustomerCode(), "新增数据时，客户编码不能为空！");
    Validate.notNull(costPoolDiscount.getAmount(), "新增数据时，金额不能为空！");
    Validate.notBlank(costPoolDiscount.getOperationType(), "新增数据时，操作类型不能为空！");
    Validate.isTrue(costPoolDiscount.getAmount().compareTo(BigDecimal.ZERO) > 0, "新增数据时，金额必须大于0");
    //验证客户信息
    List<CustomerVo> byCustomerCodes = customerVoService.findByCustomerCodes(Lists.newArrayList(costPoolDiscount.getCustomerCode()));
    Validate.isTrue(!CollectionUtils.isEmpty(byCustomerCodes), "客户不存在！");
    costPoolDiscount.setCustomerName(byCustomerCodes.get(0).getCustomerName());
    Validate.notBlank(costPoolDiscount.getCustomerName(), "新增数据时，客户名称不能为空！");
  }

  /**
   * 创建表单信息
   *
   * @param costPoolDiscount
   */
  private void createForm(CostPoolDiscount costPoolDiscount) {
    LoginDetails loginUser = this.loginUserService.getLoginUser();
    Date date = new Date();
    costPoolDiscount.setPoolCode(generateCodeService.generateCode(PoolDiscountConstant.DISCOUNT_POOL_CODE, 1).get(0));
    costPoolDiscount.setCreateAccount(loginUser.getAccount());
    costPoolDiscount.setCreateName(loginUser.getUsername());
    costPoolDiscount.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    costPoolDiscount.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    costPoolDiscount.setCreateTime(date);
    costPoolDiscount.setModifyAccount(loginUser.getAccount());
    costPoolDiscount.setModifyName(loginUser.getUsername());
    costPoolDiscount.setModifyTime(date);
    costPoolDiscount.setTenantCode(TenantUtils.getTenantCode());
    if (StringUtils.isBlank(costPoolDiscount.getPoolGroup())){
      costPoolDiscount.setPoolGroup(PoolGroupEnum.DEFAULT.getDictCode());
    }
    if (StringUtils.isBlank(costPoolDiscount.getPoolType())){
      costPoolDiscount.setPoolType(PoolTypeEnum.DISCOUNT.getDictCode());
    }
    if (StringUtils.isBlank(costPoolDiscount.getPoolType())){
      costPoolDiscount.setPayType(PoolPayTypeEnum.DISCOUNT.getDictCode());
    }
    if (StringUtils.isBlank(costPoolDiscount.getPoolType())){
      costPoolDiscount.setUseType(PoolUseTypeEnum.DEFAULT.getDictCode());
    }
  }
}

