package com.biz.crm.dms.business.costpool.discount.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.costpool.discount.local.entity.CostPoolDiscount;
import com.biz.crm.dms.business.costpool.discount.local.entity.CostPoolDiscountDetail;
import com.biz.crm.dms.business.costpool.discount.local.repository.CostPoolDiscountDetailRepository;
import com.biz.crm.dms.business.costpool.discount.local.service.CostPoolDiscountDetailService;
import com.biz.crm.dms.business.costpool.discount.local.service.CostPoolDiscountService;
import com.biz.crm.dms.business.costpool.discount.sdk.constant.PoolDiscountConstant;
import com.biz.crm.dms.business.costpool.discount.sdk.dto.CostPoolDiscountDto;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolOperationTypeEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.strategy.DetailOperationTypeStrategy;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * 折扣费用池明细,按照租户进行隔离(CostPoolDiscountDetail)表服务实现类
 *
 * @author songjingen
 * @since 2021-12-21 16:46:05
 */
@Service("costPoolDiscountDetailService")
public class CostPoolDiscountDetailServiceImpl implements CostPoolDiscountDetailService {

  @Autowired(required = false)
  private CostPoolDiscountDetailRepository costPoolDiscountDetailRepository;
  @Autowired(required = false)
  private CostPoolDiscountService costPoolDiscountService;
  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired(required = false)
  private CustomerVoService customerVoService;
  @Autowired(required = false)
  private List<DetailOperationTypeStrategy> detailOperationTypeStrategies;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  /**
   * 分页查询数据
   *
   * @param pageable               分页对象
   * @param costPoolDiscountDetail 实体对象
   * @return
   */
  @Override
  public Page<CostPoolDiscountDetail> findByConditions(Pageable pageable, CostPoolDiscountDetail costPoolDiscountDetail) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(costPoolDiscountDetail)) {
      costPoolDiscountDetail = new CostPoolDiscountDetail();
    }
    costPoolDiscountDetail.setTenantCode(TenantUtils.getTenantCode());
    Page<CostPoolDiscountDetail> byConditions = this.costPoolDiscountDetailRepository.findByConditions(pageable, costPoolDiscountDetail);
    return byConditions;
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public CostPoolDiscountDetail findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return new CostPoolDiscountDetail();
    }
    //查询明细
    CostPoolDiscountDetail poolDetail = this.costPoolDiscountDetailRepository.getById(id);
    if (Objects.isNull(poolDetail)) {
      return new CostPoolDiscountDetail();
    }
    //查询主表
    CostPoolDiscount costPoolDiscount = this.costPoolDiscountService.findByPoolCode(poolDetail.getPoolCode());
    //组装
    poolDetail.setCostPoolDiscount(costPoolDiscount);
    return poolDetail;
  }

  /**
   * 新增数据
   *
   * @param costPoolDiscountDetail 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public CostPoolDiscountDetail create(CostPoolDiscountDetail costPoolDiscountDetail) {
    this.createForm(costPoolDiscountDetail);
    this.createValidate(costPoolDiscountDetail);
    this.costPoolDiscountDetailRepository.saveOrUpdate(costPoolDiscountDetail);
    return costPoolDiscountDetail;
  }

  /**
   * 根据费用池编码查询数据集合
   *
   * @param poolCode
   * @return
   */
  @Override
  public List<CostPoolDiscountDetail> findByPoolCode(String poolCode) {
    if (StringUtils.isBlank(poolCode)) {
      return null;
    }
    return this.costPoolDiscountDetailRepository.findByPoolCode(poolCode);
  }

  @Override
  public List<CostPoolDiscountDetail> findByPoolCodeAndFromCode(String poolCode, String fromCode) {
    if (StringUtils.isAnyBlank(poolCode, fromCode)) {
      return new ArrayList<>(0);
    }
    return this.costPoolDiscountDetailRepository.findByPoolCodeAndFromCode(poolCode, fromCode);
  }

  /**
   * 批量新增
   *
   * @param discountDetailList
   */
  @Override
  @Transactional
  public void createBatch(List<CostPoolDiscountDetail> discountDetailList) {
    if (CollectionUtils.isEmpty(discountDetailList)) {
      return;
    }
    for (CostPoolDiscountDetail costPoolDiscountDetail : discountDetailList) {
      this.createForm(costPoolDiscountDetail);
      this.createValidate(costPoolDiscountDetail);
    }
    this.costPoolDiscountDetailRepository.saveBatch(discountDetailList);
  }

  /**
   * 批量更新
   *
   * @param discountDetailList
   */
  @Override
  @Transactional
  public void updateBatch(List<CostPoolDiscountDetail> discountDetailList) {
    if (CollectionUtils.isEmpty(discountDetailList)) {
      return;
    }
    this.costPoolDiscountDetailRepository.updateBatchById(discountDetailList);
  }

  /**
   * 根据折扣池明细编码查询数据
   *
   * @param poolDetailCode
   * @return
   */
  @Override
  public CostPoolDiscountDetail findByPoolDetailCode(String poolDetailCode) {
    if (StringUtils.isBlank(poolDetailCode)) {
      return null;
    }
    return this.costPoolDiscountDetailRepository.findByPoolDetailCode(poolDetailCode);
  }

  @Override
  public List<CostPoolDiscountDetail> findByPoolDetailCodes(List<String> poolDetailCodes) {
    if (CollectionUtils.isEmpty(poolDetailCodes)) {
      return null;
    }
    return this.costPoolDiscountDetailRepository.findByPoolDetailCodes(poolDetailCodes);
  }

  /**
   * 修改
   *
   * @param poolDiscountDetail
   */
  @Override
  @Transactional
  public void update(CostPoolDiscountDetail poolDiscountDetail) {
    Validate.notNull(poolDiscountDetail, "修改时，对象不能为空！");
    Validate.notBlank(poolDiscountDetail.getId(), "修改时，主键不能为空！");
    AbstractCrmUserIdentity loginUser = loginUserService.getAbstractLoginUser();
    poolDiscountDetail.setModifyTime(new Date());
    poolDiscountDetail.setModifyAccount(loginUser.getAccount());
    poolDiscountDetail.setModifyName(loginUser.getUsername());
    this.costPoolDiscountDetailRepository.updateById(poolDiscountDetail);
  }

  @Override
  @Transactional
  public void handleAdjust(CostPoolDiscountDto costPoolDiscountDto) {
    this.handleAdjustValidate(costPoolDiscountDto);
    //调用策略类，根据操作类型确认执行哪个策略实现
    PoolOperationTypeEnum operationTypeEnum = PoolOperationTypeEnum.getByKey(costPoolDiscountDto.getOperationType());
    Validate.notNull(operationTypeEnum, "不支持此操作类型进行调整费用池信息");
    String operationTypeGroup = operationTypeEnum.getGroup();
    Validate.isTrue(!CollectionUtils.isEmpty(detailOperationTypeStrategies), "未查询到操作类型对应的策略实现");
    for (DetailOperationTypeStrategy operationTypeStrategy : detailOperationTypeStrategies) {
      String group = operationTypeStrategy.getOperationTypeGroup();
      if (StringUtils.equals(group, operationTypeGroup)) {
        operationTypeStrategy.onSaveDiscountInfos(costPoolDiscountDto);
        break;
      }
    }
  }

  /**
   * 创建验证
   *
   * @param costPoolDiscountDetail
   */
  private void createValidate(CostPoolDiscountDetail costPoolDiscountDetail) {
    Validate.notNull(costPoolDiscountDetail, "新增时，对象信息不能为空！");
    costPoolDiscountDetail.setId(null);
    costPoolDiscountDetail.setPoolDetailCode(generateCodeService.generateCode(PoolDiscountConstant.DISCOUNT_POOL_CODE_DETAIL, 1).get(0));
    Validate.notBlank(costPoolDiscountDetail.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notNull(costPoolDiscountDetail.getAccountDateTime(), "新增数据时，上账时间 yyyy-MM-dd HH:mm:ss不能为空！");
    Validate.notNull(costPoolDiscountDetail.getFreezeAmount(), "新增数据时，冻结金额（数量）不能为空！");
    Validate.notNull(costPoolDiscountDetail.getHasUseAmount(), "新增数据时，已使用金额（数量）不能为空！");
    Validate.notNull(costPoolDiscountDetail.getOccupyAmount(), "新增数据时，占用金额（数量）不能为空！");
    Validate.notBlank(costPoolDiscountDetail.getOperationCode(), "新增数据时，操作记录编号不能为空！");
    Validate.notBlank(costPoolDiscountDetail.getOperationType(), "新增数据时，操作类型不能为空！");
    Validate.notBlank(costPoolDiscountDetail.getPoolCode(), "新增数据时，费用池编号不能为空！");
    Validate.notBlank(costPoolDiscountDetail.getPoolDetailCode(), "新增数据时，费用池明细编号不能为空！");
    Validate.notNull(costPoolDiscountDetail.getTotalAmount(), "新增数据时，总金额（数量）不能为空！");
    Validate.notNull(costPoolDiscountDetail.getUsableAmount(), "新增数据时，剩余可使用金额（数量）不能为空！");

  }

  /**
   * 创建表单信息
   *
   * @param costPoolDiscountDetail
   */
  private void createForm(CostPoolDiscountDetail costPoolDiscountDetail) {
    AbstractCrmUserIdentity loginUser = loginUserService.getAbstractLoginUser();
    Date date = new Date();
    costPoolDiscountDetail.setCreateAccount(loginUser.getAccount());
    costPoolDiscountDetail.setCreateName(loginUser.getUsername());
    costPoolDiscountDetail.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    costPoolDiscountDetail.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    costPoolDiscountDetail.setCreateTime(date);
    costPoolDiscountDetail.setModifyAccount(loginUser.getAccount());
    costPoolDiscountDetail.setModifyName(loginUser.getUsername());
    costPoolDiscountDetail.setModifyTime(date);
    costPoolDiscountDetail.setTenantCode(TenantUtils.getTenantCode());
  }

  /**
   * 调整验证
   *
   * @param costPoolDiscount
   */
  private void handleAdjustValidate(CostPoolDiscountDto costPoolDiscount) {
    Validate.notNull(costPoolDiscount, "新增时，对象信息不能为空！");
    Validate.notBlank(costPoolDiscount.getCustomerCode(), "新增数据时，客户编码不能为空！");
    Validate.notNull(costPoolDiscount.getAmount(), "新增数据时，金额不能为空！");
    Validate.notBlank(costPoolDiscount.getOperationType(), "新增数据时，操作类型不能为空！");
    Validate.isTrue(costPoolDiscount.getAmount().compareTo(BigDecimal.ZERO) > 0, "新增数据时，金额必须大于0");
    //验证客户信息
    List<CustomerVo> byCustomerCodes = customerVoService.findByCustomerCodes(Lists.newArrayList(costPoolDiscount.getCustomerCode()));
    Validate.isTrue(!CollectionUtils.isEmpty(byCustomerCodes), "客户不存在！");
    costPoolDiscount.setCustomerName(byCustomerCodes.get(0).getCustomerName());
    Validate.notBlank(costPoolDiscount.getCustomerName(), "新增数据时，客户名称不能为空！");
  }

}

