package com.biz.crm.dms.business.costpool.local.service.internal;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.dms.business.costpool.sdk.dto.CostPoolDto;
import com.biz.crm.dms.business.costpool.sdk.model.CostPoolModel;
import com.biz.crm.dms.business.costpool.sdk.service.CostPoolVoService;
import com.biz.crm.dms.business.costpool.sdk.strategy.CostPoolRegister;
import com.biz.crm.dms.business.costpool.sdk.strategy.CostPoolStrategy;
import com.biz.crm.dms.business.costpool.sdk.vo.CostPoolVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 费用池基础service
 *
 * @author songjingen
 */
@Service
public class CostPoolVoServiceImpl implements CostPoolVoService {

  @Autowired(required = false)
  private List<CostPoolStrategy> costPoolStrategies;
  @Autowired(required = false)
  private List<CostPoolRegister> costPoolRegisters;

  @Override
  public List<CostPoolVo> handleRequestCostPoolVos(CostPoolDto costPoolDto) {
    if (Objects.isNull(costPoolDto) || CollectionUtils.isEmpty(costPoolStrategies)) {
      return null;
    }
    String poolType = costPoolDto.getPoolType();
    List<CostPoolVo> costPoolVos = new ArrayList<>();
    for (CostPoolStrategy costPoolStrategy : costPoolStrategies) {
      String poolTypeRegisterKey = costPoolStrategy.getPoolTypeRegisterKey();
      if (StringUtils.equals(poolType, poolTypeRegisterKey)) {
        costPoolVos = costPoolStrategy.onRequestCostPoolVos(costPoolDto);
        break;
      }
    }
    return costPoolVos;
  }

  @Transactional
  @Override
  public void handleAdjust(JSONArray jsonArray) {
    Validate.isTrue(jsonArray != null && jsonArray.size() > 0, "调整费用池信息时，入参json数组不能为空！");
    Validate.isTrue(!CollectionUtils.isEmpty(costPoolStrategies), "调整费用池信息时，未查询到费用池基础模块观察者实现类");
    for (int i = 0; i < jsonArray.size(); i++) {
      JSONObject jsonObject = jsonArray.getJSONObject(i);
      String poolType = jsonObject.getString("poolType");
      for (CostPoolStrategy costPoolStrategy : costPoolStrategies) {
        String poolTypeRegisterKey = costPoolStrategy.getPoolTypeRegisterKey();
        if (StringUtils.equals(poolType, poolTypeRegisterKey)) {
          costPoolStrategy.onHandleAdjust(jsonObject);
          break;
        }
      }
    }
  }


  @Override
  public List<CostPoolModel> findCostPoolModel() {
    if (CollectionUtils.isEmpty(costPoolRegisters)) {
      return new ArrayList<>();
    }
    List<CostPoolModel> costPoolModels = Lists.newArrayList();
    for (CostPoolRegister costPoolRegister : costPoolRegisters) {
      CostPoolModel costPoolModel = new CostPoolModel();
      costPoolModel.setKey(costPoolRegister.getKey());
      costPoolModel.setName(costPoolRegister.getName());
      costPoolModels.add(costPoolModel);
    }
    return costPoolModels;
  }

  @Override
  public Map<String, BigDecimal> findUsableAmountByCustomerCodeStuMap(String customerCode) {
    if (StringUtils.isBlank(customerCode) || CollectionUtils.isEmpty(costPoolStrategies)) {
      return null;
    }
    List<CostPoolModel> costPoolModel = this.findCostPoolModel();
    HashMap<String, BigDecimal> map = new HashMap<>();
    for (CostPoolModel poolModel : costPoolModel) {
      CostPoolDto costPoolDto = new CostPoolDto();
      costPoolDto.setPoolType(poolModel.getKey());
      costPoolDto.setCustomerCode(customerCode);
      BigDecimal amount = BigDecimal.ZERO;
      List<CostPoolVo> costPoolVos = this.handleRequestCostPoolVos(costPoolDto);
      if (!CollectionUtils.isEmpty(costPoolVos)) {
        amount = costPoolVos.stream().map(CostPoolVo::getUsableAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
      }
      map.put(poolModel.getKey(), amount);
    }
    return map;
  }
}
