package com.biz.crm.dms.business.costpool.replenishment.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.costpool.replenishment.local.entity.CostPoolReplenishment;
import com.biz.crm.dms.business.costpool.replenishment.local.entity.CostPoolReplenishmentDetail;
import com.biz.crm.dms.business.costpool.replenishment.local.entity.CostPoolReplenishmentProduct;
import com.biz.crm.dms.business.costpool.replenishment.local.repository.CostPoolReplenishmentDetailRepository;
import com.biz.crm.dms.business.costpool.replenishment.local.service.CostPoolReplenishmentDetailService;
import com.biz.crm.dms.business.costpool.replenishment.local.service.CostPoolReplenishmentProductService;
import com.biz.crm.dms.business.costpool.replenishment.local.service.CostPoolReplenishmentService;
import com.biz.crm.dms.business.costpool.replenishment.sdk.constant.PoolReplenishmentConstant;
import com.biz.crm.dms.business.costpool.replenishment.sdk.dto.CostPoolReplenishmentDto;
import com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolOperationTypeEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.strategy.DetailOperationTypeStrategy;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.security.sdk.vo.LoginDetails;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 货补费用池明细(CostPoolReplenishmentDetail)表服务实现类
 *
 * @author HanJiaJun
 * @since 2021-12-28 15:17:36
 */
@Service("costPoolReplenishmentDetailService")
public class CostPoolReplenishmentDetailServiceImpl implements CostPoolReplenishmentDetailService {

  @Autowired(required = false)
  private CostPoolReplenishmentDetailRepository costPoolReplenishmentDetailRepository;
  @Autowired(required = false)
  private List<DetailOperationTypeStrategy> detailOperationTypeStrategies;
  @Autowired(required = false)
  private CustomerVoService customerVoService;
  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired(required = false)
  private CostPoolReplenishmentService costPoolReplenishmentService;
  @Autowired(required = false)
  private CostPoolReplenishmentProductService costPoolReplenishmentProductService;

  /**
   * 分页查询数据
   *
   * @param pageable                    分页对象
   * @param costPoolReplenishmentDetail 实体对象
   * @return
   */
  @Override
  public Page<CostPoolReplenishmentDetail> findByConditions(Pageable pageable, CostPoolReplenishmentDetail costPoolReplenishmentDetail) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(costPoolReplenishmentDetail)) {
      costPoolReplenishmentDetail = new CostPoolReplenishmentDetail();
    }
    costPoolReplenishmentDetail.setTenantCode(TenantUtils.getTenantCode());
    return this.costPoolReplenishmentDetailRepository.findByConditions(pageable, costPoolReplenishmentDetail);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public CostPoolReplenishmentDetail findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return new CostPoolReplenishmentDetail();
    }
    CostPoolReplenishmentDetail costPoolReplenishmentDetail = this.costPoolReplenishmentDetailRepository.getById(id);
    if (Objects.isNull(costPoolReplenishmentDetail)) {
      return new CostPoolReplenishmentDetail();
    }
    CostPoolReplenishment costPoolReplenishment = this.costPoolReplenishmentService.findByPoolCode(costPoolReplenishmentDetail.getPoolCode());
    List<CostPoolReplenishment> list = new ArrayList<>();
    list.add(costPoolReplenishment);
    List<CostPoolReplenishment> costPoolReplenishments = this.splicingProduct(list);
    if (CollectionUtils.isEmpty(costPoolReplenishments)) {
      return costPoolReplenishmentDetail;
    }
    costPoolReplenishmentDetail.setCostPoolReplenishment(costPoolReplenishments.get(0));
    return costPoolReplenishmentDetail;
  }

  /**
   * 拼接货补产品表信息
   *
   * @param records
   * @return
   */
  private List<CostPoolReplenishment> splicingProduct(List<CostPoolReplenishment> records) {
    //费用池编码
    List<String> codes = records.stream().map(CostPoolReplenishment::getPoolCode).collect(Collectors.toList());
    //产品表信息
    List<CostPoolReplenishmentProduct> products = this.costPoolReplenishmentProductService.findByPoolCodes(codes);
    if (CollectionUtils.isEmpty(products)) {
      return records;
    }
    Map<String, List<CostPoolReplenishmentProduct>> productMap = products.stream().collect(Collectors.groupingBy(CostPoolReplenishmentProduct::getPoolCode));
    if (CollectionUtils.isEmpty(productMap)) {
      return records;
    }
    //主表和子表数据关联
    for (CostPoolReplenishment record : records) {
      List<CostPoolReplenishmentProduct> product = productMap.get(record.getPoolCode());
      record.setCostPoolReplenishmentProduct(product);
    }
    return records;
  }

  /**
   * 新增数据
   *
   * @param costPoolReplenishmentDetail 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public CostPoolReplenishmentDetail create(CostPoolReplenishmentDetail costPoolReplenishmentDetail) {
    this.createForm(costPoolReplenishmentDetail);
    this.createValidate(costPoolReplenishmentDetail);
    this.costPoolReplenishmentDetailRepository.saveOrUpdate(costPoolReplenishmentDetail);
    return costPoolReplenishmentDetail;
  }

  /**
   * 修改新据
   *
   * @param costPoolReplenishmentDetail 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public CostPoolReplenishmentDetail update(CostPoolReplenishmentDetail costPoolReplenishmentDetail) {
    this.updateValidate(costPoolReplenishmentDetail);
    this.updateForm(costPoolReplenishmentDetail);
    this.costPoolReplenishmentDetailRepository.saveOrUpdate(costPoolReplenishmentDetail);
    return costPoolReplenishmentDetail;
  }

  /**
   * 批量新增
   *
   * @param replenishmentDetailList
   */
  @Override
  public void createBatch(List<CostPoolReplenishmentDetail> replenishmentDetailList) {
    if (CollectionUtils.isEmpty(replenishmentDetailList)) {
      return;
    }
    replenishmentDetailList.forEach(e -> {
      this.createForm(e);
      this.createValidate(e);
    });
    this.costPoolReplenishmentDetailRepository.saveBatch(replenishmentDetailList);
  }

  /**
   * 批量更新
   *
   * @param replenishmentDetailList
   */
  @Override
  @Transactional
  public void updateBatch(List<CostPoolReplenishmentDetail> replenishmentDetailList) {
    if (CollectionUtils.isEmpty(replenishmentDetailList)) {
      return;
    }
    this.costPoolReplenishmentDetailRepository.updateBatchById(replenishmentDetailList);
  }

  /**
   * 通过费用池编码查询集合
   *
   * @param poolCode
   * @return
   */
  @Override
  public List<CostPoolReplenishmentDetail> findByPoolCode(String poolCode) {
    if (StringUtils.isBlank(poolCode)) {
      return new ArrayList<>(0);
    }
    return this.costPoolReplenishmentDetailRepository.findByPoolCode(poolCode);
  }

  @Override
  public List<CostPoolReplenishmentDetail> findByPoolCodeAndFromCode(String poolCode, String fromCode) {
    if (StringUtils.isAnyBlank(poolCode, fromCode)) {
      return new ArrayList<>(0);
    }
    return this.costPoolReplenishmentDetailRepository.findByPoolCodeAndFromCode(poolCode, fromCode);
  }


  /**
   * 调整货补池
   *
   * @param costPoolReplenishmentDto
   */
  @Override
  @Transactional
  public void handleAdjust(CostPoolReplenishmentDto costPoolReplenishmentDto) {
    this.handleAdjustValidate(costPoolReplenishmentDto);
    //调用策略类，根据操作类型确认执行哪个策略实现
    PoolOperationTypeEnum operationTypeEnum = PoolOperationTypeEnum.getByKey(costPoolReplenishmentDto.getOperationType());
    Validate.notNull(operationTypeEnum, "不支持此操作类型进行调整费用池信息");
    String operationTypeGroup = operationTypeEnum.getGroup();
    Validate.isTrue(!CollectionUtils.isEmpty(detailOperationTypeStrategies), "未查询到操作类型对应的策略实现");
    for (DetailOperationTypeStrategy operationTypeStrategy : detailOperationTypeStrategies) {
      String group = operationTypeStrategy.getOperationTypeGroup();
      if (StringUtils.equals(group, operationTypeGroup)) {
        operationTypeStrategy.onSaveDiscountInfos(costPoolReplenishmentDto);
        break;
      }
    }
  }

  /**
   * 根据编码查询对象
   *
   * @param poolDetailCode
   * @return
   */
  @Override
  public CostPoolReplenishmentDetail findByPoolDetailCode(String poolDetailCode) {
    if (StringUtils.isBlank(poolDetailCode)) {
      return null;
    }
    return this.costPoolReplenishmentDetailRepository.findByPoolDetailCode(poolDetailCode);
  }

  /**
   * 创建验证
   *
   * @param costPoolReplenishmentDetail
   */
  private void createValidate(CostPoolReplenishmentDetail costPoolReplenishmentDetail) {
    Validate.notNull(costPoolReplenishmentDetail, "新增时，对象信息不能为空！");
    costPoolReplenishmentDetail.setId(null);
    Validate.notBlank(costPoolReplenishmentDetail.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notNull(costPoolReplenishmentDetail.getAccountDateTime(), "新增数据时，上账时间 yyyy-MM-dd HH:mm:ss不能为空！");
    Validate.notNull(costPoolReplenishmentDetail.getFreezeAmount(), "新增数据时，冻结金额（数量）不能为空！");
    Validate.notNull(costPoolReplenishmentDetail.getHasUseAmount(), "新增数据时，已使用金额（数量）不能为空！");
    Validate.notNull(costPoolReplenishmentDetail.getOccupyAmount(), "新增数据时，占用金额（数量）不能为空！");
    Validate.notBlank(costPoolReplenishmentDetail.getOperationCode(), "新增数据时，操作记录编号不能为空！");
    Validate.notBlank(costPoolReplenishmentDetail.getOperationType(), "新增数据时，操作类型不能为空！");
    Validate.notBlank(costPoolReplenishmentDetail.getPoolCode(), "新增数据时，费用池编号不能为空！");
    Validate.notBlank(costPoolReplenishmentDetail.getPoolDetailCode(), "新增数据时，费用池明细编号不能为空！");
    Validate.notNull(costPoolReplenishmentDetail.getTotalAmount(), "新增数据时，总金额（数量）不能为空！");
    Validate.notNull(costPoolReplenishmentDetail.getUsableAmount(), "新增数据时，剩余可使用金额（数量）不能为空！");

  }

  /**
   * 修改验证
   *
   * @param costPoolReplenishmentDetail
   */
  private void updateValidate(CostPoolReplenishmentDetail costPoolReplenishmentDetail) {
    Validate.notNull(costPoolReplenishmentDetail, "修改时，对象信息不能为空！");
    Validate.notBlank(costPoolReplenishmentDetail.getId(), "修改数据时，不能为空！");
    Validate.notBlank(costPoolReplenishmentDetail.getTenantCode(), "修改数据时，租户编号不能为空！");
    Validate.notNull(costPoolReplenishmentDetail.getAccountDateTime(), "修改数据时，上账时间 yyyy-MM-dd HH:mm:ss不能为空！");
    Validate.notNull(costPoolReplenishmentDetail.getFreezeAmount(), "修改数据时，冻结金额（数量）不能为空！");
    Validate.notNull(costPoolReplenishmentDetail.getHasUseAmount(), "修改数据时，已使用金额（数量）不能为空！");
    Validate.notNull(costPoolReplenishmentDetail.getOccupyAmount(), "修改数据时，占用金额（数量）不能为空！");
    Validate.notBlank(costPoolReplenishmentDetail.getOperationCode(), "修改数据时，操作记录编号不能为空！");
    Validate.notBlank(costPoolReplenishmentDetail.getOperationType(), "修改数据时，操作类型不能为空！");
    Validate.notBlank(costPoolReplenishmentDetail.getPoolCode(), "修改数据时，费用池编号不能为空！");
    Validate.notBlank(costPoolReplenishmentDetail.getPoolDetailCode(), "修改数据时，费用池明细编号不能为空！");
    Validate.notNull(costPoolReplenishmentDetail.getTotalAmount(), "修改数据时，总金额（数量）不能为空！");
    Validate.notNull(costPoolReplenishmentDetail.getUsableAmount(), "修改数据时，剩余可使用金额（数量）不能为空！");

  }

  /**
   * 调整验证
   *
   * @param costPoolReplenishmentDto
   */
  private void handleAdjustValidate(CostPoolReplenishmentDto costPoolReplenishmentDto) {
    Validate.notNull(costPoolReplenishmentDto, "新增时，对象信息不能为空！");
    Validate.notBlank(costPoolReplenishmentDto.getCustomerCode(), "新增数据时，客户编码不能为空！");
    Validate.notNull(costPoolReplenishmentDto.getAmount(), "新增数据时，金额不能为空！");
    Validate.isTrue(costPoolReplenishmentDto.getAmount().compareTo(BigDecimal.ZERO) > 0, "新增数据时，金额必须大于0");
    //验证客户信息
    List<CustomerVo> byCustomerCodes = customerVoService.findByCustomerCodes(Lists.newArrayList(costPoolReplenishmentDto.getCustomerCode()));
    Validate.isTrue(!CollectionUtils.isEmpty(byCustomerCodes), "客户不存在！");
    costPoolReplenishmentDto.setCustomerName(byCustomerCodes.get(0).getCustomerName());
    Validate.notBlank(costPoolReplenishmentDto.getCustomerName(), "新增数据时，客户名称不能为空！");
  }

  /**
   * 组装修改公共参数
   *
   * @param costPoolReplenishmentDetail
   */
  private void updateForm(CostPoolReplenishmentDetail costPoolReplenishmentDetail) {
    LoginDetails loginUser = this.loginUserService.getLoginUser();
    costPoolReplenishmentDetail.setModifyAccount(loginUser.getAccount());
    costPoolReplenishmentDetail.setModifyName(loginUser.getUsername());
    costPoolReplenishmentDetail.setModifyTime(new Date());
  }

  /**
   * 创建表单信息
   *
   * @param costPoolReplenishmentDetail
   */
  private void createForm(CostPoolReplenishmentDetail costPoolReplenishmentDetail) {
    LoginDetails loginUser = this.loginUserService.getLoginUser();
    Date date = new Date();
    costPoolReplenishmentDetail.setId(null);
    costPoolReplenishmentDetail.setPoolDetailCode(generateCodeService.generateCode(PoolReplenishmentConstant.DISCOUNT_POOL_CODE_DETAIL, 1).get(0));
    costPoolReplenishmentDetail.setCreateAccount(loginUser.getAccount());
    costPoolReplenishmentDetail.setCreateName(loginUser.getUsername());
    costPoolReplenishmentDetail.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    costPoolReplenishmentDetail.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    costPoolReplenishmentDetail.setCreateTime(date);
    costPoolReplenishmentDetail.setModifyAccount(loginUser.getAccount());
    costPoolReplenishmentDetail.setModifyName(loginUser.getUsername());
    costPoolReplenishmentDetail.setModifyTime(date);
    costPoolReplenishmentDetail.setTenantCode(TenantUtils.getTenantCode());
  }
}

