package com.biz.crm.dms.business.costpool.replenishment.local.service.internal;

import com.biz.crm.dms.business.costpool.replenishment.local.entity.CostPoolReplenishment;
import com.biz.crm.dms.business.costpool.replenishment.local.entity.CostPoolReplenishmentOperation;
import com.biz.crm.dms.business.costpool.replenishment.local.entity.CostPoolReplenishmentProduct;
import com.biz.crm.dms.business.costpool.replenishment.local.service.CostPoolReplenishmentOperationService;
import com.biz.crm.dms.business.costpool.replenishment.local.service.CostPoolReplenishmentProductService;
import com.biz.crm.dms.business.costpool.replenishment.local.service.CostPoolReplenishmentService;
import com.biz.crm.dms.business.costpool.replenishment.sdk.service.CostPoolReplenishmentStatisticalVoService;
import com.biz.crm.dms.business.costpool.replenishment.sdk.dto.CostPoolReplenishmentOperationDto;
import com.biz.crm.dms.business.costpool.replenishment.sdk.enums.CostPoolReplenishmentOperationGroupEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolOperationTypeEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.vo.CostPoolReplenishmentStatisticalVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author HanJiaJun
 * @describe:
 * @createTime 2022年06月01日 10:50:00
 */
@Service
public class CostPoolReplenishmentStatisticalVoServiceImpl implements CostPoolReplenishmentStatisticalVoService {

  @Autowired(required = false)
  private CostPoolReplenishmentService costPoolReplenishmentService;
  @Autowired(required = false)
  private CostPoolReplenishmentOperationService costPoolReplenishmentOperationService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private CostPoolReplenishmentProductService costPoolReplenishmentProductService;


  @Override
  public List<CostPoolReplenishmentStatisticalVo> statisticsByCustomerCodeAndDate(String customerCode, Date beginDate, Date endDate) {
    Validate.notBlank(customerCode, "客户编码不能为空！");
    Validate.notNull(beginDate, "开始时间不能为空！");
    Validate.notNull(endDate, "结束时间不能为空！");
    Validate.isTrue(endDate.compareTo(beginDate) > -1, "开始时间应小于等于结束时间");
    //通过客户编码查询货补池
    List<CostPoolReplenishment> byCustomerCode = costPoolReplenishmentService
        .findByCustomerCode(customerCode);
    if (CollectionUtils.isEmpty(byCustomerCode)) {
      return new ArrayList<>();
    }
    //货补池编码
    List<String> poolCodes = byCustomerCode.stream()
        .filter(e -> StringUtils.isNotBlank(e.getPoolCode()))
        .map(CostPoolReplenishment::getPoolCode)
        .collect(Collectors.toList());
    //查询当前时间段记录
    CostPoolReplenishmentOperationDto nowDto = new CostPoolReplenishmentOperationDto();
    nowDto.setPoolCodeList(poolCodes);
    nowDto.setStartTime(beginDate);
    nowDto.setEndTime(endDate);
    List<CostPoolReplenishmentOperation> now = costPoolReplenishmentOperationService.findByPoolCodesAndDate(nowDto);
    //查询期初
    CostPoolReplenishmentOperationDto beforeDto = new CostPoolReplenishmentOperationDto();
    beforeDto.setPoolCodeList(poolCodes);
    beforeDto.setEndTime(beginDate);
    List<CostPoolReplenishmentOperation> beginning = costPoolReplenishmentOperationService.findByPoolCodesAndDate(beforeDto);
    Map<String, List<CostPoolReplenishmentOperation>> nowMap = now.stream().collect(Collectors.groupingBy(CostPoolReplenishmentOperation::getPoolCode));
    Map<String, List<CostPoolReplenishmentOperation>> beginningMap = beginning.stream().collect(Collectors.groupingBy(CostPoolReplenishmentOperation::getPoolCode));
    List<CostPoolReplenishmentStatisticalVo> costPoolReplenishmentStatisticalVos = (List<CostPoolReplenishmentStatisticalVo>) nebulaToolkitService.copyCollectionByWhiteList(byCustomerCode, CostPoolReplenishment.class, CostPoolReplenishmentStatisticalVo.class, HashSet.class, ArrayList.class);
    //统计
    for (CostPoolReplenishmentStatisticalVo vo : costPoolReplenishmentStatisticalVos) {
      //初始化金额
      initializeAmount(vo);
      //统计期初
      Map<String, BigDecimal> beginningStatisticalAmount = getStatisticalAmount(vo, beginningMap.get(vo.getPoolCode()));
      //上月期末余额
      BigDecimal beginningOnAmount = beginningStatisticalAmount.get(CostPoolReplenishmentOperationGroupEnum.ON.getDictCode());
      BigDecimal beginningUseAmount = beginningStatisticalAmount.get(CostPoolReplenishmentOperationGroupEnum.USE.getDictCode());
      vo.setLastEndAmount(beginningOnAmount.subtract(beginningUseAmount.abs()));
      //统计当前
      Map<String, BigDecimal> nowStatisticalAmount = getStatisticalAmount(vo, nowMap.get(vo.getPoolCode()));
      BigDecimal nowOnAmount = nowStatisticalAmount.get(CostPoolReplenishmentOperationGroupEnum.ON.getDictCode());
      BigDecimal nowUseAmount = nowStatisticalAmount.get(CostPoolReplenishmentOperationGroupEnum.USE.getDictCode());
      vo.setOnAmount(nowOnAmount);
      vo.setUsedAmount(nowUseAmount.abs());
      //当前可用 上月期末余额+（本月入账-本月使用-本月占用）
      BigDecimal subtract = nowOnAmount.subtract(nowUseAmount.abs());
      BigDecimal nowUsableAmount = subtract.subtract(vo.getOccupyAmount().abs());
      vo.setUsableAmount(vo.getLastEndAmount().add(nowUsableAmount));
    }
    //拼接产品信息
    joiningProductMsg(costPoolReplenishmentStatisticalVos,poolCodes);
    return costPoolReplenishmentStatisticalVos;
  }

  /**
   * 拼接产品信息
   * @param costPoolReplenishmentStatisticalVos
   * @param poolCodes
   */
  private void joiningProductMsg(List<CostPoolReplenishmentStatisticalVo> costPoolReplenishmentStatisticalVos, List<String> poolCodes) {
    if (CollectionUtils.isEmpty(poolCodes)){
      return;
    }
    List<CostPoolReplenishmentProduct> byPoolCodes = costPoolReplenishmentProductService.findByPoolCodes(poolCodes);
    if (CollectionUtils.isEmpty(byPoolCodes)){
      return;
    }
    Map<String, List<CostPoolReplenishmentProduct>> productMap = byPoolCodes.stream().collect(Collectors.groupingBy(CostPoolReplenishmentProduct::getPoolCode));
    for (CostPoolReplenishmentStatisticalVo vo : costPoolReplenishmentStatisticalVos) {
      String poolCode = vo.getPoolCode();
      List<CostPoolReplenishmentProduct> products = productMap.get(poolCode);
      if (!CollectionUtils.isEmpty(products)){
        for (CostPoolReplenishmentProduct product : products) {
          //拼接产品信息，用逗号隔开
          if (StringUtils.isBlank(vo.getGoodsProductCode())){
            vo.setGoodsProductCode(product.getGoodsProductCode());
          }else{
            String goodsProductCode = vo.getGoodsProductCode();
            String code = goodsProductCode.concat(",").concat(product.getGoodsProductCode());
            vo.setGoodsProductCode(code);
          }
          if (StringUtils.isBlank(vo.getGoodsProductName())){
            vo.setGoodsProductName(product.getGoodsProductName());
          }else{
            String goodsProductName = vo.getGoodsProductName();
            String name = goodsProductName.concat(",").concat(product.getGoodsProductName());
            vo.setGoodsProductName(name);
          }
        }
      }
    }
  }

  /**
   * 统计 上账以及可使用
   * 目前货补池没有占有 故不统计
   *
   * @param vo
   * @param costPoolReplenishmentOperations
   */
  private Map<String, BigDecimal> getStatisticalAmount(CostPoolReplenishmentStatisticalVo vo, List<CostPoolReplenishmentOperation> costPoolReplenishmentOperations) {
    Map<String, BigDecimal> res = new HashMap<>();
    res.put(CostPoolReplenishmentOperationGroupEnum.ON.getDictCode(), BigDecimal.ZERO);
    res.put(CostPoolReplenishmentOperationGroupEnum.USE.getDictCode(), BigDecimal.ZERO);
    if (CollectionUtils.isEmpty(costPoolReplenishmentOperations)) {
      return res;
    }
    //上账
    BigDecimal onAmount = BigDecimal.ZERO;
    //已使用
    BigDecimal usedAmount = BigDecimal.ZERO;
    for (CostPoolReplenishmentOperation operation : costPoolReplenishmentOperations) {
      String operationType = operation.getOperationType();
      PoolOperationTypeEnum typeEnum = PoolOperationTypeEnum.getByKey(operationType);
      //统计上账类型
      if (typeEnum != null && typeEnum.getGroupEnum() == CostPoolReplenishmentOperationGroupEnum.ON) {
        onAmount = onAmount.add(operation.getOperationAmount());
      }
      //统计使用类型
      if (typeEnum != null && typeEnum.getGroupEnum() == CostPoolReplenishmentOperationGroupEnum.USE) {
        usedAmount = usedAmount.add(operation.getOperationAmount());
      }
    }
    //组装
    if (Objects.nonNull(onAmount)) {
      res.put(CostPoolReplenishmentOperationGroupEnum.ON.getDictCode(), onAmount);
    }
    if (Objects.nonNull(usedAmount)) {
      res.put(CostPoolReplenishmentOperationGroupEnum.USE.getDictCode(), usedAmount);
    }
    return res;
  }

  /**
   * 初始化金额成0
   *
   * @param vo
   */
  private void initializeAmount(CostPoolReplenishmentStatisticalVo vo) {
    vo.setOnAmount(BigDecimal.ZERO);
    vo.setOccupyAmount(BigDecimal.ZERO);
    vo.setUsedAmount(BigDecimal.ZERO);
    vo.setLastEndAmount(BigDecimal.ZERO);
    vo.setUsableAmount(BigDecimal.ZERO);
  }
}
