package com.biz.crm.dms.business.costpool.replenishment.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.costpool.replenishment.local.entity.CostPoolReplenishment;
import com.biz.crm.dms.business.costpool.replenishment.local.entity.CostPoolReplenishmentProduct;
import com.biz.crm.dms.business.costpool.replenishment.local.repository.CostPoolReplenishmentRepository;
import com.biz.crm.dms.business.costpool.replenishment.local.service.CostPoolReplenishmentProductService;
import com.biz.crm.dms.business.costpool.replenishment.local.service.CostPoolReplenishmentService;
import com.biz.crm.dms.business.costpool.replenishment.sdk.constant.PoolReplenishmentConstant;
import com.biz.crm.dms.business.costpool.replenishment.sdk.dto.CostPoolReplenishmentDto;
import com.biz.crm.dms.business.costpool.replenishment.sdk.dto.CostPoolReplenishmentProductDto;
import com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolGroupEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolOperationTypeEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolPayTypeEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolTypeEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolUseTypeEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.strategy.OperationTypeStrategy;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 货补费用池主表(CostPoolReplenishment)表服务实现类
 *
 * @author HanJiaJun
 * @since 2021-12-28 15:17:36
 */
@Service("costPoolReplenishmentService")
public class CostPoolReplenishmentServiceImpl implements CostPoolReplenishmentService {

  @Autowired(required = false)
  private CostPoolReplenishmentRepository costPoolReplenishmentRepository;
  @Autowired(required = false)
  private List<OperationTypeStrategy> operationTypeStrategies;
  @Autowired(required = false)
  private CustomerVoService customerVoService;
  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired(required = false)
  private CostPoolReplenishmentProductService costPoolReplenishmentProductService;

  /**
   * 分页查询数据
   *
   * @param pageable                 分页对象
   * @param costPoolReplenishmentDto 实体对象
   * @return
   */
  @Override
  public Page<CostPoolReplenishment> findByConditions(Pageable pageable, CostPoolReplenishmentDto costPoolReplenishmentDto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(costPoolReplenishmentDto)) {
      costPoolReplenishmentDto = new CostPoolReplenishmentDto();
    }
    Page<CostPoolReplenishment> byConditions = this.costPoolReplenishmentRepository.findByConditions(pageable, costPoolReplenishmentDto);
    return byConditions;
  }


  /**
   * 拼接货补产品表信息
   *
   * @param records
   * @return
   */
  private List<CostPoolReplenishment> splicingProduct(List<CostPoolReplenishment> records) {
    //费用池编码
    List<String> codes = records.stream().map(CostPoolReplenishment::getPoolCode).collect(Collectors.toList());
    //产品表信息
    List<CostPoolReplenishmentProduct> products = this.costPoolReplenishmentProductService.findByPoolCodes(codes);
    if (CollectionUtils.isEmpty(products)) {
      return records;
    }
    Map<String, List<CostPoolReplenishmentProduct>> productMap = products.stream().collect(Collectors.groupingBy(CostPoolReplenishmentProduct::getPoolCode));
    if (CollectionUtils.isEmpty(productMap)) {
      return records;
    }
    //主表和子表数据关联
    for (CostPoolReplenishment record : records) {
      List<CostPoolReplenishmentProduct> product = productMap.get(record.getPoolCode());
      record.setCostPoolReplenishmentProduct(product);
    }
    return records;
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public CostPoolReplenishment findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    CostPoolReplenishment costPoolReplenishment = this.costPoolReplenishmentRepository.getById(id);
    if (Objects.isNull(costPoolReplenishment)) {
      return null;
    }
    List<CostPoolReplenishment> list = new ArrayList();
    list.add(costPoolReplenishment);
    List<CostPoolReplenishment> costPoolReplenishments = this.splicingProduct(list);
    if (CollectionUtils.isEmpty(costPoolReplenishments)) {
      return null;
    }
    return costPoolReplenishments.get(0);
  }

  /**
   * 新增数据
   *
   * @param costPoolReplenishment 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public CostPoolReplenishment create(CostPoolReplenishment costPoolReplenishment) {
    this.createForm(costPoolReplenishment);
    this.createValidate(costPoolReplenishment);
    this.costPoolReplenishmentRepository.saveOrUpdate(costPoolReplenishment);
    //创建货补产品表信息
    List<CostPoolReplenishmentProduct> costPoolReplenishmentProduct = costPoolReplenishment.getCostPoolReplenishmentProduct();
    //无产品信息
    if (CollectionUtils.isEmpty(costPoolReplenishmentProduct)) {
      return costPoolReplenishment;
    }
    for (CostPoolReplenishmentProduct poolReplenishmentProduct : costPoolReplenishmentProduct) {
      poolReplenishmentProduct.setPoolCode(costPoolReplenishment.getPoolCode());
      poolReplenishmentProduct.setProductLevelCode(costPoolReplenishment.getGoodsProductLevelCode());
    }
    this.costPoolReplenishmentProductService.create(costPoolReplenishmentProduct);
    return costPoolReplenishment;
  }

  /**
   * 修改新据
   *
   * @param costPoolReplenishment 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public CostPoolReplenishment update(CostPoolReplenishment costPoolReplenishment) {
    this.updateValidate(costPoolReplenishment);
    this.updateForm(costPoolReplenishment);
    this.costPoolReplenishmentRepository.saveOrUpdate(costPoolReplenishment);
    return costPoolReplenishment;
  }

  /**
   * 根据条件查询数据对象
   *
   * @param costPoolReplenishmentDto
   * @return
   */
  @Override
  public CostPoolReplenishment findByCostPoolReplenishmentDto(CostPoolReplenishmentDto costPoolReplenishmentDto) {
    if (costPoolReplenishmentDto == null) {
      return null;
    }
    if (StringUtils.isAnyBlank(costPoolReplenishmentDto.getCustomerCode(),
        costPoolReplenishmentDto.getGoodsProductLevelCode(),
        costPoolReplenishmentDto.getUseType())) {
      return null;
    }
    //查询主表关联子表信息
    List<CostPoolReplenishment> costPoolReplenishmentList = this.costPoolReplenishmentRepository.findByCostPoolReplenishmentDto(costPoolReplenishmentDto);
    if (CollectionUtils.isEmpty(costPoolReplenishmentList)) {
      return null;
    }
    List<CostPoolReplenishmentProductDto> poolReplenishmentProduct = costPoolReplenishmentDto.getCostPoolReplenishmentProduct();
    //获取前端传的商品集合编码
    List<String> dtoCodes = new ArrayList<>();
    if (!CollectionUtils.isEmpty(poolReplenishmentProduct)) {
      dtoCodes = poolReplenishmentProduct.stream()
          .map(CostPoolReplenishmentProductDto::getGoodsProductCode)
          .collect(Collectors.toList());
    }
    //校验
    CostPoolReplenishment poolReplenishment = this.validateProductList(dtoCodes, costPoolReplenishmentList);
    if (poolReplenishment == null) {
      return null;
    } else {
      return poolReplenishment;
    }
  }

  /**
   * 校验两个集合中产品是否重复
   *
   * @param dtoCodes
   * @param costPoolReplenishmentList
   */
  private CostPoolReplenishment validateProductList(List<String> dtoCodes, List<CostPoolReplenishment> costPoolReplenishmentList) {
    for (CostPoolReplenishment poolReplenishment : costPoolReplenishmentList) {
      List<CostPoolReplenishmentProduct> costPoolReplenishmentProduct = poolReplenishment.getCostPoolReplenishmentProduct();
      //获取数据库中的商品集合编码
      List<String> codes = new ArrayList<>();
      if (!CollectionUtils.isEmpty(costPoolReplenishmentProduct)) {
        codes = costPoolReplenishmentProduct.stream()
            .map(CostPoolReplenishmentProduct::getGoodsProductCode)
            .collect(Collectors.toList());
      }
      //校验
      //两个集合都是0，则重复
      if (codes.size() == 0 && dtoCodes.size() == 0) {
        return poolReplenishment;
      }
      //两个集合长度相等并且完全包含，则重复
      if (codes.size() == dtoCodes.size() && codes.containsAll(dtoCodes)) {
        return poolReplenishment;
      }
    }
    return null;
  }

  /**
   * 调整货补池
   *
   * @param costPoolReplenishmentDto
   */
  @Override
  @Transactional
  public void handleAdjust(CostPoolReplenishmentDto costPoolReplenishmentDto) {
    this.handleAdjustValidate(costPoolReplenishmentDto);
    //调用策略类，根据操作类型确认执行哪个策略实现
    PoolOperationTypeEnum operationTypeEnum = PoolOperationTypeEnum.getByKey(costPoolReplenishmentDto.getOperationType());
    Validate.notNull(operationTypeEnum, "不支持此操作类型进行调整费用池信息");
    String operationTypeGroup = operationTypeEnum.getGroup();
    Validate.isTrue(!CollectionUtils.isEmpty(operationTypeStrategies), "未查询到操作类型对应的策略实现");
    for (OperationTypeStrategy operationTypeStrategy : operationTypeStrategies) {
      String group = operationTypeStrategy.getOperationTypeGroup();
      if (StringUtils.equals(group, operationTypeGroup)) {
        operationTypeStrategy.onSaveDiscountInfos(costPoolReplenishmentDto);
        break;
      }
    }
  }

  /**
   * 根据编码查询对象
   *
   * @param poolCode
   * @return
   */
  @Override
  public CostPoolReplenishment findByPoolCode(String poolCode) {
    if (StringUtils.isBlank(poolCode)) {
      return null;
    }
    return this.costPoolReplenishmentRepository.findByPoolCode(poolCode);
  }

  /**
   * 根据费用池编码集合和客户编码查询
   *
   * @param poolCodes
   * @param customerCode
   * @return
   */
  @Override
  public List<CostPoolReplenishment> findByPoolCodesAndCustomerCode(List<String> poolCodes, String customerCode) {
    if (CollectionUtils.isEmpty(poolCodes)) {
      return new ArrayList<>(0);
    }
    List<CostPoolReplenishment> byPoolCodesAndCustomerCode = this.costPoolReplenishmentRepository.findByPoolCodesAndCustomerCode(poolCodes, customerCode);
    if (CollectionUtils.isEmpty(byPoolCodesAndCustomerCode)) {
      return new ArrayList<>(0);
    }
    List<CostPoolReplenishment> costPoolReplenishments = this.splicingProduct(byPoolCodesAndCustomerCode);
    return costPoolReplenishments;
  }

  @Override
  public List<CostPoolReplenishment> findByCustomerCode(String customerCode) {
    if (StringUtils.isBlank(customerCode)) {
      return new ArrayList<>(0);
    }
    return this.costPoolReplenishmentRepository.findByCustomerCode(customerCode);
  }

  /**
   * 创建验证
   *
   * @param costPoolReplenishment
   */
  private void createValidate(CostPoolReplenishment costPoolReplenishment) {
    Validate.notNull(costPoolReplenishment, "新增时，对象信息不能为空！");
    costPoolReplenishment.setId(null);
    Validate.notBlank(costPoolReplenishment.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(costPoolReplenishment.getCustomerCode(), "新增数据时，客户编码不能为空！");
    Validate.notBlank(costPoolReplenishment.getCustomerName(), "新增数据时，客户名称不能为空！");
    Validate.notNull(costPoolReplenishment.getFreezeAmount(), "新增数据时，冻结金额（数量）不能为空！");
    Validate.notNull(costPoolReplenishment.getHasUseAmount(), "新增数据时，已使用金额（数量）不能为空！");
    Validate.notNull(costPoolReplenishment.getOccupyAmount(), "新增数据时，占用金额（数量）不能为空！");
    Validate.notBlank(costPoolReplenishment.getPayType(), "新增数据时，支付方式不能为空！");
    Validate.notBlank(costPoolReplenishment.getPoolCode(), "新增数据时，费用池编号不能为空！");
    Validate.notBlank(costPoolReplenishment.getPoolGroup(), "新增数据时，费用池分组不能为空！");
    Validate.notBlank(costPoolReplenishment.getPoolType(), "新增数据时，费用池类型不能为空！");
    Validate.notNull(costPoolReplenishment.getTotalAmount(), "新增数据时，总金额（数量）不能为空！");
    Validate.notNull(costPoolReplenishment.getUsableAmount(), "新增数据时，剩余可使用金额（数量）不能为空！");
    Validate.notBlank(costPoolReplenishment.getUseType(), "新增数据时，费用使用类型不能为空！");

  }

  /**
   * 修改验证
   *
   * @param costPoolReplenishment
   */
  private void updateValidate(CostPoolReplenishment costPoolReplenishment) {
    Validate.notNull(costPoolReplenishment, "修改时，对象信息不能为空！");
    Validate.notBlank(costPoolReplenishment.getId(), "修改数据时，不能为空！");
    Validate.notBlank(costPoolReplenishment.getTenantCode(), "修改数据时，租户编号不能为空！");
    Validate.notBlank(costPoolReplenishment.getCustomerCode(), "修改数据时，客户编码不能为空！");
    Validate.notBlank(costPoolReplenishment.getCustomerName(), "修改数据时，客户名称不能为空！");
    Validate.notNull(costPoolReplenishment.getFreezeAmount(), "修改数据时，冻结金额（数量）不能为空！");
    Validate.notNull(costPoolReplenishment.getHasUseAmount(), "修改数据时，已使用金额（数量）不能为空！");
    Validate.notNull(costPoolReplenishment.getOccupyAmount(), "修改数据时，占用金额（数量）不能为空！");
    Validate.notBlank(costPoolReplenishment.getPayType(), "修改数据时，支付方式不能为空！");
    Validate.notBlank(costPoolReplenishment.getPoolCode(), "修改数据时，费用池编号不能为空！");
    Validate.notBlank(costPoolReplenishment.getPoolGroup(), "修改数据时，费用池分组不能为空！");
    Validate.notBlank(costPoolReplenishment.getPoolType(), "修改数据时，费用池类型不能为空！");
    Validate.notNull(costPoolReplenishment.getTotalAmount(), "修改数据时，总金额（数量）不能为空！");
    Validate.notNull(costPoolReplenishment.getUsableAmount(), "修改数据时，剩余可使用金额（数量）不能为空！");
    Validate.notBlank(costPoolReplenishment.getUseType(), "修改数据时，费用使用类型不能为空！");

  }

  /**
   * 调整验证
   *
   * @param costPoolReplenishmentDto
   */
  private void handleAdjustValidate(CostPoolReplenishmentDto costPoolReplenishmentDto) {
    Validate.notNull(costPoolReplenishmentDto, "新增时，对象信息不能为空！");
    Validate.notBlank(costPoolReplenishmentDto.getCustomerCode(), "新增数据时，客户编码不能为空！");
    Validate.notNull(costPoolReplenishmentDto.getAmount(), "新增数据时，金额不能为空！");
    Validate.isTrue(costPoolReplenishmentDto.getAmount().compareTo(BigDecimal.ZERO) > 0, "新增数据时，金额必须大于0");
    //验证客户信息
    List<CustomerVo> byCustomerCodes = customerVoService.findByCustomerCodes(Lists.newArrayList(costPoolReplenishmentDto.getCustomerCode()));
    Validate.isTrue(!CollectionUtils.isEmpty(byCustomerCodes), "客户不存在！");
    costPoolReplenishmentDto.setCustomerName(byCustomerCodes.get(0).getCustomerName());
    Validate.notBlank(costPoolReplenishmentDto.getCustomerName(), "新增数据时，客户名称不能为空！");
  }

  /**
   * 组装修改公共参数
   *
   * @param costPoolReplenishment
   */
  private void updateForm(CostPoolReplenishment costPoolReplenishment) {
    AbstractCrmUserIdentity loginUser = loginUserService.getAbstractLoginUser();
    costPoolReplenishment.setModifyAccount(loginUser.getAccount());
    costPoolReplenishment.setModifyName(loginUser.getUsername());
    costPoolReplenishment.setModifyTime(new Date());
  }

  /**
   * 创建表单信息
   *
   * @param costPoolReplenishment
   */
  private void createForm(CostPoolReplenishment costPoolReplenishment) {
    AbstractCrmUserIdentity loginUser = loginUserService.getAbstractLoginUser();
    Date date = new Date();
    costPoolReplenishment.setId(null);
    String ruleCode = StringUtils.join(PoolReplenishmentConstant.DISCOUNT_POOL_CODE, DateFormatUtils.format(new Date(), "yyyyMMdd"));
    String code = this.generateCodeService.generateCode(ruleCode, 1, 5, 2, TimeUnit.DAYS).get(0);
    costPoolReplenishment.setPoolCode(code);
    costPoolReplenishment.setCreateAccount(loginUser.getAccount());
    costPoolReplenishment.setCreateName(loginUser.getUsername());
    costPoolReplenishment.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    costPoolReplenishment.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    costPoolReplenishment.setCreateTime(date);
    costPoolReplenishment.setModifyAccount(loginUser.getAccount());
    costPoolReplenishment.setModifyName(loginUser.getUsername());
    costPoolReplenishment.setModifyTime(date);
    costPoolReplenishment.setTenantCode(TenantUtils.getTenantCode());
    costPoolReplenishment.setPayType(PoolPayTypeEnum.Replenishment.getDictCode());
    costPoolReplenishment.setUseType(PoolUseTypeEnum.DEFAULT.getDictCode());
    costPoolReplenishment.setPoolGroup(PoolGroupEnum.DEFAULT.getDictCode());
    costPoolReplenishment.setPoolType(PoolTypeEnum.Replenishment.getDictCode());
  }
}

