package com.biz.crm.dms.business.costpool.replenishment.local.service.observer;

import com.biz.crm.dms.business.costpool.replenishment.local.entity.CostPoolReplenishment;
import com.biz.crm.dms.business.costpool.replenishment.local.entity.CostPoolReplenishmentOperation;
import com.biz.crm.dms.business.costpool.replenishment.local.service.CostPoolReplenishmentOperationService;
import com.biz.crm.dms.business.costpool.replenishment.local.service.CostPoolReplenishmentService;
import com.biz.crm.dms.business.costpool.replenishment.sdk.dto.CostPoolReplenishmentOperationDto;
import com.biz.crm.dms.business.costpool.replenishment.sdk.enums.CostPoolReplenishmentOperationGroupEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolOperationTypeEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolTypeEnum;
import com.biz.crm.dms.business.costpool.sdk.enums.CostPoolSummaryEnum;
import com.biz.crm.dms.business.costpool.sdk.observer.CostPoolAmountStatisticsObserver;
import com.biz.crm.dms.business.costpool.sdk.vo.CostPoolAmountStatisticsVo;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * @author HanJiaJun
 * @describe:
 * @createTime 2022年04月11日 10:38:00
 */
@Component
public class CostPoolAmountStatisticsReplenishmentServerImpl implements CostPoolAmountStatisticsObserver {


  @Autowired(required = false)
  private CostPoolReplenishmentService costPoolReplenishmentService;
  @Autowired(required = false)
  private CostPoolReplenishmentOperationService costPoolReplenishmentOperationService;

  /**
   * 货补池统计
   *
   * @param customerCode 客户编码
   * @param beginDate    开始日期
   * @param endDate      结束日期
   * @return
   */
  @Override
  public List<CostPoolAmountStatisticsVo> findAmountStatisticsVoByCustomerCodeAndDate(String customerCode, Date beginDate, Date endDate) {
    Validate.notBlank(customerCode, "客户编码不能为空！");
    Validate.notNull(beginDate, "开始时间不能为空！");
    Validate.notNull(endDate, "结束时间不能为空！");
    Validate.isTrue(endDate.compareTo(beginDate) > -1, "开始时间应小于等于结束时间");
    List<CostPoolAmountStatisticsVo> res = new ArrayList<>();
    //通过客户编码查询货补池
    List<CostPoolReplenishment> byCustomerCode = costPoolReplenishmentService
        .findByCustomerCode(customerCode);
    //货补池编码
    List<String> poolCodes = byCustomerCode.stream()
        .filter(e -> StringUtils.isNotBlank(e.getPoolCode()))
        .map(CostPoolReplenishment::getPoolCode)
        .collect(Collectors.toList());
    //定义当前时间段和期初的数据集合
    List<CostPoolReplenishmentOperation> list = new ArrayList<>();
    List<CostPoolReplenishmentOperation> beforeDateList = new ArrayList<>();
    //边界校验货补池编号, 为空不进行查询
    if (!CollectionUtils.isEmpty(poolCodes)) {
      CostPoolReplenishmentOperationDto operationDto = new CostPoolReplenishmentOperationDto();
      operationDto.setPoolCodeList(poolCodes);
      operationDto.setStartTime(beginDate);
      operationDto.setEndTime(endDate);
      //查询当前时间货补池
      list = costPoolReplenishmentOperationService
          .findByPoolCodesAndDate(operationDto);
      //查询期初时间货补池
      CostPoolReplenishmentOperationDto beforeDateDto = new CostPoolReplenishmentOperationDto();
      beforeDateDto.setPoolCodeList(poolCodes);
      beforeDateDto.setStartTime(beginDate);
      beforeDateList = costPoolReplenishmentOperationService.findByPoolCodesAndBeforeDate(beforeDateDto);
    }
    //当前时间段记录
    Map<String, List<CostPoolReplenishmentOperation>> map = list.stream().collect(Collectors.groupingBy(CostPoolReplenishmentOperation::getOperationType));
    //期初记录
    Map<String, List<CostPoolReplenishmentOperation>> beforeDateMap = beforeDateList.stream().collect(Collectors.groupingBy(CostPoolReplenishmentOperation::getOperationType));
    //统计当前时间
    List<CostPoolAmountStatisticsVo> now = countData(map);
    BigDecimal nowAmount = now.stream().map(CostPoolAmountStatisticsVo::getTotalAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
    //统计期初
    List<CostPoolAmountStatisticsVo> before = countData(beforeDateMap);
    BigDecimal beforeAmount = before.stream().map(CostPoolAmountStatisticsVo::getTotalAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
    //计算期末
    BigDecimal afterAmount = beforeAmount.add(nowAmount);
    //添加上账和使用记录
    for (CostPoolAmountStatisticsVo vo : now) {
      vo.setTotalAmount(vo.getTotalAmount().abs());
    }
    res.addAll(now);
    //添加期初记录
    CostPoolAmountStatisticsVo beforeObj = new CostPoolAmountStatisticsVo();
    beforeObj.setTotalAmount(beforeAmount.abs());
    beforeObj.setPoolType(PoolTypeEnum.Replenishment.getKey());
    beforeObj.setSummary(CostPoolSummaryEnum.BEGIN.getKey());
    res.add(beforeObj);
    //添加期末记录
    CostPoolAmountStatisticsVo afterObj = new CostPoolAmountStatisticsVo();
    afterObj.setTotalAmount(afterAmount.abs());
    afterObj.setPoolType(PoolTypeEnum.Replenishment.getKey());
    afterObj.setSummary(CostPoolSummaryEnum.END.getKey());
    res.add(afterObj);
    return res;
  }

  /**
   * 统计数据
   *
   * @param map
   * @return
   */
  private List<CostPoolAmountStatisticsVo> countData(Map<String, List<CostPoolReplenishmentOperation>> map) {
    List<CostPoolAmountStatisticsVo> count = new ArrayList<>();
    //上账
    AtomicReference<BigDecimal> onSum = new AtomicReference<>(new BigDecimal(BigInteger.ZERO));
    CostPoolAmountStatisticsVo onVo = new CostPoolAmountStatisticsVo();
    //使用
    AtomicReference<BigDecimal> useSum = new AtomicReference<>(new BigDecimal(BigInteger.ZERO));
    CostPoolAmountStatisticsVo useVo = new CostPoolAmountStatisticsVo();
    //占用
    AtomicReference<BigDecimal> occupySum = new AtomicReference<>(new BigDecimal(BigInteger.ZERO));
    CostPoolAmountStatisticsVo occupyVo = new CostPoolAmountStatisticsVo();
    map.forEach((k, v) -> {
      //统计上账类型 ，操作类型编号是1,2,3,7,8,9
      if (k.equals(PoolOperationTypeEnum.INIT.getKey())
          || k.equals(PoolOperationTypeEnum.MANUAL_ACCOUNT.getKey())
          || k.equals(PoolOperationTypeEnum.REBATE_ACCOUNT.getKey())
          || k.equals(PoolOperationTypeEnum.FREEZE.getKey())
          || k.equals(PoolOperationTypeEnum.UNFREEZE.getKey())
          || k.equals(PoolOperationTypeEnum.ACT_ACCOUNT.getKey())
      ) {
        for (CostPoolReplenishmentOperation op : v) {
          onSum.getAndUpdate(b -> b.add(op.getOperationAmount()));
        }
      }
      //统计使用 ，操作类型编号是5,6
      else if (k.equals(PoolOperationTypeEnum.MANUAL_INDUCE.getKey())
          || k.equals(PoolOperationTypeEnum.ORDER_USE.getKey())) {
        for (CostPoolReplenishmentOperation op : v) {
          useSum.getAndUpdate(b -> b.add(op.getOperationAmount()));
        }
      }
      //统计使用 ，操作类型编号是5,6
      else if (k.equals(PoolOperationTypeEnum.OCCUPY_USE.getKey())
          || k.equals(PoolOperationTypeEnum.OCCUPY_RELEASE.getKey())) {
        for (CostPoolReplenishmentOperation op : v) {
          occupySum.getAndUpdate(b -> b.add(op.getOperationAmount()));
        }
      }
    });
    //合并数据
    onVo.setSummary(CostPoolReplenishmentOperationGroupEnum.ON.getKey());
    onVo.setTotalAmount(onSum.get());
    onVo.setPoolType(PoolTypeEnum.Replenishment.getKey());
    useVo.setSummary(CostPoolReplenishmentOperationGroupEnum.USE.getKey());
    useVo.setTotalAmount(useSum.get());
    useVo.setPoolType(PoolTypeEnum.Replenishment.getKey());
    occupyVo.setSummary(CostPoolReplenishmentOperationGroupEnum.OCCUPY.getKey());
    occupyVo.setTotalAmount(occupySum.get());
    occupyVo.setPoolType(PoolTypeEnum.Replenishment.getKey());
    count.add(onVo);
    count.add(useVo);
    count.add(occupyVo);
    return count;
  }

}
