package com.biz.crm.dms.business.delivery.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.dms.business.delivery.local.repository.DeliveryAttachmentRepository;
import com.biz.crm.dms.business.delivery.local.repository.DeliveryRepository;
import com.biz.crm.dms.business.delivery.sdk.event.DeliveryEventListener;
import com.biz.crm.dms.business.delivery.sdk.service.DeliveryDetailVoService;
import com.biz.crm.dms.business.delivery.sdk.dto.DeliveryDetailDto;
import com.biz.crm.dms.business.delivery.sdk.enums.InvoiceStatusEnum;
import com.biz.crm.dms.business.delivery.sdk.enums.OutboundStatusEnum;
import com.biz.crm.dms.business.delivery.sdk.enums.ReceiptStatusEnum;
import com.biz.crm.dms.business.delivery.sdk.vo.DeliveryAttachmentVo;
import com.biz.crm.dms.business.delivery.sdk.vo.DeliveryDetailVo;
import com.biz.crm.dms.business.delivery.sdk.vo.DeliveryVo;
import com.biz.crm.dms.business.delivery.local.entity.DeliveryAttachmentEntity;
import com.biz.crm.dms.business.delivery.local.entity.DeliveryEntity;
import com.biz.crm.dms.business.delivery.local.repository.DeliveryVoRepository;
import com.biz.crm.dms.business.delivery.sdk.constant.DeliveryConstant;
import com.biz.crm.dms.business.delivery.sdk.dto.DeliveryAttachmentDto;
import com.biz.crm.dms.business.delivery.sdk.dto.DeliveryDto;
import com.biz.crm.dms.business.delivery.sdk.service.DeliveryVoService;
import com.biz.crm.dms.business.order.common.sdk.enums.MaterielOrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCategoryEnum;
import com.biz.crm.dms.business.order.sdk.service.OrderVoService;
import com.biz.crm.dms.business.order.sdk.vo.OrderDetailVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderVo;
import com.biz.crm.dms.business.psi.product.sdk.dto.delivery.ProductDeliveryBillCreateDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.delivery.ProductDeliveryBillDetailDto;
import com.biz.crm.dms.business.psi.product.sdk.enums.productstock.AssociatedDocumentsType;
import com.biz.crm.dms.business.psi.product.sdk.enums.productstock.ProductStockOperationType;
import com.biz.crm.dms.business.psi.product.sdk.enums.productstock.StockType;
import com.biz.crm.dms.business.psi.product.sdk.service.productstock.ProductDeliveryBillVoService;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 发货单Vo服务接口实现类
 *
 * @author pengxi
 * @date 2022/02/16
 */
@Slf4j
@Service
public class DeliveryVoServiceImpl implements DeliveryVoService {

  @Autowired(required = false)
  private DeliveryRepository deliveryRepository;

  @Autowired(required = false)
  private DeliveryVoRepository deliveryVoRepository;

  @Autowired(required = false)
  private DeliveryAttachmentRepository deliveryAttachmentRepository;

  @Autowired(required = false)
  private DeliveryDetailVoService deliveryDetailVoService;

  @Autowired(required = false)
  private ProductDeliveryBillVoService productDeliveryBillVoService;

  @Autowired(required = false)
  private OrderVoService orderVoService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private List<DeliveryEventListener> deliveryEventListeners;

  @Autowired(required = false)
  private ProductVoService productVoService;

  @Override
  public Page<DeliveryVo> findByConditions(Pageable pageable, DeliveryDto dto) {
    Page<DeliveryVo> deliveryVoPage = this.deliveryVoRepository.findByConditions(pageable, dto);
    if (deliveryVoPage.getTotal() == 0) {
      return deliveryVoPage;
    }
    Set<String> deliveryCodes = deliveryVoPage.getRecords().stream().map(DeliveryVo::getDeliveryCode).collect(Collectors.toSet());
    List<DeliveryDetailVo> detailVos = this.deliveryDetailVoService.findByDeliveryCodes(deliveryCodes);
    Map<String, List<DeliveryDetailVo>> detailMap = detailVos.stream().collect(Collectors.groupingBy(DeliveryDetailVo::getDeliveryCode));
    deliveryVoPage.getRecords().forEach(deliveryVo->{
      List<DeliveryDetailVo> deliveryDetailVos = detailMap.get(deliveryVo.getDeliveryCode());
      if (CollectionUtils.isNotEmpty(deliveryDetailVos)) {
        // 汇总发货数量和金额
        BigDecimal totalDeliveryQuantity = Optional.ofNullable(deliveryDetailVos).orElse(Lists.newArrayList()).stream().map(DeliveryDetailVo::getDeliveryQuantity).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal totalDeliverAmount = Optional.ofNullable(deliveryDetailVos).orElse(Lists.newArrayList()).stream().map(DeliveryDetailVo::getDeliverAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
        deliveryVo.setTotalDeliveryQuantity(totalDeliveryQuantity);
        deliveryVo.setTotalDeliverAmount(totalDeliverAmount);
      }
      // 把明细信息汇总显示到头上
      this.buildDeliveryHeadInfo(deliveryVo, deliveryDetailVos);
    });
    return deliveryVoPage;
  }

  @Override
  public DeliveryVo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    DeliveryEntity entity = this.deliveryRepository.getById(id);
    if (entity == null) {
      return null;
    }
    DeliveryVo deliveryVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, DeliveryVo.class, HashSet.class, ArrayList.class);
    // 查询发货单明细
    List<DeliveryDetailVo> deliveryDetailVos = this.deliveryDetailVoService.findDetailByDeliveryCodes(Collections.singleton(deliveryVo.getDeliveryCode()));
    deliveryVo.setDeliveryDetailVos(deliveryDetailVos);
    if (CollectionUtils.isNotEmpty(deliveryDetailVos)) {
      // 汇总发货数量和金额
      BigDecimal totalDeliveryQuantity = Optional.ofNullable(deliveryDetailVos).orElse(Lists.newArrayList()).stream().map(DeliveryDetailVo::getDeliveryQuantity).reduce(BigDecimal.ZERO, BigDecimal::add);
      BigDecimal totalDeliverAmount = Optional.ofNullable(deliveryDetailVos).orElse(Lists.newArrayList()).stream().map(DeliveryDetailVo::getDeliverAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
      deliveryVo.setTotalDeliveryQuantity(totalDeliveryQuantity);
      deliveryVo.setTotalDeliverAmount(totalDeliverAmount);
    }
    // 把明细信息汇总显示到头上
    this.buildDeliveryHeadInfo(deliveryVo, deliveryDetailVos);
    // 组装商品图片
    this.buildPictureMediaList(deliveryDetailVos);
    // 查询发货单附件
    List<DeliveryAttachmentEntity> attachmentEntities = this.deliveryAttachmentRepository.findByDeliveryCodes(TenantUtils.getTenantCode(), Collections.singleton(deliveryVo.getDeliveryCode()));
    if (CollectionUtils.isEmpty(attachmentEntities)) {
      return deliveryVo;
    }
    List<DeliveryAttachmentVo> attachmentVos = (List<DeliveryAttachmentVo>) this.nebulaToolkitService.copyCollectionByWhiteList(attachmentEntities, DeliveryAttachmentEntity.class, DeliveryAttachmentVo.class, HashSet.class, ArrayList.class);
    deliveryVo.setDeliveryAttachmentVos(attachmentVos);
    return deliveryVo;
  }

  @Override
  @Transactional
  public void create(DeliveryDto dto) {
    this.createValidation(dto);
    // redis生成发货单编码，编码规则为FHD+年月日+5位顺序数。每天都从00001开始编
    String ruleCode = StringUtils.join(DeliveryConstant.DELIVERY_CODE, DateFormatUtils.format(new Date(), "yyyyMMdd"));
    List<String> codeList = this.generateCodeService.generateCode(ruleCode, 1, 5, 2, TimeUnit.DAYS);
    Validate.isTrue(CollectionUtils.isNotEmpty(codeList), "添加信息时，生成发货单编码失败！");
    dto.setDeliveryCode(codeList.get(0));
    DeliveryEntity dbEntity = this.deliveryRepository.findByDeliveryCode(TenantUtils.getTenantCode(), codeList.get(0));
    Validate.isTrue(Objects.isNull(dbEntity), String.format("添加信息时，发货单编码[%s]已存在！", codeList.get(0)));
    DeliveryEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, DeliveryEntity.class, HashSet.class, ArrayList.class);
    entity.setTenantCode(TenantUtils.getTenantCode());
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setReceiptStatus(ReceiptStatusEnum.START.getDictCode());
    this.deliveryRepository.save(entity);
    // 批量新增发货单明细
    this.saveBatchDeliveryDetail(entity, dto.getDeliveryDetailDtoList());
    // 批量新增发货单附件
    this.saveBatchDeliveryAttachment(entity, dto.getDeliveryAttachmentDtoList());
    // 增加订单上发货数量及状态更新
    this.increaseOrderDelivery(dto.getDeliveryDetailDtoList());
    // 发货单信息被创建事件通知
    if(!CollectionUtils.isEmpty(deliveryEventListeners)) {
      deliveryEventListeners.forEach(deliveryEventListener -> deliveryEventListener.onCreated(dto));
    }
  }

  @Override
  @Transactional
  public void createDeliveryAndOutbound(DeliveryDto dto) {
    // 创建发货单
    this.create(dto);
    // 创建出库单
    this.createOutbound(dto);
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空！");
    List<DeliveryEntity> entities = this.deliveryRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.deliveryRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空！");
    List<DeliveryEntity> entities = this.deliveryRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    // 找出修改前正常状态的列表
    List<DeliveryEntity> enableEntities = entities.stream().filter(e->EnableStatusEnum.ENABLE.getCode().equals(e.getEnableStatus())).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(enableEntities)) {
      return;
    }
    Set<String> deliveryCodes = enableEntities.stream().map(DeliveryEntity::getDeliveryCode).collect(Collectors.toSet());
    List<DeliveryDetailVo> deliveryDetailVos = this.deliveryDetailVoService.findDetailByDeliveryCodes(deliveryCodes);
    Optional<DeliveryDetailVo> optional = Optional.ofNullable(deliveryDetailVos).orElse(Lists.newArrayList())
        .stream().filter(
            e-> !InvoiceStatusEnum.START.getDictCode().equals(e.getInvoiceStatus())
            || !OutboundStatusEnum.START.getDictCode().equals(e.getOutboundStatus()))
        .findAny();
    Validate.isTrue(!optional.isPresent(), "只能作废待出库且待开票的发货单！");
    this.deliveryRepository.updateEnableStatusByIds(enableEntities.stream().map(DeliveryEntity::getId).collect(Collectors.toList()), EnableStatusEnum.DISABLE);
    // 作废明细
    this.deliveryDetailVoService.updateEnableStatusByDeliveryCodes(deliveryCodes, EnableStatusEnum.DISABLE);
    // 减少订单上发货数量及状态更新
    this.reduceOrderDelivery(enableEntities);
    // 发货单信息被作废时事件通知
    if(!CollectionUtils.isEmpty(deliveryEventListeners)) {
      List<DeliveryVo> deliveryVos = (List<DeliveryVo>) this.nebulaToolkitService.copyCollectionByWhiteList(enableEntities, DeliveryEntity.class, DeliveryVo.class, HashSet.class, ArrayList.class);
      Map<String, List<DeliveryDetailVo>> deliveryDetailMap = Optional.ofNullable(deliveryDetailVos).orElse(Lists.newArrayList()).stream().collect(Collectors.groupingBy(DeliveryDetailVo::getDeliveryCode));
      deliveryVos.forEach(d-> d.setDeliveryDetailVos(deliveryDetailMap.get(d.getDeliveryCode())));
      deliveryEventListeners.forEach(deliveryEventListener -> deliveryEventListener.onDisabled(deliveryVos));
    }
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空！");
    List<DeliveryEntity> entities = this.deliveryRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    Optional<DeliveryEntity> optional = entities.stream().filter(e->EnableStatusEnum.DISABLE.getCode().equals(e.getEnableStatus())).findAny();
    Validate.isTrue(optional.isPresent(), "只能删除已作废的发货单！");
    this.deliveryRepository.updateDelFlagByIds(ids);
    // 发货单信息被删除时事件通知
    if(!CollectionUtils.isEmpty(deliveryEventListeners)) {
      List<DeliveryVo> deliveryVos = (List<DeliveryVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, DeliveryEntity.class, DeliveryVo.class, HashSet.class, ArrayList.class);
      Set<String> deliveryCodes = deliveryVos.stream().map(DeliveryVo::getDeliveryCode).collect(Collectors.toSet());
      List<DeliveryDetailVo> deliveryDetailVos = this.deliveryDetailVoService.findDetailByDeliveryCodes(deliveryCodes);
      Map<String, List<DeliveryDetailVo>> deliveryDetailMap = Optional.ofNullable(deliveryDetailVos).orElse(Lists.newArrayList()).stream().collect(Collectors.groupingBy(DeliveryDetailVo::getDeliveryCode));
      deliveryVos.forEach(d-> d.setDeliveryDetailVos(deliveryDetailMap.get(d.getDeliveryCode())));
      deliveryEventListeners.forEach(deliveryEventListener -> deliveryEventListener.onDeleted(deliveryVos));
    }
  }

  @Override
  @Transactional
  public void receivingByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空！");
    List<DeliveryEntity> entities = this.deliveryRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.deliveryRepository.receivingByIds(ids);
    // 更新收货数量
    Set<String> deliveryCodes = entities.stream().map(DeliveryEntity::getDeliveryCode).collect(Collectors.toSet());
    this.deliveryDetailVoService.receivingDeliveryCodes(deliveryCodes);
    // 发货单信息被收货时事件通知
    if(!CollectionUtils.isEmpty(deliveryEventListeners)) {
      List<DeliveryVo> deliveryVos = (List<DeliveryVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, DeliveryEntity.class, DeliveryVo.class, HashSet.class, ArrayList.class);
      List<DeliveryDetailVo> deliveryDetailVos = this.deliveryDetailVoService.findDetailByDeliveryCodes(deliveryCodes);
      Map<String, List<DeliveryDetailVo>> deliveryDetailMap = Optional.ofNullable(deliveryDetailVos).orElse(Lists.newArrayList()).stream().collect(Collectors.groupingBy(DeliveryDetailVo::getDeliveryCode));
      deliveryVos.forEach(d-> d.setDeliveryDetailVos(deliveryDetailMap.get(d.getDeliveryCode())));
      deliveryEventListeners.forEach(deliveryEventListener -> deliveryEventListener.onReceived(deliveryVos));
    }
  }

  @Override
  public List<DeliveryVo> findByOrderCode(String orderCode) {
    if (StringUtils.isBlank(orderCode)) {
      return null;
    }
    List<DeliveryDetailVo> deliveryDetailVos = this.deliveryDetailVoService.findByOrderCodes(Collections.singleton(orderCode));
    if (CollectionUtils.isEmpty(deliveryDetailVos)) {
      return null;
    }
    Map<String, List<DeliveryDetailVo>> detailVoMap = deliveryDetailVos.stream().collect(Collectors.groupingBy(DeliveryDetailVo::getDeliveryCode));
    Set<String> deliveryCodes = deliveryDetailVos.stream().map(DeliveryDetailVo::getDeliveryCode).collect(Collectors.toSet());
    List<DeliveryEntity> deliveryEntities = this.deliveryRepository.findByDeliveryCodes(deliveryCodes);
    if (CollectionUtils.isEmpty(deliveryEntities)) {
      return null;
    }
    List<DeliveryVo> deliveryVos = (List<DeliveryVo>) this.nebulaToolkitService.copyCollectionByWhiteList(deliveryEntities, DeliveryEntity.class, DeliveryVo.class, HashSet.class, ArrayList.class);
    List<OrderVo> orderVos = this.orderVoService.findByOrderCodes(Collections.singletonList(orderCode));
    Map<String, OrderVo> orderMap = orderVos.stream().collect(Collectors.toMap(OrderVo::getOrderCode, Function.identity()));
    deliveryVos.forEach(deliveryVo -> {
      List<DeliveryDetailVo> detailVoList = detailVoMap.get(deliveryVo.getDeliveryCode());
      if (CollectionUtils.isNotEmpty(detailVoList)) {
        detailVoList.forEach(deliveryDetailVo -> {
          OrderVo orderVo = orderMap.get(deliveryDetailVo.getOrderCode());
          List<OrderDetailVo> orderDetailVos = orderVo.getOrderDetails();
          Map<String, OrderDetailVo> orderDetailVoMap = Optional.ofNullable(orderDetailVos).orElse(new ArrayList<>()).stream().collect(Collectors.toMap(d->d.getOrderCode() + "_" + d.getOrderDetailCode(), Function.identity()));
          OrderDetailVo orderDetailVo = orderDetailVoMap.get(deliveryDetailVo.getOrderCode() + "_" + deliveryDetailVo.getOrderDetailCode());
          if (orderDetailVo != null) {
            // 本次已发销售金额
            deliveryDetailVo.setDeliverSalesAmount(deliveryDetailVo.getDeliveryQuantity().divide(orderDetailVo.getQuantity(), 4, RoundingMode.HALF_UP).multiply(orderDetailVo.getSalesAmount()));
            // 本次已发实际支付金额
            deliveryDetailVo.setDeliverShouldPaymentAmount(deliveryDetailVo.getDeliveryQuantity().divide(orderDetailVo.getQuantity(), 4, RoundingMode.HALF_UP).multiply(orderDetailVo.getShouldPaymentAmount()));
            deliveryDetailVo.setOrderType(orderVo.getOrderType());
            deliveryDetailVo.setOriginalOrderCode(orderVo.getOriginalOrderCode());
            deliveryDetailVo.setGoodsCode(orderDetailVo.getGoodsCode());
            deliveryDetailVo.setGoodsName(orderDetailVo.getGoodsName());
            deliveryDetailVo.setItemType(orderDetailVo.getItemType());
            deliveryDetailVo.setPresetUnitPrice(orderDetailVo.getPresetUnitPrice());
            deliveryDetailVo.setShouldPaymentAmount(orderDetailVo.getShouldPaymentAmount());
            deliveryDetailVo.setSalesAmount(orderDetailVo.getSalesAmount());
            deliveryDetailVo.setQuantity(orderDetailVo.getQuantity());
            deliveryDetailVo.setUnite(orderDetailVo.getUnite());
            deliveryDetailVo.setSpec(orderDetailVo.getSpec());
          }
        });
      }
      // 把明细信息汇总显示到头上
      this.buildDeliveryHeadInfo(deliveryVo, detailVoList);
      // 组装商品图片
      this.buildPictureMediaList(detailVoList);
      deliveryVo.setDeliveryDetailVos(detailVoList);
    });
    return deliveryVos;
  }

  //////////////////////////////////////////////// 下面是工具方法 ////////////////////////////////////////////////

  /**
   * 组装商品图片
   */
  private void buildPictureMediaList(List<DeliveryDetailVo> deliveryDetailVos){
    if (CollectionUtils.isEmpty(deliveryDetailVos)) {
      return;
    }
    // 查询sku的图片  规格详情
    List<String> productCodes = deliveryDetailVos.stream().map(DeliveryDetailVo::getGoodsCode).distinct().collect(Collectors.toList());
    List<ProductVo> productList = productVoService.findDetailsByIdsOrProductCodes(null, productCodes);
    if (CollectionUtils.isEmpty(productList)) {
      return;
    }
    Map<String, ProductVo> productVoMap = productList.stream().collect(Collectors.toMap(ProductVo::getProductCode,Function.identity()));
    deliveryDetailVos.forEach(dd->{
      ProductVo productVo = productVoMap.get(dd.getGoodsCode());
      if (productVo!=null) {
        dd.setPictureMediaList(productVo.getPictureMediaList());
      }
    });
  }

  /**
   * 把明细信息汇总显示到头上
   */
  private void buildDeliveryHeadInfo(DeliveryVo deliveryVo, List<DeliveryDetailVo> deliveryDetailVos){
    if (CollectionUtils.isEmpty(deliveryDetailVos)) {
      return;
    }
    Set<String> orderTypes = new HashSet<>(deliveryDetailVos.size());
    Set<String> orderCodes = new HashSet<>(deliveryDetailVos.size());
    Set<String> outboundCodes = new HashSet<>(deliveryDetailVos.size());
    Set<String> outboundStatus = new HashSet<>(deliveryDetailVos.size());
    Set<String> invoiceStatus = new HashSet<>(deliveryDetailVos.size());
    Set<String> relateCodes = new HashSet<>(deliveryDetailVos.size());
    Set<String> relateNames = new HashSet<>(deliveryDetailVos.size());
    Set<String> orgCodes = new HashSet<>(deliveryDetailVos.size());
    Set<String> orgNames = new HashSet<>(deliveryDetailVos.size());
    // 是否待出库
    AtomicReference<Boolean> isPendingOutbound = new AtomicReference<>(Boolean.TRUE);
    // 是否完全出库
    AtomicReference<Boolean> isCompletelyOutbound = new AtomicReference<>(Boolean.TRUE);
    deliveryDetailVos.forEach(dd->{
      if (StringUtils.isNotBlank(dd.getOrderType())) {
        orderTypes.add(dd.getOrderType());
      }
      if (StringUtils.isNotBlank(dd.getOrderCode())) {
        orderCodes.add(dd.getOrderCode());
      }
      if (StringUtils.isNotBlank(dd.getOutboundCode())) {
        outboundCodes.add(dd.getOutboundCode());
      }
      if (StringUtils.isNotBlank(dd.getOutboundStatus())) {
        outboundStatus.add(dd.getOutboundStatus());
        OutboundStatusEnum outboundStatusEnum = OutboundStatusEnum.codeToEnum(dd.getOutboundStatus());
        switch (outboundStatusEnum) {
          case START:
            // 只要有一个待出库，则显示状态不能是完全出库
            isCompletelyOutbound.set(Boolean.FALSE);
            break;
          case PARTIAL:
            // 只要有一个部分出库，则显示状态不能是完全出库
            isCompletelyOutbound.set(Boolean.FALSE);
            // 只要有一个部分出库，则显示状态不能是待出库
            isPendingOutbound.set(Boolean.FALSE);
            break;
          case COMPLETE:
            // 只要有一个完全出库，则显示状态不能是待出库
            isPendingOutbound.set(Boolean.FALSE);
            break;
          default:
            break;
        }
      }
      if (StringUtils.isNotBlank(dd.getInvoiceStatus())) {
        invoiceStatus.add(dd.getInvoiceStatus());
      }
      if (MaterielOrderTypeEnum.ORGANIZATION.getDictCode().equals(dd.getOrderType())) {
        if (StringUtils.isNotBlank(dd.getRelateCode())) {
          orgCodes.add(dd.getRelateCode());
        }
        if (StringUtils.isNotBlank(dd.getRelateName())) {
          orgNames.add(dd.getRelateName());
        }
        if (StringUtils.isNotBlank(dd.getOrgCode())) {
          orgCodes.add(dd.getOrgCode());
        }
        if (StringUtils.isNotBlank(dd.getOrgName())) {
          orgNames.add(dd.getOrgName());
        }
      } else {
        if (StringUtils.isNotBlank(dd.getRelateCode())) {
          relateCodes.add(dd.getRelateCode());
        }
        if (StringUtils.isNotBlank(dd.getRelateName())) {
          relateNames.add(dd.getRelateName());
        }
      }
    });
    // 出库状态的处理
    if(isCompletelyOutbound.get()) {
      deliveryVo.setOutboundStatus(OutboundStatusEnum.COMPLETE.getDictCode());
    } else if (isPendingOutbound.get()) {
      deliveryVo.setOutboundStatus(OutboundStatusEnum.START.getDictCode());
    } else {
      deliveryVo.setOutboundStatus(OutboundStatusEnum.PARTIAL.getDictCode());
    }
    deliveryVo.setOrderType(StringUtils.join(orderTypes.toArray(),","));
    deliveryVo.setOrderCodes(StringUtils.join(orderCodes.toArray(),","));
    deliveryVo.setOutboundCodes(StringUtils.join(outboundCodes.toArray(),","));
    deliveryVo.setInvoiceStatus(StringUtils.join(invoiceStatus.toArray(),","));
    deliveryVo.setRelateCodes(StringUtils.join(relateCodes.toArray(),","));
    deliveryVo.setRelateNames(StringUtils.join(relateNames.toArray(),","));
    deliveryVo.setOrgCodes(StringUtils.join(orgCodes.toArray(),","));
    deliveryVo.setOrgNames(StringUtils.join(orgNames.toArray(),","));
  }

  /**
   * 构建发货单明细
   */
  private void saveBatchDeliveryDetail(DeliveryEntity entity, List<DeliveryDetailDto> deliveryDetailDtoList) {
    // redis生成发货单编码，编码规则为FHG+年月日+5位顺序数。每天都从00001开始编
    String ruleCode = StringUtils.join(DeliveryConstant.DELIVERY_DETAIL_CODE, DateFormatUtils.format(new Date(), "yyyyMMdd"));
    List<String> codeList = this.generateCodeService.generateCode(ruleCode, deliveryDetailDtoList.size(), 5, 2, TimeUnit.DAYS);
    Validate.isTrue(CollectionUtils.isNotEmpty(codeList), "添加信息时，生成发货单行编码失败！");
    // 发货单明细行号与订单明细行号保持一致
    for (int i = 0, deliveryDetailDtoListSize = deliveryDetailDtoList.size(); i < deliveryDetailDtoListSize; i++) {
      DeliveryDetailDto detailDto = deliveryDetailDtoList.get(i);
      detailDto.setTenantCode(entity.getTenantCode());
      detailDto.setDeliveryCode(entity.getDeliveryCode());
      detailDto.setDeliveryDetailCode(codeList.get(i));
      detailDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      detailDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      detailDto.setOutboundStatus(OutboundStatusEnum.START.getDictCode());
      detailDto.setInvoiceStatus(InvoiceStatusEnum.START.getDictCode());
      detailDto.setOrderCategory(entity.getOrderCategory());
    }
    this.deliveryDetailVoService.createBatch(deliveryDetailDtoList);
  }

  /**
   * 批量新增发货单附件
   */
  private void saveBatchDeliveryAttachment(DeliveryEntity entity, List<DeliveryAttachmentDto> deliveryAttachmentDtoList) {
    if (CollectionUtils.isEmpty(deliveryAttachmentDtoList)) {
      return;
    }
    List<DeliveryAttachmentEntity> attachmentEntities = Lists.newArrayList();
    deliveryAttachmentDtoList.forEach(d->{
      DeliveryAttachmentEntity dimensionEntity = new DeliveryAttachmentEntity();
      dimensionEntity.setTenantCode(entity.getTenantCode());
      dimensionEntity.setDeliveryCode(entity.getDeliveryCode());
      dimensionEntity.setFileType(d.getFileType());
      dimensionEntity.setFileName(d.getFileName());
      dimensionEntity.setRelativePath(d.getRelativePath());
      attachmentEntities.add(dimensionEntity);
    });
    if (CollectionUtils.isEmpty(attachmentEntities)) {
      return;
    }
    this.deliveryAttachmentRepository.saveBatch(attachmentEntities);
  }

  /**
   * 增加订单上发货数量及状态更新
   * 正向（即增加发货数量）
   */
  private void increaseOrderDelivery(List<DeliveryDetailDto> detailDtoList) {
    if(CollectionUtils.isEmpty(detailDtoList)) {
      return;
    }
    Map<String, List<DeliveryDetailDto>> deliveryDetailMap = detailDtoList.stream().collect(Collectors.groupingBy(DeliveryDetailDto::getOrderCode));
    deliveryDetailMap.forEach((orderCode, deliveryDetailDtoList) -> {
      BigDecimal totalDeliveryQuantity = Optional.ofNullable(deliveryDetailDtoList).orElse(Lists.newArrayList()).stream().map(DeliveryDetailDto::getDeliveryQuantity).reduce(BigDecimal.ZERO, BigDecimal::add);
      // 正向
      if (BigDecimal.ZERO.compareTo(totalDeliveryQuantity) < 1) {
        log.info("作废发货单时增加订单{}发货数量{}及状态更新", orderCode, totalDeliveryQuantity);
        this.orderVoService.updateOrderStatusByDeliveryQuantity(orderCode, totalDeliveryQuantity, Boolean.TRUE);
      }
    });
  }

  /**
   * 减少订单上发货数量及状态更新
   * 反向（即减少发货数量）
   */
  private void reduceOrderDelivery(List<DeliveryEntity> entities) {
    if(CollectionUtils.isEmpty(entities)) {
      return;
    }
    Set<String> deliveryCodes = entities.stream().map(DeliveryEntity::getDeliveryCode).collect(Collectors.toSet());
    List<DeliveryDetailVo> deliveryDetailVos = deliveryDetailVoService.findByDeliveryCodes(deliveryCodes);
    Map<String, List<DeliveryDetailVo>> deliveryDetailMap = deliveryDetailVos.stream().collect(Collectors.groupingBy(DeliveryDetailVo::getOrderCode));
    deliveryDetailMap.forEach((orderCode, deliveryDetailVoList) -> {
      BigDecimal totalDeliveryQuantity = Optional.ofNullable(deliveryDetailVoList).orElse(Lists.newArrayList()).stream().map(DeliveryDetailVo::getDeliveryQuantity).reduce(BigDecimal.ZERO, BigDecimal::add);
      // 反向
      if (BigDecimal.ZERO.compareTo(totalDeliveryQuantity) < 1) {
        log.info("作废发货单时减少订单{}发货数量{}及状态更新", orderCode, totalDeliveryQuantity);
        this.orderVoService.updateOrderStatusByDeliveryQuantity(orderCode, totalDeliveryQuantity, Boolean.FALSE);
      }
    });
  }

  /**
   * 创建出库单，按仓库分组
   */
  private void createOutbound(DeliveryDto dto) {
    List<DeliveryDetailDto> deliveryDetailDtoList = dto.getDeliveryDetailDtoList();
    Map<String, List<DeliveryDetailDto>> detailMap = deliveryDetailDtoList.stream().collect(Collectors.groupingBy(DeliveryDetailDto::getCurrentWarehouseCode));
    detailMap.forEach((warehouseCode,itemDetailDtoList)->{
      ProductDeliveryBillCreateDto productDeliveryBillCreateDto = new ProductDeliveryBillCreateDto();
      productDeliveryBillCreateDto.setDeliveryTime(new Date());
      productDeliveryBillCreateDto.setRelationShipmentOrderCode(dto.getDeliveryCode());
      if (OrderCategoryEnum.SALES_ORDER.getDictCode().equals(dto.getOrderCategory())) {
        // 销售订单
        productDeliveryBillCreateDto.setProductStockOperationType(ProductStockOperationType.SALE_DELIVER.name());
        productDeliveryBillCreateDto.setRelationShipmentOrderType(AssociatedDocumentsType.SALE.name());
        productDeliveryBillCreateDto.setType(StockType.PRODUCT.name());
      } else if (OrderCategoryEnum.MATERIAL_ORDER.getDictCode().equals(dto.getOrderCategory())) {
        // 促销物料订单
        productDeliveryBillCreateDto.setProductStockOperationType(ProductStockOperationType.MATERIAL_ORDER_DELIVER.name());
        productDeliveryBillCreateDto.setRelationShipmentOrderType(AssociatedDocumentsType.MATERIAL_STORE_PURCHASE.name());
        productDeliveryBillCreateDto.setType(StockType.MATERIAL.name());
      } else {
        Validate.isTrue(Boolean.FALSE, "未知订单类别！");
      }
      List<ProductDeliveryBillDetailDto> productDeliveryBillDetailDtoList = new ArrayList<>();
      itemDetailDtoList.forEach(deliveryDetailDto->{
        productDeliveryBillCreateDto.setDeliveryWarehouseCode(deliveryDetailDto.getCurrentWarehouseCode());
        productDeliveryBillCreateDto.setDeliveryWarehouseName(deliveryDetailDto.getCurrentWarehouseName());
        ProductDeliveryBillDetailDto productDeliveryBillDetailDto = new ProductDeliveryBillDetailDto();
        productDeliveryBillDetailDto.setDeliveryWarehouseCode(deliveryDetailDto.getCurrentWarehouseCode());
        productDeliveryBillDetailDto.setDeliveryWarehouseName(deliveryDetailDto.getCurrentWarehouseName());
        productDeliveryBillDetailDto.setCustomerCode(deliveryDetailDto.getRelateCode());
        productDeliveryBillDetailDto.setCustomerName(deliveryDetailDto.getRelateName());
        productDeliveryBillDetailDto.setOrderCode(deliveryDetailDto.getOrderCode());
        productDeliveryBillDetailDto.setOrderItemCode(deliveryDetailDto.getOrderDetailCode());
        productDeliveryBillDetailDto.setItemType(deliveryDetailDto.getItemType());
        productDeliveryBillDetailDto.setProductCode(deliveryDetailDto.getGoodsCode());
        productDeliveryBillDetailDto.setProductName(deliveryDetailDto.getGoodsName());
        productDeliveryBillDetailDto.setProductUnit(deliveryDetailDto.getUnite());
        productDeliveryBillDetailDto.setProductSpec(deliveryDetailDto.getSpec());
        productDeliveryBillDetailDto.setDeliveryQuantity(deliveryDetailDto.getDeliveryQuantity());
        productDeliveryBillDetailDto.setCurrentQuantity(deliveryDetailDto.getDeliveryQuantity());
        productDeliveryBillDetailDto.setDeliveryOutQuantity(BigDecimal.ZERO);
        productDeliveryBillDetailDto.setDeliveryingQuantity(deliveryDetailDto.getDeliveryQuantity());
        productDeliveryBillDetailDto.setAddress(deliveryDetailDto.getDetailedAddress());
        productDeliveryBillDetailDto.setShipmentItemCode(deliveryDetailDto.getDeliveryDetailCode());
        productDeliveryBillDetailDto.setOrderType(deliveryDetailDto.getOrderType());
        productDeliveryBillDetailDto.setOriginalOrderCode(deliveryDetailDto.getOriginalOrderCode());
        //productDeliveryBillDetailDto.setShipmentItemQuantity("发货单行项总数");
        productDeliveryBillDetailDtoList.add(productDeliveryBillDetailDto);
      });
      productDeliveryBillCreateDto.setDetailDtos(productDeliveryBillDetailDtoList);
      log.info("创建发货单{}并创建出库单{}", dto.getDeliveryCode(), JSON.toJSONString(productDeliveryBillCreateDto));
      this.productDeliveryBillVoService.create(productDeliveryBillCreateDto);
    });
  }

  /**
   * 新增校验差异部分
   * @param dto
   */
  private void createValidation(DeliveryDto dto) {
    dto.setId(null);
    Validate.notBlank(dto.getOrderCategory(), "订单类别不能为空");
    if (OrderCategoryEnum.SALES_ORDER.getDictCode().equals(dto.getOrderCategory())) {
      Validate.notBlank(dto.getRelateCode(), "选择销售订单时客户编码不能为空");
      Validate.notBlank(dto.getRelateName(), "选择销售订单时客户名称不能为空");
    }
    Validate.notEmpty(dto.getDeliveryDetailDtoList(), "发货单明细不能为空");
  }
}
