package com.biz.crm.dms.business.delivery.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.dms.business.delivery.sdk.dto.DeliveryDto;
import com.biz.crm.dms.business.delivery.sdk.service.DeliveryVoService;
import com.biz.crm.dms.business.delivery.sdk.vo.DeliveryVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * 发货单管理: DeliveryVo: 发货单
 *
 * @author pengxi
 * @date 2022/02/16
 */
@Slf4j
@RestController
@RequestMapping("/v1/delivery/delivery")
@Api(tags = "发货单: DeliveryVo: 发货单")
public class DeliveryVoController {

  @Autowired(required = false)
  private DeliveryVoService deliveryVoService;

  /**
   * 获取发货单分页列表
   *
   * @param pageable 分页信息
   * @param dto      请求参数dto
   * @return Page<DeliveryVo> 普通发货单分页信息
   */
  @ApiOperation(value = "获取发货单分页列表")
  @GetMapping("/findByConditions")
  public Result<Page<DeliveryVo>> findByConditions(@PageableDefault(50) Pageable pageable, @ApiParam(name = "deliveryDto", value = "分页Dto") DeliveryDto dto) {
    try {
      return Result.ok(this.deliveryVoService.findByConditions(pageable, dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 通过ID获取发货单
   *
   * @param id 发货单ID
   * @return 发货单
   */
  @ApiOperation(value = "通过ID获取发货单")
  @GetMapping("/findById")
  public Result<DeliveryVo> findById(@RequestParam(value = "id") @ApiParam(name = "id", value = "主键ID") String id) {
    try {
      return Result.ok(this.deliveryVoService.findById(id));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 通过订单号获取发货单
   *
   * @param orderCode 订单号
   * @return 发货单
   */
  @ApiOperation(value = "通过订单号获取发货单")
  @GetMapping("/findByOrderCode")
  public Result<List<DeliveryVo>> findByOrderCode(@RequestParam(value = "orderCode") @ApiParam(name = "orderCode", value = "订单号", required = true) String orderCode) {
    try {
      return Result.ok(this.deliveryVoService.findByOrderCode(orderCode));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 创建
   *
   * @param deliveryDto
   * @return
   */
  @ApiOperation(value = "创建")
  @PostMapping(value = "")
  public Result create(@RequestBody DeliveryDto deliveryDto) {
    try {
      this.deliveryVoService.create(deliveryDto);
      return Result.ok();
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 确认发货并创建出库单
   *
   * @param deliveryDto
   * @return
   */
  @ApiOperation(value = "确认发货并创建出库单")
  @PostMapping(value = "createDeliveryAndOutbound")
  public Result createDeliveryAndOutbound(@RequestBody DeliveryDto deliveryDto) {
    try {
      this.deliveryVoService.createDeliveryAndOutbound(deliveryDto);
      return Result.ok();
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 逻辑删除
   *
   * @param ids
   * @return
   */
  @ApiOperation(value = "逻辑删除")
  @DeleteMapping("/delete")
  public Result delete(@RequestParam("ids") List<String> ids) {
    try {
      this.deliveryVoService.updateDelFlagByIds(ids);
      return Result.ok("删除成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 启用（原型设计及需求文档没有该功能）
   *
   * @param ids
   * @return
   */
  @Deprecated
  @ApiOperation(value = "启用（原型设计及需求文档没有该功能）")
  @PatchMapping("/enable")
  public Result enable(@RequestParam("ids") List<String> ids) {
    try {
      this.deliveryVoService.enableBatch(ids);
      return Result.ok("启用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 禁用（作废）
   *
   * @param ids
   * @return
   */
  @ApiOperation(value = "禁用（作废）")
  @PatchMapping("/disable")
  public Result disable(@RequestParam("ids") List<String> ids) {
    try {
      this.deliveryVoService.disableBatch(ids);
      return Result.ok("作废成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 收货
   *
   * @param ids
   * @return
   */
  @ApiOperation(value = "收货")
  @PatchMapping("/handleReceiving")
  public Result handleReceiving(@RequestParam("ids") List<String> ids) {
    try {
      this.deliveryVoService.receivingByIds(ids);
      return Result.ok("收货成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 收货
   * - 因为网关问题未修复好
   * - 为小程序不能发起patch请求
   *
   * @param ids
   * @return 收货结果
   */
  @ApiOperation(value = "收货")
  @PostMapping("/handleReceivingTurnPatch")
  public Result handleReceivingTurnPatch(@RequestParam("ids") List<String> ids) {
    try {
      this.deliveryVoService.receivingByIds(ids);
      return Result.ok("收货成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
