package com.biz.crm.dms.business.delivery.local.notifier;

import com.biz.crm.dms.business.delivery.local.entity.DeliveryDetailDeductEntity;
import com.biz.crm.dms.business.delivery.local.entity.DeliveryDetailEntity;
import com.biz.crm.dms.business.delivery.local.repository.DeliveryDetailDeductRepository;
import com.biz.crm.dms.business.delivery.local.repository.DeliveryDetailRepository;
import com.biz.crm.dms.business.delivery.sdk.service.DeliveryDetailVoService;
import com.biz.crm.dms.business.order.common.sdk.model.TallyItemRegisterModel;
import com.biz.crm.dms.business.order.common.sdk.register.TallyItemRegister;
import com.biz.crm.dms.business.order.sdk.event.OrderCloseListener;
import com.biz.crm.dms.business.order.sdk.vo.OrderDetailPayVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderDetailVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderPayVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 订单关闭监听器impl
 *
 * @author hefan
 * @date 2022/05/13
 */
@Component
public class OrderCloseListenerImpl implements OrderCloseListener {

  @Autowired(required = false)
  private DeliveryDetailVoService deliveryDetailVoService;

  @Autowired(required = false)
  private DeliveryDetailRepository deliveryDetailRepository;

  @Autowired(required = false)
  private DeliveryDetailDeductRepository deliveryDetailDeductRepository;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Qualifier("CapitalTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister capitalTallyItemRegister;

  @Override
  public OrderVo onReturn(OrderVo orderVo) {

    /**
     * - 根据订单编码查询所有发货单
     * -- 发货单，没出库的部分要不要归还
     *
     */
    OrderVo returnOrder = this.nebulaToolkitService.copyObjectByBlankList(orderVo, OrderVo.class, HashSet.class, LinkedList.class);
    // 查询订单在发货段已经发出的数量
    List<DeliveryDetailEntity> detailEntities = this.deliveryDetailRepository.findByOrderCodes(TenantUtils.getTenantCode(), Sets.newHashSet(orderVo.getOrderCode()));
    Map<String, List<DeliveryDetailEntity>> odcToDeliveryDetailMap = detailEntities.stream()
        .collect(
            Collectors.groupingBy(DeliveryDetailEntity::getOrderDetailCode)
        );
    // 查询订单在发货单已经发出的钱
    List<DeliveryDetailDeductEntity> recordDeductEntities = this.deliveryDetailDeductRepository.findByOrderCodes(Sets.newHashSet(orderVo.getOrderCode()));
    Map<String, List<DeliveryDetailDeductEntity>> recordDeductMap = recordDeductEntities.stream()
        .collect(
            Collectors.groupingBy(DeliveryDetailDeductEntity::getOrderDetailCode)
        );
    // 已发货价值，收集容器；
    Map<String, BigDecimal> map = new HashMap<>();
    // 遍历商品行
    List<OrderDetailVo> orderDetails = orderVo.getOrderDetails();
    // 订单退还明细
    List<OrderDetailVo> returnOrderDetails = new ArrayList<>(orderDetails.size());
    // 订单总的已发出钱
    BigDecimal orderTotalDeliverAmount = BigDecimal.ZERO;
    for (OrderDetailVo orderDetail : orderDetails) {
      String orderDetailCode = orderDetail.getOrderDetailCode();
      // 商品未发货数量统计
      // - 属性拷贝 -商品行
      OrderDetailVo returnOrderDetail = this.nebulaToolkitService.copyObjectByBlankList(orderDetail, OrderDetailVo.class, HashSet.class, LinkedList.class);
      // - 商品已发出数量 和 钱
      List<DeliveryDetailEntity> deliveryDetailEntities = odcToDeliveryDetailMap.get(orderDetailCode);
      BigDecimal totalDeliveryQuantity = BigDecimal.ZERO;
      BigDecimal totalDeliverAmount = BigDecimal.ZERO;
      if (!CollectionUtils.isEmpty(deliveryDetailEntities)) {
        for (DeliveryDetailEntity deliveryDetail : deliveryDetailEntities) {
          BigDecimal deliveryQuantity = deliveryDetail.getDeliveryQuantity();
          totalDeliveryQuantity = totalDeliveryQuantity.add(deliveryQuantity);
          BigDecimal deliverAmount = deliveryDetail.getDeliverAmount();
          totalDeliverAmount = totalDeliverAmount.add(deliverAmount);
          orderTotalDeliverAmount = orderTotalDeliverAmount.add(deliverAmount);
        }
      }
      // - 商品未发出数量
      returnOrderDetail.setQuantity(orderDetail.getQuantity().subtract(totalDeliveryQuantity));
      // - 商品未发出钱
      returnOrderDetail.setShouldPaymentAmount(orderDetail.getShouldPaymentAmount().subtract(totalDeliverAmount));
      // 商品未发货价值统计
      List<OrderDetailPayVo> orderDetailPays = orderDetail.getOrderDetailPays();
      List<OrderDetailPayVo> returnOrderDetailPays = new ArrayList<>(orderDetailPays.size());
      for (OrderDetailPayVo orderDetailPay : orderDetailPays) {
        // - 属性拷贝
        OrderDetailPayVo returnOrderDetailPayVo = this.nebulaToolkitService.copyObjectByBlankList(orderDetailPay, OrderDetailPayVo.class, HashSet.class, LinkedList.class);
        // -统计 某商品，某该扣项，已经发货多少钱
        List<DeliveryDetailDeductEntity> recordDeductList = recordDeductMap.get(orderDetailCode);
        BigDecimal totalAmount = Optional.ofNullable(recordDeductList)
            .orElse(Lists.newArrayList())
            .stream()
            .filter(d -> d.getItemKey().equals(orderDetailPay.getItemKey()))
            .map(DeliveryDetailDeductEntity::getItemAmount)
            .reduce(BigDecimal.ZERO, BigDecimal::add);
        // 叠加 某该扣项，已经发出的钱
        BigDecimal temp = map.get(orderDetailPay.getItemKey());
        if (temp != null) {
          temp = temp.add(totalAmount);
        } else {
          temp = totalAmount;
        }
        map.put(orderDetailPay.getItemKey(), temp);
        // - 某商品 某该扣项 还有多少钱没发出
        returnOrderDetailPayVo.setItemAmount(orderDetailPay.getItemAmount().subtract(totalAmount));
        returnOrderDetailPays.add(returnOrderDetailPayVo);
      }
      returnOrderDetail.setOrderDetailPays(returnOrderDetailPays);
      returnOrderDetails.add(returnOrderDetail);
    }
    returnOrder.setOrderDetails(returnOrderDetails);
    // 计算订单剩余价值
    List<OrderPayVo> orderPays = orderVo.getOrderPays();
    // - 计算各支付项已经发出的钱
    // -- 授信与实际支付的比值
    BigDecimal itemAmount = this.getCreditPayAmount(orderPays);
    BigDecimal actualAmountPaid = returnOrder.getActualAmountPaid();
    BigDecimal creditRate = itemAmount.divide(actualAmountPaid, 4, RoundingMode.HALF_UP);
    // -- 授信已经发出的钱
    BigDecimal creditDeliverAmount = orderTotalDeliverAmount.multiply(creditRate);
    // -- 资金已经发出的钱
    BigDecimal capitalDeliverAmount = orderTotalDeliverAmount.subtract(creditDeliverAmount);
    map.put(creditTallyItemRegister.findTallyItemRegisterModel().getItemKey(), creditDeliverAmount);
    map.put(capitalTallyItemRegister.findTallyItemRegisterModel().getItemKey(), capitalDeliverAmount);
    // -计算每个记账项目该退的钱
    returnOrder.setActualAmountPaid(actualAmountPaid.subtract(orderTotalDeliverAmount));
    // -- 订单退还支付
    List<OrderPayVo> returnOrderPays = new ArrayList<>(orderPays.size());
    for (OrderPayVo orderPay : orderPays) {
      OrderPayVo returnOrderPayVo = this.nebulaToolkitService.copyObjectByBlankList(orderPay, OrderPayVo.class, HashSet.class, LinkedList.class);
      String itemKey = orderPay.getItemKey();
      // map里取不到的即有可能是支付项目
      BigDecimal temp = map.get(itemKey);
      if (temp != null) {
        BigDecimal subtract = orderPay.getItemAmount().subtract(temp);
        returnOrderPayVo.setItemAmount(subtract);
      }
      returnOrderPays.add(returnOrderPayVo);
    }
    returnOrder.setOrderPays(returnOrderPays);
    return returnOrder;
  }

  @Qualifier("CreditTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister creditTallyItemRegister;

  private BigDecimal getCreditPayAmount(List<OrderPayVo> orderPays) {
    BigDecimal itemAmount = BigDecimal.ZERO;
    if (CollectionUtils.isEmpty(orderPays)) {
      return itemAmount;
    }
    for (OrderPayVo orderPay : orderPays) {
      String itemKey = orderPay.getItemKey();
      if (StringUtils.isBlank(itemKey)) {
        continue;
      }
      TallyItemRegisterModel tallyItemRegisterModel = creditTallyItemRegister.findTallyItemRegisterModel();
      String itemKey1 = tallyItemRegisterModel.getItemKey();
      if (itemKey.equals(itemKey1)) {
        itemAmount = orderPay.getItemAmount();
        break;
      }
    }
    return itemAmount;
  }
}
