package com.biz.crm.dms.business.delivery.local.repository;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.delivery.local.entity.DeliveryDetailEntity;
import com.biz.crm.dms.business.delivery.local.mapper.DeliveryDetailMapper;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * 发货单明细表的数据库访问类 {@link DeliveryDetailEntity}
 *
 * @author pengxi
 * @date 2021/12/30
 */
@Component
public class DeliveryDetailRepository extends ServiceImpl<DeliveryDetailMapper, DeliveryDetailEntity> {

  /**
   * 根据订单编码集合获取详情集合
   *
   * @param tenantCode 租户编码
   * @param orderCodes 订单编码集合
   * @return List<DeliveryEntity>
   */
  public List<DeliveryDetailEntity> findByOrderCodes(String tenantCode, Set<String> orderCodes) {
    return this.lambdaQuery()
        .eq(DeliveryDetailEntity::getTenantCode, tenantCode)
        .eq(DeliveryDetailEntity::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .in(DeliveryDetailEntity::getOrderCode, orderCodes)
        .list();
  }

  /**
   * 根据发货单编码集合获取详情集合
   *
   * @param tenantCode 租户编码
   * @param deliveryCodes 发货单编码集合
   * @return List<DeliveryEntity>
   */
  public List<DeliveryDetailEntity> findByDeliveryCodes(String tenantCode, Set<String> deliveryCodes) {
    return this.lambdaQuery()
        .eq(DeliveryDetailEntity::getTenantCode, tenantCode)
        .in(DeliveryDetailEntity::getDeliveryCode, deliveryCodes)
        .list();
  }

  /**
   * 根据客户和商品编码集合及时间范围获取发货单明细
   *
   * @param tenantCode 租户编码
   * @param relateCodes 客户编码集合
   * @param goodsCodes 商品/物料编码集合
   * @param startTime 开始时间
   * @param endTime 结束时间
   * @return List<DeliveryEntity>
   */
  public List<DeliveryDetailEntity> findByRelateCodesInAndGoodsCodesInAndBetweenStartTimeAndEndTime(String tenantCode, Set<String> relateCodes, Set<String> goodsCodes, Date startTime, Date endTime) {
    return this.lambdaQuery()
        .eq(DeliveryDetailEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(DeliveryDetailEntity::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(DeliveryDetailEntity::getTenantCode, tenantCode)
        .in(CollectionUtils.isNotEmpty(relateCodes), DeliveryDetailEntity::getRelateCode, relateCodes)
        .in(CollectionUtils.isNotEmpty(goodsCodes), DeliveryDetailEntity::getGoodsCode, goodsCodes)
        .gt(Objects.nonNull(startTime), DeliveryDetailEntity::getCreateTime, startTime)
        .lt(Objects.nonNull(endTime), DeliveryDetailEntity::getCreateTime, endTime)
        .list();
  }

  /**
   * 根据客户和商品编码集合及时间范围获取发货单明细
   *
   * @param tenantCode 租户编码
   * @param deliveryCode 订单编码
   * @param deliveryDetailCode 当前仓库编码
   * @return List<DeliveryEntity>
   */
  public DeliveryDetailEntity findByDeliveryCodeAndDeliveryDetailCode(String tenantCode, String deliveryCode, String deliveryDetailCode) {
    return this.lambdaQuery()
        .eq(DeliveryDetailEntity::getTenantCode, tenantCode)
        .eq(DeliveryDetailEntity::getDeliveryCode, deliveryCode)
        .eq(DeliveryDetailEntity::getDeliveryDetailCode, deliveryDetailCode)
        .one();
  }

  /**
   * 更新出库状态和出库数量通过发货单明细id
   *
   * @param outboundStatus   出库状态
   * @param outboundQuantity 出库数量
   * @param id               订单id
   */
  public void updateOutboundStatusAndOutboundQuantityById(String outboundStatus, BigDecimal outboundQuantity, String id) {
    this.lambdaUpdate()
        .set(DeliveryDetailEntity::getOutboundStatus, outboundStatus)
        .set(DeliveryDetailEntity::getOutboundQuantity, outboundQuantity)
        .eq(DeliveryDetailEntity::getId, id)
        .update();
  }

  /**
   * 根据发货单编码删除
   *
   * @param tenantCode 租户编码
   * @param deliveryCodes 发货单编码集合
   */
  public void deleteByDeliveryCodes(String tenantCode, Set<String> deliveryCodes) {
    this.lambdaUpdate()
        .eq(DeliveryDetailEntity::getTenantCode, tenantCode)
        .in(DeliveryDetailEntity::getDeliveryCode, deliveryCodes)
        .remove();
  }

  /**
   * 收货
   *
   * @param deliveryCode 发货单号
   */
  public void receivingByDeliveryCode(String deliveryCode, BigDecimal receivingQuantity) {
    this.lambdaUpdate().eq(DeliveryDetailEntity::getDeliveryCode, deliveryCode)
        .set(DeliveryDetailEntity::getReceivingQuantity, receivingQuantity)
        .update();
  }

  /**
   * 批量更新启用禁用状态
   *
   * @param tenantCode 租户编码
   * @param deliveryCodes 发货单编码集合
   * @param enableStatus 启用禁用状态
   */
  public void updateEnableStatusByDeliveryCodes(String tenantCode, Set<String> deliveryCodes, EnableStatusEnum enableStatus) {
    this.lambdaUpdate()
        .eq(DeliveryDetailEntity::getTenantCode, tenantCode)
        .in(DeliveryDetailEntity::getDeliveryCode, deliveryCodes)
        .set(DeliveryDetailEntity::getEnableStatus, enableStatus.getCode())
        .update();
  }
}
