package com.biz.crm.dms.business.delivery.local.repository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.delivery.local.entity.DeliveryEntity;
import com.biz.crm.dms.business.delivery.local.mapper.DeliveryMapper;
import com.biz.crm.dms.business.delivery.sdk.enums.ReceiptStatusEnum;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Set;

/**
 * 发货单表的数据库访问类 {@link DeliveryEntity}
 *
 * @author pengxi
 * @date 2021/12/30
 */
@Component
public class DeliveryRepository extends ServiceImpl<DeliveryMapper, DeliveryEntity> {

  /**
   * 根据订单配置编码获取详情
   *
   * @param tenantCode 租户编码
   * @param deliveryCode 发货单编码
   * @return DeliveryEntity
   */
  public DeliveryEntity findByDeliveryCode(String tenantCode, String deliveryCode) {
    LambdaQueryWrapper<DeliveryEntity> wrapper = Wrappers.lambdaQuery();
    wrapper.eq(DeliveryEntity::getTenantCode, tenantCode);
    wrapper.eq(DeliveryEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
    wrapper.eq(DeliveryEntity::getDeliveryCode, deliveryCode);
    wrapper.last("limit 1");
    return this.baseMapper.selectOne(wrapper);
  }

  /**
   * 根据id集合获取详情集合
   *
   * @param deliveryCodes 发货单编码集合
   * @return List<DeliveryEntity>
   */
  public List<DeliveryEntity> findByDeliveryCodes(Set<String> deliveryCodes) {
    return this.lambdaQuery()
        .eq(DeliveryEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .in(DeliveryEntity::getDeliveryCode, deliveryCodes)
        .list();
  }
  
  /**
   * 根据id集合获取详情集合
   *
   * @param ids ID集合
   * @return List<DeliveryEntity>
   */
  public List<DeliveryEntity> findByIds(List<String> ids) {
    return this.lambdaQuery()
        .eq(DeliveryEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .in(DeliveryEntity::getId, ids)
        .list();
  }

  /**
   * 批量更新启用禁用状态
   *
   * @param ids          ID集合
   * @param enableStatus 启用禁用状态
   */
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enableStatus) {
    this.lambdaUpdate()
        .in(DeliveryEntity::getId, ids)
        .set(DeliveryEntity::getEnableStatus, enableStatus.getCode())
        .update();
  }

  /**
   * 逻辑删除
   *
   * @param ids ID集合
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate().in(DeliveryEntity::getId, ids)
        .set(DeliveryEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

  /**
   * 收货
   *
   * @param ids ID集合
   */
  public void receivingByIds(List<String> ids) {
    this.lambdaUpdate().in(DeliveryEntity::getId, ids)
        .set(DeliveryEntity::getReceiptStatus, ReceiptStatusEnum.COMPLETE.getDictCode())
        .update();
  }

}
