package com.biz.crm.dms.business.delivery.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.delivery.local.entity.DeliveryDetailEntity;
import com.biz.crm.dms.business.delivery.local.entity.DeliveryDetailDeductEntity;
import com.biz.crm.dms.business.delivery.local.repository.DeliveryDetailDeductRepository;
import com.biz.crm.dms.business.delivery.local.repository.DeliveryDetailRepository;
import com.biz.crm.dms.business.delivery.sdk.dto.DeliverySalesTargetDto;
import com.biz.crm.dms.business.delivery.sdk.dto.DeliveryUnshippedPageDto;
import com.biz.crm.dms.business.delivery.sdk.enums.OutboundStatusEnum;
import com.biz.crm.dms.business.delivery.sdk.service.DeliveryDetailVoService;
import com.biz.crm.dms.business.delivery.sdk.dto.DeliveryDetailDto;
import com.biz.crm.dms.business.delivery.sdk.service.DeliveryUnshippedVoService;
import com.biz.crm.dms.business.delivery.sdk.vo.DeliveryDetailDeductVo;
import com.biz.crm.dms.business.delivery.sdk.vo.DeliveryDetailVo;
import com.biz.crm.dms.business.delivery.sdk.vo.DeliveryUnshippedVo;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCategoryEnum;
import com.biz.crm.dms.business.order.sdk.service.OrderVoService;
import com.biz.crm.dms.business.order.sdk.vo.OrderDetailPayVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderDetailVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderVo;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.DeliveryStockDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductStockOperationDto;
import com.biz.crm.dms.business.psi.product.sdk.enums.productstock.ProductStockOperationType;
import com.biz.crm.dms.business.psi.product.sdk.enums.productstock.StockType;
import com.biz.crm.dms.business.psi.product.sdk.service.productstock.DeliveryStockVoService;
import com.biz.crm.dms.business.psi.product.sdk.service.productstock.ProductStockVoService;
import com.biz.crm.dms.business.psi.product.sdk.vo.productstock.DeliveryStockVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 发货单Vo服务接口实现类
 *
 * @author pengxi
 * @date 2022/02/16
 */
@Slf4j
@Service
public class DeliveryDetailVoServiceImpl implements DeliveryDetailVoService {

  @Autowired(required = false)
  private DeliveryDetailRepository deliveryDetailRepository;

  @Autowired(required = false)
  private DeliveryDetailDeductRepository deliveryDetailDeductRepository;

  @Autowired(required = false)
  private DeliveryStockVoService deliveryStockVoService;

  @Autowired(required = false)
  private ProductStockVoService productStockVoService;

  @Autowired(required = false)
  private DeliveryUnshippedVoService deliveryUnshippedVoService;

  @Autowired(required = false)
  private OrderVoService orderVoService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public List<DeliveryDetailVo> findByOrderCodes(Set<String> orderCodes) {
    if (CollectionUtils.isEmpty(orderCodes)) {
      return Lists.newArrayList();
    }
    List<DeliveryDetailEntity> detailEntities = this.deliveryDetailRepository.findByOrderCodes(TenantUtils.getTenantCode(), orderCodes);
    if (CollectionUtils.isEmpty(detailEntities)) {
      return Lists.newArrayList();
    }
    List<DeliveryDetailVo> deliveryDetailVos = (List<DeliveryDetailVo>) this.nebulaToolkitService.copyCollectionByWhiteList(detailEntities, DeliveryDetailEntity.class, DeliveryDetailVo.class, HashSet.class, ArrayList.class);
    // 获取发货单行上该扣项目
    this.findDeductByDeliveryDetailVos(deliveryDetailVos);
    return deliveryDetailVos;
  }

  @Override
  public List<DeliveryDetailVo> findByDeliveryCodes(Set<String> deliveryCodes) {
    if (CollectionUtils.isEmpty(deliveryCodes)) {
      return Lists.newArrayList();
    }
    List<DeliveryDetailEntity> detailEntities = this.deliveryDetailRepository.findByDeliveryCodes(TenantUtils.getTenantCode(), deliveryCodes);
    if (CollectionUtils.isEmpty(detailEntities)) {
      return Lists.newArrayList();
    }
    List<DeliveryDetailVo> deliveryDetailVos = (List<DeliveryDetailVo>) this.nebulaToolkitService.copyCollectionByWhiteList(detailEntities, DeliveryDetailEntity.class, DeliveryDetailVo.class, HashSet.class, ArrayList.class);
    // 获取发货单行上该扣项目
    this.findDeductByDeliveryDetailVos(deliveryDetailVos);
    return deliveryDetailVos;
  }

  /**
   * 获取可选的订单及发货单信息
   *
   * 1、根据（页面或传入）条件查询可发货订单信息
   * 2、根据（可发货订单范围内）订单行商品获取对应库存信息
   * 3、根据（可发货订单范围内）订单号查询发货记录，用于计算已发货数量
   * 4、获取可用库存和实际库存
   * 5、计算最大可发货数量
   */
  @Override
  public Page<DeliveryDetailVo> findOrderAndStockDeliveryDetails(Pageable pageable, DeliveryUnshippedPageDto dto) {
    // 1、
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 3000));
    Page<DeliveryUnshippedVo> deliveryUnshippedVoPage = this.deliveryUnshippedVoService.findByOrderUnshippedPageDto(pageable, dto);
    List<DeliveryUnshippedVo> orderVoList = deliveryUnshippedVoPage.getRecords();
    if (CollectionUtils.isEmpty(orderVoList)) {
      return new Page<>();
    }
    List<String> orderCodes = orderVoList.stream().map(DeliveryUnshippedVo::getOrderCode).distinct().collect(Collectors.toList());
    List<OrderVo> orderVos = this.orderVoService.findByOrderCodes(orderCodes);
    if (CollectionUtils.isEmpty(orderVos)) {
      return new Page<>();
    }
    Map<String, OrderVo> orderMap = orderVos.stream().collect(Collectors.toMap(OrderVo::getOrderCode,Function.identity()));

    // 2、注意：分组key为订单号+订单行号
    // （1）占用数量：读取库存模块，该订单、该行商品、在该仓库当前占用数量。
    // （2）剩余可发货数量：占用数量-该订单该行商品该仓库已发货且未出库数量。
    Map<String, List<DeliveryStockVo>> deliveryStockVoMap = this.findDeliveryStockByOrderVos(orderVos);

    // 3、
    List<DeliveryDetailVo> recordDetailVos = this.findByOrderCodes(new HashSet<>(orderCodes));
    Map<String, List<DeliveryDetailVo>> recordDetailMap = recordDetailVos.stream().collect(Collectors.groupingBy(x -> StringUtils.join(x.getOrderCode(), "_", x.getOrderDetailCode())));
    List<DeliveryDetailVo> optionalDetails = Lists.newArrayList();
    orderVoList.forEach(o->{
      OrderVo orderVo = orderMap.get(o.getOrderCode());
      if (orderVo == null) {
        return;
      }
      List<OrderDetailVo> orderDetailVos = orderVo.getOrderDetails();
      if (CollectionUtils.isEmpty(orderDetailVos)) {
        return;
      }
      orderDetailVos.forEach(od->{
        // 4、注意：分组key为订单号+订单行号
        String orderLineKey = StringUtils.join(od.getOrderCode(), "_", od.getOrderDetailCode());
        List<DeliveryStockVo> deliveryStockVoList = deliveryStockVoMap.get(orderLineKey);
        Validate.notEmpty(deliveryStockVoList , String.format("异常数据：订单[%s]没有库存信息！", orderLineKey));
        deliveryStockVoList.forEach(stockVo->{
          // 发货单明细行字段先从订单明细行拷贝，再从订单头获取完善字段
          DeliveryDetailVo deliveryDetailVo = this.nebulaToolkitService.copyObjectByWhiteList(od, DeliveryDetailVo.class, HashSet.class, ArrayList.class);
          deliveryDetailVo.setOriginalOrderCode(o.getOriginalOrderCode());
          deliveryDetailVo.setOrderType(o.getOrderType());
          deliveryDetailVo.setRelateCode(o.getRelateCode());
          deliveryDetailVo.setRelateName(o.getRelateName());
          deliveryDetailVo.setOrgCode(o.getOrgCode());
          deliveryDetailVo.setOrgName(o.getOrgName());
          deliveryDetailVo.setDetailedAddress(o.getDetailedAddress());
          deliveryDetailVo.setCityCode(o.getCityCode());
          deliveryDetailVo.setContactName(o.getContactName());
          deliveryDetailVo.setContactPhone(o.getContactPhone());
          deliveryDetailVo.setPickUpWay(orderVo.getPickUpWay());
          deliveryDetailVo.setHistoryWarehouseCode(stockVo.getWarehouseCode());
          deliveryDetailVo.setHistoryWarehouseName(stockVo.getWarehouseName());
          deliveryDetailVo.setAvailableStock(stockVo.getAvailableStock());
          deliveryDetailVo.setTotalStock(stockVo.getTotalStock());
          deliveryDetailVo.setFrozenQuantity(stockVo.getFrozenQuantity());
          List<DeliveryDetailVo> deliveryDetailRecords = recordDetailMap.get(orderLineKey);
          // 计算已发货未出库数量
          BigDecimal notOutboundQuantity = Optional.ofNullable(deliveryDetailRecords).orElse(Lists.newArrayList()).stream()
              .filter(dd->stockVo.getWarehouseCode().equals(dd.getCurrentWarehouseCode()))
              .reduce(BigDecimal.ZERO, (x, y) -> x.add(y.getDeliveryQuantity().subtract(y.getOutboundQuantity()!=null?y.getOutboundQuantity():BigDecimal.ZERO)), BigDecimal::add);
          // 拿到库存有效占用数
          BigDecimal frozenQuantity = stockVo.getFrozenQuantity() != null ? stockVo.getFrozenQuantity() : BigDecimal.ZERO;
          deliveryDetailVo.setMaxAvailableStock(frozenQuantity.subtract(notOutboundQuantity));
          // 订单行上该扣项目即发货单明细行上该扣项目
          if (CollectionUtils.isNotEmpty(od.getOrderDetailPays())) {
            List<DeliveryDetailDeductVo> orderDetailDeductVos = (List<DeliveryDetailDeductVo>) this.nebulaToolkitService.copyCollectionByWhiteList(od.getOrderDetailPays(), OrderDetailPayVo.class, DeliveryDetailDeductVo.class, HashSet.class, ArrayList.class);
            deliveryDetailVo.setOrderDetailDeductVos(orderDetailDeductVos);
          }
          optionalDetails.add(deliveryDetailVo);
        });
      });
    });
    Page<DeliveryDetailVo> pageResult = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    pageResult.setRecords(optionalDetails);
    pageResult.setTotal(deliveryUnshippedVoPage.getTotal());
    return pageResult;
  }

  @Override
  public Page<DeliveryDetailVo> findOptionalOrderAndDeliveryDetails(Pageable pageable, DeliveryUnshippedPageDto dto) {
    Page<DeliveryDetailVo> deliveryDetailVoPage = this.findOrderAndStockDeliveryDetails(pageable, dto);
    List<DeliveryDetailVo> optionalDetails = Lists.newArrayList();
    if (CollectionUtils.isNotEmpty(deliveryDetailVoPage.getRecords())) {
      deliveryDetailVoPage.getRecords().forEach(dd->{
        if (BigDecimal.ZERO.compareTo(dd.getMaxAvailableStock()) >= 0) {
          // 最大可发货数量小于等于0的不返回
          return;
        }
        optionalDetails.add(dd);
      });
    }
    Page<DeliveryDetailVo> pageResult = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    pageResult.setRecords(optionalDetails);
    pageResult.setTotal(deliveryDetailVoPage.getTotal());
    return pageResult;
  }

  /**
   * 根据发货单获取发货单详情
   *
   * 1、根据发货单号查询发货单明细记录
   * 2、根据发货单明细记录中的订单获取订单信息
   * 3、查询订单明细记录
   * 4、查询库存信息
   * 5、计算发货单明细行上最大可发货数量
   */
  @Override
  public List<DeliveryDetailVo> findDetailByDeliveryCodes(Set<String> deliveryCodes) {
    // 1、
    List<DeliveryDetailVo> detailVos = this.findByDeliveryCodes(deliveryCodes);
    if (CollectionUtils.isEmpty(detailVos)) {
      return Lists.newArrayList();
    }

    // 2、查看订单信息
    List<String> orderCodes = detailVos.stream().map(DeliveryDetailVo::getOrderCode).distinct().collect(Collectors.toList());
    DeliveryUnshippedPageDto optionalOrderDto = new DeliveryUnshippedPageDto();
    optionalOrderDto.setOrderCodes(orderCodes);
    Page<DeliveryDetailVo> optionalDetails = this.findOrderAndStockDeliveryDetails(null, optionalOrderDto);
    // 注意：订单+行号+仓库维度
    Map<String, DeliveryDetailVo> optionalDetailMap = Optional.ofNullable(optionalDetails.getRecords()).orElse(Lists.newArrayList())
        .stream().collect(Collectors.toMap(dd->StringUtils.join(dd.getOrderCode(), "_", dd.getOrderDetailCode(), "_", dd.getHistoryWarehouseCode()), Function.identity()));

    // 遍历填充发货单明细行上订单及库存信息
    detailVos.forEach(dd-> {
      // 根据订单+行号+仓库定位可选订单
      String dimensionKey = StringUtils.join(dd.getOrderCode(), "_", dd.getOrderDetailCode(), "_", dd.getHistoryWarehouseCode());
      DeliveryDetailVo orderDetailVo = optionalDetailMap.get(dimensionKey);
      if (orderDetailVo != null) {
        // 本次已发销售金额
        dd.setDeliverSalesAmount(dd.getDeliveryQuantity().divide(orderDetailVo.getQuantity(), 4, RoundingMode.HALF_UP).multiply(orderDetailVo.getSalesAmount()));
        // 本次已发实际支付金额
        dd.setDeliverShouldPaymentAmount(dd.getDeliveryQuantity().divide(orderDetailVo.getQuantity(), 4, RoundingMode.HALF_UP).multiply(orderDetailVo.getShouldPaymentAmount()));
        dd.setOrderCode(orderDetailVo.getOrderCode());
        dd.setOrderDetailCode(orderDetailVo.getOrderDetailCode());
        dd.setOrderType(orderDetailVo.getOrderType());
        dd.setItemType(orderDetailVo.getItemType());
        dd.setGoodsCode(orderDetailVo.getGoodsCode());
        dd.setGoodsName(orderDetailVo.getGoodsName());
        dd.setSpec(orderDetailVo.getSpec());
        dd.setUnite(orderDetailVo.getUnite());
        dd.setQuantity(orderDetailVo.getQuantity());
        dd.setSalesAmount(orderDetailVo.getSalesAmount());
        dd.setShouldPaymentAmount(orderDetailVo.getShouldPaymentAmount());
        dd.setPresetUnitPrice(orderDetailVo.getPresetUnitPrice());
        dd.setRelateCode(orderDetailVo.getRelateCode());
        dd.setRelateName(orderDetailVo.getRelateName());
        dd.setCityCode(orderDetailVo.getCityCode());
        dd.setOrgCode(orderDetailVo.getOrgCode());
        dd.setOrgName(orderDetailVo.getOrgName());
        dd.setHistoryWarehouseCode(orderDetailVo.getHistoryWarehouseCode());
        dd.setHistoryWarehouseName(orderDetailVo.getHistoryWarehouseName());
        dd.setDetailedAddress(orderDetailVo.getDetailedAddress());
        dd.setContactName(orderDetailVo.getContactName());
        dd.setContactPhone(orderDetailVo.getContactPhone());
        dd.setPickUpWay(orderDetailVo.getPickUpWay());
        dd.setTotalStock(orderDetailVo.getTotalStock());
        dd.setAvailableStock(orderDetailVo.getAvailableStock());
        dd.setMaxAvailableStock(orderDetailVo.getMaxAvailableStock());
        dd.setOrderDetailDeductVos(orderDetailVo.getOrderDetailDeductVos());
        dd.setOutboundQuantity(dd.getOutboundQuantity()==null?BigDecimal.ZERO:dd.getOutboundQuantity());
      }
    });
    return detailVos;
  }

  /**
   * 创建发货单明细逻辑
   * 1、获取可选的订单及发货单信息
   * 2、检查以仓库+商品分组维度下，发货数量是否超出
   * 3、发货单明细校验完毕，构建保存数据
   * 4、计算本次发货单明细行上该扣项目占订单明细行上该扣项目的百分比，进而换算出本次应该保存多少
   *
   * 注意：
   * 订单+行号分组key格式为"订单编码_订单行编码"
   * 仓库+商品分组key格式为"仓库编码_商品编码"
   */
  @Override
  @Transactional
  public void createBatch(List<DeliveryDetailDto> deliveryDetailDtoList) {
    if (CollectionUtils.isEmpty(deliveryDetailDtoList)) {
      return;
    }
    // 1、
    List<String> orderCodes = deliveryDetailDtoList.stream().map(DeliveryDetailDto::getOrderCode).distinct().collect(Collectors.toList());
    DeliveryUnshippedPageDto optionalOrderDto = new DeliveryUnshippedPageDto();
    optionalOrderDto.setOrderCodes(orderCodes);
    Page<DeliveryDetailVo> optionalDetails = this.findOrderAndStockDeliveryDetails(null, optionalOrderDto);
    // 注意：订单+行号+仓库维度才能确定唯一
    Map<String, DeliveryDetailVo> optionalDetailMap = Optional.ofNullable(optionalDetails.getRecords()).orElse(Lists.newArrayList())
        .stream().collect(Collectors.toMap(dd->StringUtils.join(dd.getOrderCode(), "_", dd.getOrderDetailCode(), "_", dd.getHistoryWarehouseCode()), Function.identity()));

    // 以订单号_仓库编码_商品编码分组
    Map<String, DeliveryStockVo> deliveryStockVoMap = this.findDeliveryStockInfoMapByDeliveryDetailDtoList(deliveryDetailDtoList);

    // 根据请求报文订单号获取所有发货记录
    List<DeliveryDetailEntity> recordDetailEntities = this.deliveryDetailRepository.findByOrderCodes(TenantUtils.getTenantCode(), new HashSet<>(orderCodes));

    // 按单号+行号+当前仓库编码分组
    Map<String, List<DeliveryDetailEntity>> recordDetailMap = recordDetailEntities.stream()
        .collect(Collectors.groupingBy(dd -> StringUtils.join(dd.getOrderCode(), "_" + dd.getOrderDetailCode(), "_", dd.getCurrentWarehouseCode())));

    // 根据请求报文订单号获取已发货明细行上该扣项目记录
    List<DeliveryDetailDeductEntity> recordDeductEntities = this.deliveryDetailDeductRepository.findByOrderCodes(orderCodes);
    Map<String, List<DeliveryDetailDeductEntity>> recordDeductMap = recordDeductEntities.stream().collect(Collectors.groupingBy(x ->
            StringUtils.join(x.getOrderCode(), "_", x.getOrderDetailCode())));

    // 2、第一个循环中校验订单+行号+仓库维度的最大可发货数量是否超出
    List<DeliveryDetailEntity> detailEntities = new ArrayList<>(deliveryDetailDtoList.size());
    List<DeliveryDetailDeductEntity> detailDeductEntities = new ArrayList<>();
    deliveryDetailDtoList.forEach(dd->{
      Validate.notNull(dd.getHistoryWarehouseCode() , "历史仓库编码不能为空！");
      Validate.notNull(dd.getHistoryWarehouseName() , "历史仓库名称不能为空！");
      Validate.notNull(dd.getCurrentWarehouseCode() , "当前仓库编码不能为空！");
      Validate.notNull(dd.getCurrentWarehouseName() , "当前仓库名称不能为空！");
      if (dd.getDeliveryQuantity()==null || dd.getDeliveryQuantity().compareTo(BigDecimal.ZERO) == 0) {
        // 如果填写的发货数量为0或者空，视作无效数据跳过
        return;
      }
      String tip = String.format("订单号[%s]%s[%s]%s[%s]", dd.getOrderCode(), dd.getHistoryWarehouseName(), dd.getHistoryWarehouseCode(), dd.getGoodsName(), dd.getGoodsCode());
      Validate.isTrue(dd.getDeliveryQuantity().compareTo(BigDecimal.ZERO) > 0 , String.format("%s发货数量必须大于0！", tip));
      // 单号+行号+仓库
      DeliveryDetailVo optionalDetailVo = optionalDetailMap.get(StringUtils.join(dd.getOrderCode(), "_", dd.getOrderDetailCode(), "_", dd.getHistoryWarehouseCode()));
      Validate.notNull(optionalDetailVo, String.format("%s未找到订单信息，不能创建发货单！", tip));
      BigDecimal maxAvailableStock = optionalDetailVo.getMaxAvailableStock();
      Validate.notNull(maxAvailableStock, String.format("%s最大可发货数量为空，不能创建发货单！", tip));
      Validate.isTrue(dd.getDeliveryQuantity().compareTo(maxAvailableStock) < 1, String.format("%s超过最大可发货数量，不能创建发货单！", tip));
      // 单号+仓库+商品
      DeliveryStockVo deliveryStockVo = deliveryStockVoMap.get(StringUtils.join(dd.getOrderCode(), "_", dd.getHistoryWarehouseCode(), "_", dd.getGoodsCode()));
      Validate.notNull(deliveryStockVo, String.format("%s未找到库存信息，不能创建发货单！", tip));
      // 占用数量：读取库存模块，该订单、该行商品、在该仓库当前占用数量
      BigDecimal frozenStock = deliveryStockVo.getFrozenStock();
      Validate.notNull(frozenStock, String.format("%s库存占用数量为空，不能创建发货单！", tip));
      // 单号+行号+仓库
      List<DeliveryDetailEntity> detailEntityList = recordDetailMap.get(StringUtils.join(dd.getOrderCode(), "_", dd.getOrderDetailCode(), "_", dd.getCurrentWarehouseCode()));
      // 该订单该行商品该仓库已发货且未出库数量
      BigDecimal notDeliveredStock = Optional.ofNullable(detailEntityList).orElse(Lists.newArrayList()).stream()
          .reduce(BigDecimal.ZERO, (x, y) -> x.add(y.getDeliveryQuantity().subtract(y.getOutboundQuantity()!=null?y.getOutboundQuantity():BigDecimal.ZERO)), BigDecimal::add);
      // 剩余可发货数量：占用数量-该订单该行商品该仓库已发货且未出库数量
      BigDecimal surplusAvailableStock = frozenStock.subtract(notDeliveredStock);
      log.info(String.format("%s，冻结数量%s，未出库数量%s，剩余可发货数量%s，发货数量%s！", tip, frozenStock, notDeliveredStock, surplusAvailableStock, dd.getDeliveryQuantity()));
      Validate.isTrue(dd.getDeliveryQuantity().compareTo(surplusAvailableStock) < 1, String.format("%s超过剩余可发货数量%s，不能创建发货单！", tip, surplusAvailableStock));
      //本次已发销售金额：单价*本次发货数量
      DeliveryDetailEntity detailEntity = this.nebulaToolkitService.copyObjectByWhiteList(dd, DeliveryDetailEntity.class, HashSet.class, ArrayList.class);
      detailEntity.setDeliverAmount(optionalDetailVo.getPresetUnitPrice().multiply(detailEntity.getDeliveryQuantity()));
      detailEntities.add(detailEntity);
      if (CollectionUtils.isEmpty(dd.getOrderDetailDeductVos())) {
        return;
      }
      // 本次发货时判断当前订单行是否为最后一次发货
      AtomicReference<Boolean> isLastTime = new AtomicReference<>(Boolean.FALSE);
      if (optionalDetailVo.getMaxAvailableStock().compareTo(dd.getDeliveryQuantity()) == 0) {
        isLastTime.set(Boolean.TRUE);
      }
      // TODO 注意订单明细行上该扣项目vo根据单号+行号+该扣项目key+该扣项目金额数据的由来才能确定唯一
      Map<String, DeliveryDetailDeductVo> optionalDeductMap = Optional.ofNullable(optionalDetailVo.getOrderDetailDeductVos()).orElse(Lists.newArrayList())
          .stream().collect(Collectors.toMap(o-> StringUtils.join(o.getOrderCode(), "_", o.getOrderDetailCode(), "_", o.getItemKey(), "_", o.getOriginData()), Function.identity()));
      // 4、
      dd.getOrderDetailDeductVos().forEach(dto->{
        if (StringUtils.isBlank(dto.getItemKey())) {
          return;
        }
        String key = StringUtils.join(dto.getOrderCode(), "_", dto.getOrderDetailCode(), "_", dto.getItemKey(), "_", dto.getOriginData());
        DeliveryDetailDeductVo optionalDeductVo = optionalDeductMap.get(key);
        Validate.notNull(optionalDeductVo, String.format("当前订单[%s]行号[%s]不能继续发货！", dto.getOrderCode(), dto.getOrderDetailCode()));
        Validate.notNull(optionalDeductVo.getItemAmount(), String.format("当前订单[%s]行号[%s]不能继续发货！", dto.getOrderCode(), dto.getOrderDetailCode()));
        DeliveryDetailDeductEntity detailDeductEntity = new DeliveryDetailDeductEntity();
        detailDeductEntity.setTenantCode(dd.getTenantCode());
        detailDeductEntity.setDeliveryCode(dd.getDeliveryCode());
        detailDeductEntity.setDeliveryDetailCode(dd.getDeliveryDetailCode());
        detailDeductEntity.setOrderCode(dd.getOrderCode());
        detailDeductEntity.setOrderDetailCode(dd.getOrderDetailCode());
        detailDeductEntity.setItemGroupKey(dto.getItemGroupKey());
        detailDeductEntity.setItemKey(dto.getItemKey());
        detailDeductEntity.setItemName(dto.getItemName());
        detailDeductEntity.setItemAmount(dto.getItemAmount());
        detailDeductEntity.setOriginData(dto.getOriginData());
        detailDeductEntity.setOriginDataType(dto.getOriginDataType());
        if (isLastTime.get()) {
          //本次已发（政策优惠、折扣抵扣、货补抵扣、实际支付）金额：2、最后一次发货：订单上对应行的政策优惠金额-所有发货单上该行的已发政策优惠金额。
          List<DeliveryDetailDeductEntity> recordDeductList = recordDeductMap.get(StringUtils.join(dd.getOrderCode(), "_", dd.getOrderDetailCode()));
          String originData = StringUtils.isBlank(dto.getOriginData())?"":dto.getOriginData();
          BigDecimal totalAmount = Optional.ofNullable(recordDeductList).orElse(Lists.newArrayList()).stream().filter(d->d.getItemKey().equals(dto.getItemKey()) && (StringUtils.isBlank(d.getOriginData())?"":d.getOriginData()).equals(StringUtils.isBlank(dto.getOriginData())?"":dto.getOriginData())).map(DeliveryDetailDeductEntity::getItemAmount).reduce(BigDecimal.ZERO,BigDecimal::add);
          detailDeductEntity.setItemAmount(optionalDeductVo.getItemAmount().subtract(totalAmount));
        } else {
          //本次已发（政策优惠、折扣抵扣、货补抵扣、实际支付）金额：1、多次发货的情况下，非最后一次发货：（本次已发数量/订单数量并且四舍五入保留四位小数）*订单上对应行的政策优惠金额
          detailDeductEntity.setItemAmount(dd.getDeliveryQuantity().divide(optionalDetailVo.getQuantity(), 4, RoundingMode.HALF_UP).multiply(optionalDeductVo.getItemAmount()));
        }
        detailDeductEntities.add(detailDeductEntity);
      });
    });

    // 3、第二个循环中校验仓库+商品维度的实际库存量是否足够
    Map<String, List<DeliveryDetailDto>> deliveryDetailDtoMap = deliveryDetailDtoList.stream().collect(Collectors.groupingBy(dd ->
        StringUtils.join(dd.getCurrentWarehouseCode(), "_", dd.getGoodsCode())));
    deliveryDetailDtoMap.forEach((k, detailDtoList)->{
      DeliveryDetailDto deliveryDetailDto = detailDtoList.get(0);
      Validate.notNull(deliveryDetailDto.getCurrentWarehouseCode() , "当前仓库编码不能为空！");
      Validate.notNull(deliveryDetailDto.getCurrentWarehouseCode() , "当前仓库编码不能为空！");
      Validate.notNull(deliveryDetailDto.getCurrentWarehouseName() , "当前仓库名称不能为空！");
      String tip = String.format("%s[%s]%s[%s]", deliveryDetailDto.getCurrentWarehouseName(), deliveryDetailDto.getCurrentWarehouseCode(), deliveryDetailDto.getGoodsName(), deliveryDetailDto.getGoodsCode());
      // 单号+仓库+商品
      String dimensionKey = StringUtils.join(deliveryDetailDto.getOrderCode(), "_", deliveryDetailDto.getCurrentWarehouseCode(), "_", deliveryDetailDto.getGoodsCode());
      DeliveryStockVo deliveryStockVo = deliveryStockVoMap.get(dimensionKey);
      Validate.notNull(deliveryStockVo, String.format("%s未找到库存信息，不能创建发货单！", tip));
      BigDecimal totalStock = deliveryStockVo.getTotalStock();
      BigDecimal availableStock = deliveryStockVo.getAvailableStock();
      Validate.notNull(totalStock, String.format("%s实际库存量为空，不能创建发货单！", tip));
      Validate.notNull(availableStock, String.format("%s可用库存为空，不能创建发货单！", tip));
      BigDecimal itemDtoTotalQuantity = detailDtoList.stream().map(DeliveryDetailDto::getDeliveryQuantity).reduce(BigDecimal.ZERO, BigDecimal::add);
      Validate.isTrue(itemDtoTotalQuantity.compareTo(totalStock) < 1, String.format("%s超过实际库存量，不能创建发货单！", tip));
      Validate.isTrue(itemDtoTotalQuantity.compareTo(availableStock) < 1, String.format("%s超过可用库存，不能创建发货单！", tip));
    });

    this.deliveryDetailRepository.saveBatch(detailEntities);
    if (CollectionUtils.isEmpty(detailDeductEntities)) {
      return;
    }
    this.deliveryDetailDeductRepository.saveBatch(detailDeductEntities);
    // 改仓逻辑
    this.changeWarehouse(deliveryDetailDtoList);
  }

  @Override
  public List<DeliveryDetailVo> findByRelateCodesInAndGoodsCodesInAndBetweenStartTimeAndEndTime(DeliverySalesTargetDto dto) {
    if (StringUtils.isBlank(dto.getTenantCode())) {
      return Lists.newArrayList();
    }
    List<DeliveryDetailEntity> detailEntities = this.deliveryDetailRepository.findByRelateCodesInAndGoodsCodesInAndBetweenStartTimeAndEndTime(dto.getTenantCode(), dto.getCustomerCodes(), dto.getGoodsCodes(), dto.getStartTime(), dto.getEndTime());
    if (CollectionUtils.isEmpty(detailEntities)) {
      return Lists.newArrayList();
    }
    return (List<DeliveryDetailVo>) this.nebulaToolkitService.copyCollectionByWhiteList(detailEntities, DeliveryDetailEntity.class, DeliveryDetailVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public void updateOutboundStatusByOutboundQuantity(String deliveryCode, String deliveryDetailCode, BigDecimal outboundQuantity, Boolean isAdd) {
    Validate.notBlank(deliveryCode, "根据出库数量修改出库状态时，没有传入发货单编码");
    Validate.notBlank(deliveryDetailCode, "根据出库数量修改出库状态时，没有传入发货单行编码");
    Validate.notNull(isAdd, "根据出库数量修改出库状态时，业务标识没传");
    Boolean isPositive = Objects.nonNull(outboundQuantity) && outboundQuantity.compareTo(BigDecimal.ZERO) == 1;
    Validate.isTrue(isPositive, "根据出库数量修改出库状态时，出库数量不能为空、零、负数");
    DeliveryDetailEntity detailEntity = this.deliveryDetailRepository.findByDeliveryCodeAndDeliveryDetailCode(TenantUtils.getTenantCode(), deliveryCode, deliveryDetailCode);
    Validate.notNull(detailEntity, "根据出库数量修改出库状态时，没有查询到发货单信息");
    BigDecimal outboundQuantityDb = detailEntity.getOutboundQuantity();
    outboundQuantityDb = Objects.isNull(outboundQuantityDb) ? BigDecimal.ZERO : outboundQuantityDb;
    if (isAdd) {
      outboundQuantity = outboundQuantityDb.add(outboundQuantity);
    } else {
      outboundQuantity = outboundQuantityDb.subtract(outboundQuantity);
    }
    int compareToZero = outboundQuantity.compareTo(BigDecimal.ZERO);
    Validate.isTrue(compareToZero > -1, "根据出库数量修改出库状态时，已出库数量不能为负");
    if (compareToZero == 0) {
      // 回到待出库状态
      this.deliveryDetailRepository.updateOutboundStatusAndOutboundQuantityById(OutboundStatusEnum.START.getDictCode(), outboundQuantity, detailEntity.getId());
    } else {
      int compareToQuantity = outboundQuantity.compareTo(detailEntity.getDeliveryQuantity());
      Validate.isTrue(compareToQuantity <= 1, "根据出库数量修改出库状态时，已出库数量不能超过发货单商品总量");
      if (compareToQuantity == 0) {
        // 发货单出库完成
        this.deliveryDetailRepository.updateOutboundStatusAndOutboundQuantityById(OutboundStatusEnum.COMPLETE.getDictCode(), outboundQuantity, detailEntity.getId());
      } else {
        // 部分出库
        this.deliveryDetailRepository.updateOutboundStatusAndOutboundQuantityById(OutboundStatusEnum.PARTIAL.getDictCode(), outboundQuantity, detailEntity.getId());
      }
    }
  }

  @Override
  @Transactional
  public void receivingDeliveryCodes(Set<String> deliveryCodes) {
    if (CollectionUtils.isEmpty(deliveryCodes)) {
      return;
    }
    List<DeliveryDetailEntity> detailEntities = this.deliveryDetailRepository.findByDeliveryCodes(TenantUtils.getTenantCode(), deliveryCodes);
    if (CollectionUtils.isEmpty(detailEntities)) {
      return;
    }
    detailEntities.forEach(de-> this.deliveryDetailRepository.receivingByDeliveryCode(de.getDeliveryCode(), de.getDeliveryQuantity()));
  }

  @Override
  @Transactional
  public void updateEnableStatusByDeliveryCodes(Set<String> deliveryCodes, EnableStatusEnum enableStatus) {
    if (CollectionUtils.isEmpty(deliveryCodes)) {
      return;
    }
    this.deliveryDetailRepository.updateEnableStatusByDeliveryCodes(TenantUtils.getTenantCode(), deliveryCodes, enableStatus);
  }

  //////////////////////////////////////////////// 下面是工具方法 ////////////////////////////////////////////////

  /**
   * 获取发货单行上该扣项目
   */
  public void findDeductByDeliveryDetailVos(List<DeliveryDetailVo> deliveryDetailVos) {
    if (CollectionUtils.isEmpty(deliveryDetailVos)) {
      return;
    }
    // 发货单明细行上已扣项目
    List<String> deliveryCodes = deliveryDetailVos.stream().map(DeliveryDetailVo::getDeliveryCode).distinct().collect(Collectors.toList());
    List<DeliveryDetailDeductEntity> detailDeductEntities = this.deliveryDetailDeductRepository.findByDeliveryCodes(deliveryCodes);
    List<DeliveryDetailDeductVo> detailDeductVos = Lists.newArrayList();
    if (CollectionUtils.isNotEmpty(detailDeductEntities)) {
      detailDeductVos = (List<DeliveryDetailDeductVo>) this.nebulaToolkitService.copyCollectionByWhiteList(detailDeductEntities, DeliveryDetailDeductEntity.class, DeliveryDetailDeductVo.class, HashSet.class, ArrayList.class);
    }
    Map<String, List<DeliveryDetailDeductVo>> detailPayMap = Optional.ofNullable(detailDeductVos).orElse(Lists.newArrayList()).stream().collect(Collectors.groupingBy(x -> StringUtils.join(x.getDeliveryCode(), "_", x.getDeliveryDetailCode())));
    // 遍历填充发货单明细行上订单及库存信息
    deliveryDetailVos.forEach(o-> {
      // 发货单明细上的已扣除项
      o.setDeliveryDetailDeductVos(detailPayMap.get(StringUtils.join(o.getDeliveryCode(), "_", o.getDeliveryDetailCode())));
    });
  }

  /**
   * 获取仓库和商品分组的库存信息（包含默认仓库存，这里要看到商品所有仓库库存）
   *
   * 注意：分组key格式为"订单号_仓库编码_商品编码"
   */
  private Map<String, DeliveryStockVo> findDeliveryStockInfoMapByDeliveryDetailDtoList(List<DeliveryDetailDto> deliveryDetailDtoList){
    if (CollectionUtils.isEmpty(deliveryDetailDtoList)) {
      return new HashMap<>(0);
    }
    Map<String, DeliveryStockVo> stockVoMap = new HashMap<>(0);
    List<DeliveryStockDto> deliveryStockDtoList = new ArrayList<>(deliveryDetailDtoList.size());
    String stockType = null;
    DeliveryDetailDto deliveryDetailDto = deliveryDetailDtoList.get(0);
    if (OrderCategoryEnum.SALES_ORDER.getDictCode().equals(deliveryDetailDto.getOrderCategory())) {
      stockType = StockType.PRODUCT.getDictCode();
    } else if (OrderCategoryEnum.MATERIAL_ORDER.getDictCode().equals(deliveryDetailDto.getOrderCategory())) {
      stockType = StockType.MATERIAL.getDictCode();
    }
    String finalStockType = stockType;
    Map<String, List<DeliveryDetailDto>> deliveryDetailDtoMap = deliveryDetailDtoList.stream().collect(Collectors.groupingBy(DeliveryDetailDto::getOrderCode));
    deliveryDetailDtoMap.forEach((orderCode, dd) -> {
      DeliveryStockDto deliveryStockDto = new DeliveryStockDto();
      deliveryStockDto.setStockType(finalStockType);
      deliveryStockDto.setOrderCode(orderCode);
      Set<String> goodsCodes = dd.stream().map(DeliveryDetailDto::getGoodsCode).collect(Collectors.toSet());
      deliveryStockDto.setProductCodes(goodsCodes);
      deliveryStockDtoList.add(deliveryStockDto);
    });
    try {
      List<DeliveryStockVo> result = this.deliveryStockVoService.findByDeliveryMaterialStockDto(deliveryStockDtoList);
      stockVoMap = Optional.ofNullable(result).orElse(new ArrayList<>()).stream().collect(Collectors.toMap(x -> StringUtils.join(x.getOrderCode(), "_", x.getWarehouseCode(), "_", x.getProductCode()), Function.identity()));
    } catch (Exception e) {
      log.error("查询库存时异常：{}", e.getMessage(), e);
    }
    return stockVoMap;
  }

  /**
   * 获取订单+行号对应的的库存信息（只查询订单+订单行占用的仓库库存）
   *
   * 注意：分组key为订单号+订单行号
   */
  private Map<String, List<DeliveryStockVo>> findDeliveryStockByOrderVos(List<OrderVo> orderVos){
    if (CollectionUtils.isEmpty(orderVos)) {
      return new HashMap<>(0);
    }
    List<DeliveryStockDto> deliveryStockDtoList = new ArrayList<>(orderVos.size());
    orderVos.forEach(orderVo -> {
      orderVo.getOrderDetails().forEach(od->{
        DeliveryStockDto deliveryStockDto = new DeliveryStockDto();
        deliveryStockDto.setOrderCode(orderVo.getOrderCode());
        deliveryStockDto.setCityCode(orderVo.getCityCode());
        deliveryStockDto.setOrderItemCode(od.getOrderDetailCode());
        deliveryStockDtoList.add(deliveryStockDto);
      });
    });
    Map<String, List<DeliveryStockVo>> deliveryStockVoMap = new HashMap<>();
    try {
      List<DeliveryStockVo> result = this.deliveryStockVoService.findByOrderCodeAndItemCode(deliveryStockDtoList);
      log.info("查询库存结果：{}", result);
      // 注意：库存明细是流水记录，所以此处需要按订单+行号+仓库维度分组统计冻结总数
      List<DeliveryStockVo> deliveryStockVos = new ArrayList<>();
      if (CollectionUtils.isNotEmpty(result)) {
        Map<String, List<DeliveryStockVo>> handlerDeliveryStockVoMap = result.stream().collect(Collectors.groupingBy(x -> StringUtils.join(x.getOrderCode(), "_", x.getOrderItemCode(), "_", x.getWarehouseCode())));
        handlerDeliveryStockVoMap.forEach((k, ds)->{
          BigDecimal frozenQuantity = ds.stream().map(DeliveryStockVo::getFrozenQuantity).reduce(BigDecimal.ZERO, BigDecimal::add);
          DeliveryStockVo deliveryStockVo = ds.get(0);
          deliveryStockVo.setFrozenQuantity(frozenQuantity);
          deliveryStockVos.add(deliveryStockVo);
        });
      }
      deliveryStockVoMap = deliveryStockVos.stream().collect(Collectors.groupingBy(x -> StringUtils.join(x.getOrderCode(), "_", x.getOrderItemCode())));
      log.info("查询库存分组统计结果：{}", deliveryStockVoMap);
    } catch (Exception e) {
      log.error("查询库存时异常：{}", e.getMessage(), e);
    }
    return deliveryStockVoMap;
  }

  /**
   * 改仓逻辑
   * 1、未改仓：不做任何库存操作
   * 2、改仓
   * （1）解冻原仓库本次发货量
   * （2）并在新仓库新增冻结记录
   */
  private void changeWarehouse(List<DeliveryDetailDto> deliveryDetailDtoList) {
    // 需要冻结部分
    List<ProductStockOperationDto> frozenList = Lists.newArrayListWithCapacity(deliveryDetailDtoList.size());
    // 需要解冻部分
    List<ProductStockOperationDto> thawList = Lists.newArrayListWithCapacity(deliveryDetailDtoList.size());
    for (DeliveryDetailDto deliveryDetailDto : deliveryDetailDtoList) {
      if (StringUtils.equals(deliveryDetailDto.getHistoryWarehouseCode(), deliveryDetailDto.getCurrentWarehouseCode())) {
        // 1、
        log.info("创建发货单时未改仓：不做任何库存操作");
      } else {
        // 2、（1）
        ProductStockOperationDto productStockOperationDto1 = new ProductStockOperationDto();
        productStockOperationDto1.setWarehouseCode(deliveryDetailDto.getHistoryWarehouseCode());
        productStockOperationDto1.setWarehouseName(deliveryDetailDto.getHistoryWarehouseName());
        productStockOperationDto1.setProductCode(deliveryDetailDto.getGoodsCode());
        productStockOperationDto1.setProductName(deliveryDetailDto.getGoodsName());
        productStockOperationDto1.setProductStockOperationType(ProductStockOperationType.ORDER_CHANGE_WAREHOUSE_UNFREEZE.getDictCode());
        productStockOperationDto1.setCustomerCode(deliveryDetailDto.getRelateCode());
        productStockOperationDto1.setCustomerName(deliveryDetailDto.getRelateName());
        productStockOperationDto1.setOriginalOrderCode(deliveryDetailDto.getOriginalOrderCode());
        productStockOperationDto1.setOrderType(deliveryDetailDto.getOrderType());
        productStockOperationDto1.setOrderCode(deliveryDetailDto.getOrderCode());
        productStockOperationDto1.setOrderItemCode(deliveryDetailDto.getOrderDetailCode());
        productStockOperationDto1.setQuantity(deliveryDetailDto.getDeliveryQuantity());
        thawList.add(productStockOperationDto1);
        // 2、（2）
        ProductStockOperationDto productStockOperationDto2 = new ProductStockOperationDto();
        productStockOperationDto2.setWarehouseCode(deliveryDetailDto.getCurrentWarehouseCode());
        productStockOperationDto2.setWarehouseName(deliveryDetailDto.getCurrentWarehouseName());
        productStockOperationDto2.setProductCode(deliveryDetailDto.getGoodsCode());
        productStockOperationDto2.setProductName(deliveryDetailDto.getGoodsName());
        productStockOperationDto2.setProductStockOperationType(ProductStockOperationType.ORDER_CHANGE_WAREHOUSE_FREEZING.getDictCode());
        productStockOperationDto2.setCustomerCode(deliveryDetailDto.getRelateCode());
        productStockOperationDto2.setCustomerName(deliveryDetailDto.getRelateName());
        productStockOperationDto2.setOriginalOrderCode(deliveryDetailDto.getOriginalOrderCode());
        productStockOperationDto2.setOrderType(deliveryDetailDto.getOrderType());
        productStockOperationDto2.setOrderCode(deliveryDetailDto.getOrderCode());
        productStockOperationDto2.setOrderItemCode(deliveryDetailDto.getOrderDetailCode());
        productStockOperationDto2.setQuantity(deliveryDetailDto.getDeliveryQuantity());
        frozenList.add(productStockOperationDto2);
      }
    }
    if (CollectionUtils.isNotEmpty(frozenList)) {
      log.info("新增发货单时执行改仓逻辑的冻结库存：frozenList={}", frozenList);
      this.productStockVoService.frozenBatch(frozenList);
    }
    if (CollectionUtils.isNotEmpty(thawList)) {
      log.info("新增发货单时执行改仓逻辑的解冻库存：thawList={}", thawList);
      this.productStockVoService.thawBatch(thawList);
    }
  }

}
