package com.biz.crm.dms.business.delivery.sdk.service;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.delivery.sdk.dto.DeliveryDetailDto;
import com.biz.crm.dms.business.delivery.sdk.dto.DeliverySalesTargetDto;
import com.biz.crm.dms.business.delivery.sdk.dto.DeliveryUnshippedPageDto;
import com.biz.crm.dms.business.delivery.sdk.vo.DeliveryDetailVo;
import org.springframework.data.domain.Pageable;

import java.math.BigDecimal;
import java.util.List;
import java.util.Set;

/**
 * 发货单Vo服务接口类
 *
 * @author pengxi
 * @date 2022/02/16
 */
public interface DeliveryDetailVoService {

  /**
   * 根据发货单编码获取发货单信息
   *
   * @param orderCodes 订单单编码集合
   * @return DeliveryDetailEntity 发货单明细
   */
  List<DeliveryDetailVo> findByOrderCodes(Set<String> orderCodes);

  /**
   * 根据发货单编码获取发货单信息
   *
   * @param deliveryCodes 发货单编码集合
   * @return DeliveryDetailEntity 发货单明细
   */
  List<DeliveryDetailVo> findByDeliveryCodes(Set<String> deliveryCodes);

  /**
   * 获取订单、库存及发货单信息
   *
   * 1、根据（页面或传入）条件查询可发货订单信息
   * 2、根据（可发货订单范围内）订单行商品获取对应库存信息
   * 3、根据（可发货订单范围内）订单号查询发货记录，用于计算已发货数量
   * 4、获取可用库存和实际库存
   * 5、计算最大可发货数量
   *
   * @param pageable 分页参数
   * @param dto 业务参数
   * @return Page<DeliveryDetailVo>
   */
  Page<DeliveryDetailVo> findOrderAndStockDeliveryDetails(Pageable pageable, DeliveryUnshippedPageDto dto);

  /**
   * 获取可选的订单及发货单信息（最大可发货数量必须大于0）
   *
   * @param pageable 分页参数
   * @param dto 业务参数
   * @return Page<DeliveryDetailVo>
   */
  Page<DeliveryDetailVo> findOptionalOrderAndDeliveryDetails(Pageable pageable, DeliveryUnshippedPageDto dto);

  /**
   * 根据发货单编码获取发货单信息，包含以下更详细信息：
   *
   * 1、查询发货单明细记录
   * 2、查询订单明细记录
   * 3、查询订单明细行上该扣项目
   * 4、查询库存信息
   * 5、计算发货单明细行上最大可发货数量
   *
   * @param deliveryCodes 发货单编码集合
   * @return DeliveryDetailEntity 发货单明细
   */
  List<DeliveryDetailVo> findDetailByDeliveryCodes(Set<String> deliveryCodes);

  /**
   * 创建发货单信息
   *
   * @param deliveryDetailDtoList 发货单信息
   */
  void createBatch(List<DeliveryDetailDto> deliveryDetailDtoList);

  /**
   * 根据客户和商品编码集合及时间范围获取发货单明细
   *
   * @param dto 业务参数
   * @return Page<DeliveryDetailVo>
   */
  List<DeliveryDetailVo> findByRelateCodesInAndGoodsCodesInAndBetweenStartTimeAndEndTime(DeliverySalesTargetDto dto);

  /**
   * 根据发货单编码+发货单行编码维度的出库数量更新出库状态及出库数量
   *
   * @param deliveryCode 发货单编码
   * @param deliveryDetailCode 发货单明细编码
   * @param outboundQuantity 出库数量
   * @param isAdd 业务标识
   */
  void updateOutboundStatusByOutboundQuantity(String deliveryCode, String deliveryDetailCode, BigDecimal outboundQuantity, Boolean isAdd);

  /**
   * 收货数量
   *
   * @param deliveryCodes
   */
  void receivingDeliveryCodes(Set<String> deliveryCodes);

  /**
   * 批量更新启用禁用状态
   *
   * @param deliveryCodes
   * @param enableStatus
   */
  void updateEnableStatusByDeliveryCodes(Set<String> deliveryCodes, EnableStatusEnum enableStatus);
}
