package com.biz.crm.dms.business.exchange.local.service.helper;


import com.alibaba.fastjson.JSONArray;
import com.biz.crm.dms.business.costpool.capital.sdk.dto.CostPoolCapitalDto;
import com.biz.crm.dms.business.costpool.capital.sdk.enums.CapitalAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.capital.sdk.enums.PoolTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditPayDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustOperateEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditPoolTypeEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.dto.CostPoolDiscountDto;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolGroupEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolOperationTypeEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolPayTypeEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolUseTypeEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.dto.CostPoolReplenishmentDto;
import com.biz.crm.dms.business.costpool.sdk.service.CostPoolVoService;
import com.biz.crm.dms.business.exchange.sdk.vo.ExchangeDifferenceVo;
import com.biz.crm.dms.business.exchange.sdk.vo.ExchangeVo;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCategoryEnum;
import com.biz.crm.dms.business.order.sdk.vo.OrderDetailVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderVo;
import com.biz.crm.dms.business.policy.sdk.service.SalePolicyRecordVoService;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductStockOperationDto;
import com.biz.crm.dms.business.psi.product.sdk.enums.productstock.ProductStockOperationType;
import com.biz.crm.dms.business.psi.product.sdk.service.productstock.ProductStockVoService;
import com.biz.crm.mdm.business.warehouse.sdk.service.WarehouseVoService;
import com.biz.crm.mdm.business.warehouse.sdk.vo.WarehouseVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.List;

/**
 * 换货单针对订单业务辅助类
 * - 用与消除冗余代码
 * - 统一部分业务流程
 *
 * 提交审批时，占用按差额为负数的绝对值做占用
 * 审批通过时，释放按差额为负数的绝对值做释放；扣减按差额为负数的绝对值做扣减
 * 审批驳回时，释放按差额为负数的绝对值做释放
 *
 * @author pengxi
 * @date 2022/7/7
 */
@Component
public class OrderHelper {

  @Autowired(required = false)
  private CostPoolVoService costPoolVoService;

  @Autowired(required = false)
  private ProductStockVoService productStockVoService;

  /**
   * 归还库存
   *
   * @param orderVo 订单
   */
  public void returnStock(OrderVo orderVo) {
    WarehouseVo warehouse = this.findWarehouseVo(orderVo);
    Validate.notNull(warehouse, "换货单占用库存的时候，没有找到仓库信息");
    List<ProductStockOperationDto> list = this.getProductStockOperationDtos(orderVo, warehouse);
    this.productStockVoService.thawBatch(list);
  }

  @Autowired(required = false)
  private SalePolicyRecordVoService salePolicyRecordVoService;

  /**
   * 归还 促销政策的额度
   *
   * @param orderVo 订单
   */
  public void releaseSalePolicy(OrderVo orderVo) {
    this.salePolicyRecordVoService.deleteByBillCode(orderVo.getOrderCode(), orderVo.getTenantCode() , orderVo.getRelateCode());
  }

  /**
   * 支付换货单
   *
   * @param exchangeVo 换货单
   */
  public void paymentExchange(ExchangeVo exchangeVo) {
    /**
     * 先释放
     * 后扣减
     */
    this.releaseExchange(exchangeVo);
    this.useExchange(exchangeVo);
  }

  /**
   * 根据换货中差额为负数金额绝对值的扣减
   *
   * @param exchangeVo 换货单
   */
  public void useExchange(ExchangeVo exchangeVo) {
    List<ExchangeDifferenceVo> differenceVoList = exchangeVo.getDifferenceList();
    if (CollectionUtils.isEmpty(differenceVoList)) {
      return;
    }
    JSONArray array = new JSONArray();
    differenceVoList.forEach(diff->{
      if (BigDecimal.ZERO.compareTo(diff.getExchangeAmount())<1) {
        // 差额大于等于0，直接跳过，为负才需要执行
        return;
      }
      if (CreditPoolTypeEnum.CREDIT.getDictCode().equals(diff.getPoolType())) {
        CreditPayDto creditPayDto = getCreditPayDto(exchangeVo, diff.getExchangeAmount().abs(), CashAdjustOperateEnum.ORDER, CashAdjustTypeEnum.CREDIT_REDUCE);
        array.add(creditPayDto);
      }
      if (com.biz.crm.dms.business.costpool.capital.sdk.enums.PoolTypeEnum.CAPITAL.getDictCode().equals(diff.getPoolType())) {
        CostPoolCapitalDto costPoolCapitalDto = getCostPoolCapitalDto(exchangeVo, diff.getExchangeAmount().abs(), CapitalAdjustTypeEnum.ORDER_USE);
        array.add(costPoolCapitalDto);
      }
      if (com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolTypeEnum.DISCOUNT.getDictCode().equals(diff.getPoolType())) {
        CostPoolDiscountDto discountDto = getCostPoolDiscountDto(exchangeVo, diff.getExchangeAmount().abs(), PoolOperationTypeEnum.ORDER_USE);
        array.add(discountDto);
      }
      if (com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolTypeEnum.Replenishment.getDictCode().equals(diff.getPoolType())) {
        CostPoolReplenishmentDto replenishmentDto = getCostPoolReplenishmentDto(exchangeVo, diff.getSourceCode(), diff.getExchangeAmount().abs(), com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolOperationTypeEnum.ORDER_USE);
        array.add(replenishmentDto);
      }
    });
    if(CollectionUtils.isEmpty(array)) {
      return;
    }
    this.costPoolVoService.handleAdjust(array);
  }

  /**
   * 根据换货中差额为负数金额绝对值的释放
   *
   * @param exchangeVo
   */
  public void releaseExchange(ExchangeVo exchangeVo) {
    List<ExchangeDifferenceVo> differenceVoList = exchangeVo.getDifferenceList();
    if (CollectionUtils.isEmpty(differenceVoList)) {
      return;
    }
    JSONArray array = new JSONArray();
    differenceVoList.forEach(diff->{
      if (BigDecimal.ZERO.compareTo(diff.getExchangeAmount())<1) {
        // 差额大于等于0，直接跳过，为负才需要执行
        return;
      }
      if (CreditPoolTypeEnum.CREDIT.getDictCode().equals(diff.getPoolType())) {
        CreditPayDto creditPayDto = getCreditPayDto(exchangeVo, diff.getExchangeAmount().abs(), CashAdjustOperateEnum.OCCUPY_RELEASE, CashAdjustTypeEnum.OCCUPY_RELEASE);
        array.add(creditPayDto);
      }
      if (com.biz.crm.dms.business.costpool.capital.sdk.enums.PoolTypeEnum.CAPITAL.getDictCode().equals(diff.getPoolType())) {
        CostPoolCapitalDto costPoolCapitalDto = getCostPoolCapitalDto(exchangeVo, diff.getExchangeAmount().abs(), CapitalAdjustTypeEnum.OCCUPY_RELEASE);
        array.add(costPoolCapitalDto);
      }
      if (com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolTypeEnum.DISCOUNT.getDictCode().equals(diff.getPoolType())) {
        CostPoolDiscountDto discountDto = getCostPoolDiscountDto(exchangeVo, diff.getExchangeAmount().abs(), PoolOperationTypeEnum.OCCUPY_RELEASE);
        array.add(discountDto);
      }
      if (com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolTypeEnum.Replenishment.getDictCode().equals(diff.getPoolType())) {
        CostPoolReplenishmentDto replenishmentDto = getCostPoolReplenishmentDto(exchangeVo, diff.getSourceCode(), diff.getExchangeAmount().abs(), com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolOperationTypeEnum.OCCUPY_RELEASE);
        array.add(replenishmentDto);
      }
    });
    if(CollectionUtils.isEmpty(array)) {
      return;
    }
    this.costPoolVoService.handleAdjust(array);
  }

  /**
   * 根据换货中差额为负数金额绝对值的占用
   *
   * @param exchangeVo
   */
  public void occupyExchange(ExchangeVo exchangeVo) {
    List<ExchangeDifferenceVo> differenceVoList = exchangeVo.getDifferenceList();
    if (CollectionUtils.isEmpty(differenceVoList)) {
      return;
    }
    JSONArray array = new JSONArray();
    differenceVoList.forEach(diff->{
      if (BigDecimal.ZERO.compareTo(diff.getExchangeAmount())<1) {
        // 差额大于等于0，直接跳过，为负才需要执行
        return;
      }
      if (CreditPoolTypeEnum.CREDIT.getDictCode().equals(diff.getPoolType())) {
        CreditPayDto creditPayDto = getCreditPayDto(exchangeVo, diff.getExchangeAmount().abs(), CashAdjustOperateEnum.OCCUPY_USE, CashAdjustTypeEnum.OCCUPY_USE);
        array.add(creditPayDto);
      }
      if (com.biz.crm.dms.business.costpool.capital.sdk.enums.PoolTypeEnum.CAPITAL.getDictCode().equals(diff.getPoolType())) {
        CostPoolCapitalDto costPoolCapitalDto = getCostPoolCapitalDto(exchangeVo, diff.getExchangeAmount().abs(), CapitalAdjustTypeEnum.OCCUPY_USE);
        array.add(costPoolCapitalDto);
      }
      if (com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolTypeEnum.DISCOUNT.getDictCode().equals(diff.getPoolType())) {
        CostPoolDiscountDto discountDto = getCostPoolDiscountDto(exchangeVo, diff.getExchangeAmount().abs(), PoolOperationTypeEnum.OCCUPY_USE);
        array.add(discountDto);
      }
      if (com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolTypeEnum.Replenishment.getDictCode().equals(diff.getPoolType())) {
        CostPoolReplenishmentDto replenishmentDto = getCostPoolReplenishmentDto(exchangeVo, diff.getSourceCode(), diff.getExchangeAmount().abs(), com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolOperationTypeEnum.OCCUPY_USE);
        array.add(replenishmentDto);
      }
    });
    if(CollectionUtils.isEmpty(array)) {
      return;
    }
    this.costPoolVoService.handleAdjust(array);
  }

  /**
   * 占用库存
   *
   * @param orderVo
   */
  public void occupyStock(OrderVo orderVo) {
    WarehouseVo warehouse = this.findWarehouseVo(orderVo);
    Validate.notNull(warehouse, "订单占用库存的时候，没有找到仓库信息");
    List<ProductStockOperationDto> list = getProductStockOperationDtos(orderVo, warehouse);
    this.productStockVoService.frozenBatch(list);
  }

  @Autowired(required = false)
  private WarehouseVoService warehouseVoService;

  /**
   * 查询仓库
   *
   * @param orderVo 订单
   * @return {@link WarehouseVo}
   */
  private WarehouseVo findWarehouseVo(OrderVo orderVo) {
    WarehouseVo warehouse = null;
    if (orderVo.getOrderCategory().equals(OrderCategoryEnum.MATERIAL_ORDER.getDictCode())) {
      String warehouseCode = orderVo.getWarehouseCode();
      warehouse = warehouseVoService.findDetailsByCode(warehouseCode);
    } else {
      warehouse = warehouseVoService.findDetailsByCityCode(orderVo.getCityCode());
      if (warehouse == null) {
        warehouse = warehouseVoService.findDetailsByWarehouseDefault(true);
      }
    }
    return warehouse;
  }

  /**
   * 得到库存操作dto
   *
   * @param order     订单
   * @param warehouse 仓库
   * @return {@link List}<{@link ProductStockOperationDto}>
   */
  private List<ProductStockOperationDto> getProductStockOperationDtos(OrderVo order, WarehouseVo warehouse) {
    List<OrderDetailVo> orderDetails = order.getOrderDetails();
    Validate.isTrue(!CollectionUtils.isEmpty(orderDetails), "订单占用库存的时候，没有货物信息");
    List<ProductStockOperationDto> list = Lists.newArrayListWithCapacity(orderDetails.size());
    for (OrderDetailVo orderDetail : orderDetails) {
      ProductStockOperationDto productStockOperationDto = new ProductStockOperationDto();
      productStockOperationDto.setWarehouseCode(warehouse.getWarehouseCode());
      productStockOperationDto.setWarehouseName(warehouse.getWarehouseName());
      productStockOperationDto.setProductCode(orderDetail.getGoodsCode());
      productStockOperationDto.setProductName(orderDetail.getGoodsName());
      productStockOperationDto.setProductStockOperationType(ProductStockOperationType.SALE_DELIVER.getDictCode());
      productStockOperationDto.setCustomerCode(order.getRelateCode());
      productStockOperationDto.setCustomerName(order.getRelateName());
      productStockOperationDto.setOriginalOrderCode(order.getOriginalOrderCode());
      productStockOperationDto.setOrderType(order.getOrderType());
      productStockOperationDto.setOrderCode(order.getOrderCode());
      productStockOperationDto.setOrderItemCode(orderDetail.getOrderDetailCode());
      productStockOperationDto.setQuantity(orderDetail.getQuantity());
      list.add(productStockOperationDto);
    }
    return list;
  }

  /**
   * 资金dto
   *
   * @param exchangeVo            换货单
   * @param itemAmount            项目数量
   * @param capitalAdjustTypeEnum 资本调整枚举类型
   * @return {@link CostPoolCapitalDto}
   */
  private static CostPoolCapitalDto getCostPoolCapitalDto(ExchangeVo exchangeVo, BigDecimal itemAmount, CapitalAdjustTypeEnum capitalAdjustTypeEnum) {
    CostPoolCapitalDto dto = new CostPoolCapitalDto();
    dto.setPoolType(PoolTypeEnum.CAPITAL.getKey());
    dto.setCustomerCode(exchangeVo.getCustomerCode());
    dto.setFromCode(exchangeVo.getExchangeCode());
    dto.setFromDesc(capitalAdjustTypeEnum.getValue());
    dto.setAmount(itemAmount);
    dto.setAdjustMoney(itemAmount);
    dto.setAdjustType(capitalAdjustTypeEnum.getKey());
    dto.setAdjustTypeName(capitalAdjustTypeEnum.getValue());
    dto.setBillNo(exchangeVo.getExchangeCode());
    dto.setBillType(exchangeVo.getExchangeType());
    return dto;
  }

  /**
   * 得到授信支付dto
   *
   * @param exchangeVo 换货单
   * @param itemAmount 项目数量
   * @return {@link CreditPayDto}
   */
  private static CreditPayDto getCreditPayDto(ExchangeVo exchangeVo, BigDecimal itemAmount, CashAdjustOperateEnum cashAdjustOperateEnum, CashAdjustTypeEnum cashAdjustTypeEnum) {
    CreditPayDto dto = new CreditPayDto();
    dto.setPoolType(CreditPoolTypeEnum.CREDIT.getKey());
    dto.setCustomerCode(exchangeVo.getCustomerCode());
    dto.setFromCode(exchangeVo.getExchangeCode());
    dto.setFromDesc(cashAdjustOperateEnum.getValue());
    dto.setAmount(itemAmount);
    dto.setAdjustTypeCode(cashAdjustTypeEnum.getDictCode());
    dto.setAdjustOperateCode(cashAdjustOperateEnum.getDictCode());
    return dto;
  }

  /**
   * 得到折扣DTO
   *
   * @param exchangeVo 换货单
   * @param itemAmount 项目数量
   * @return {@link CostPoolDiscountDto}
   */
  private static CostPoolDiscountDto getCostPoolDiscountDto(ExchangeVo exchangeVo, BigDecimal itemAmount, PoolOperationTypeEnum poolOperationTypeEnum) {
    CostPoolDiscountDto dto = new CostPoolDiscountDto();
    dto.setPoolType(com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolTypeEnum.DISCOUNT.getKey());
    dto.setCustomerCode(exchangeVo.getCustomerCode());
    dto.setCustomerName(exchangeVo.getCustomerName());
    dto.setFromCode(exchangeVo.getExchangeCode());
    dto.setFromDesc(poolOperationTypeEnum.getValue());
    dto.setAmount(itemAmount);
    dto.setPoolGroup(PoolGroupEnum.DEFAULT.getDictCode());
    dto.setPayType(PoolPayTypeEnum.DISCOUNT.getDictCode());
    dto.setUseType(PoolUseTypeEnum.DEFAULT.getDictCode());
    dto.setOperationType(poolOperationTypeEnum.getDictCode());
    return dto;
  }

  /**
   * 得到货补 DTO
   *
   * @param exchangeVo 换货单
   * @param poolCode 货补池编码
   * @param salesAmount 销售金额
   * @return {@link CostPoolReplenishmentDto}
   */
  private static CostPoolReplenishmentDto getCostPoolReplenishmentDto(ExchangeVo exchangeVo, String poolCode, BigDecimal salesAmount, com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolOperationTypeEnum poolOperationTypeEnum) {
    CostPoolReplenishmentDto dto = new CostPoolReplenishmentDto();
    dto.setPoolType(com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolTypeEnum.Replenishment.getKey());
    dto.setCustomerCode(exchangeVo.getCustomerCode());
    dto.setCustomerName(exchangeVo.getCustomerName());
    dto.setFromCode(exchangeVo.getExchangeCode());
    dto.setFromDesc(poolOperationTypeEnum.getValue());
    dto.setAmount(salesAmount);
    dto.setPoolCode(poolCode);
    dto.setPoolGroup(com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolGroupEnum.DEFAULT.getDictCode());
    dto.setPayType(com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolPayTypeEnum.Replenishment.getDictCode());
    dto.setUseType(com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolUseTypeEnum.DEFAULT.getDictCode());
    dto.setOperationType(poolOperationTypeEnum.getDictCode());
    return dto;
  }
}
