package com.biz.crm.dms.business.exchange.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.exchange.local.entity.ExchangeDifferenceEntity;
import com.biz.crm.dms.business.exchange.local.repository.ExchangeDifferenceRepository;
import com.biz.crm.dms.business.exchange.sdk.dto.ExchangeDifferenceDto;
import com.biz.crm.dms.business.exchange.sdk.enums.OperationGuideEnum;
import com.biz.crm.dms.business.exchange.sdk.service.ExchangeDifferenceVoService;
import com.biz.crm.dms.business.exchange.sdk.vo.ExchangeDifferenceVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;

/**
 * 换货单附件子表(DmsOrderExchangeDifference)表服务实现类
 *
 * @author xi.peng
 * @since 2022-07-06 19:00:34
 */
@Slf4j
@Service
public class ExchangeDifferenceVoServiceImpl implements ExchangeDifferenceVoService {

  @Autowired(required = false)
  private ExchangeDifferenceRepository exchangeDifferenceRepository;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public List<ExchangeDifferenceVo> findByExchangeCode(String exchangeCodes) {
    if (StringUtils.isBlank(exchangeCodes)) {
      return null;
    }
    List<ExchangeDifferenceEntity> entities = this.exchangeDifferenceRepository.findByExchangeCode(exchangeCodes);
    if (CollectionUtils.isEmpty(entities)) {
      return null;
    }
    return (List<ExchangeDifferenceVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, ExchangeDifferenceEntity.class, ExchangeDifferenceVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public void createBatch(String exchangeCode, List<ExchangeDifferenceDto> dtoList) {
    if (CollectionUtils.isEmpty(dtoList)) {
      return;
    }
    //删除原来的重新创建
    this.exchangeDifferenceRepository.removeByExchangeCodes(Collections.singletonList(exchangeCode));
    dtoList.forEach(dto->{
      dto.setTenantCode(TenantUtils.getTenantCode());
      dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      dto.setExchangeCode(exchangeCode);
      this.createValidation(dto);
    });
    List<ExchangeDifferenceEntity> entities = (List<ExchangeDifferenceEntity>) this.nebulaToolkitService.copyCollectionByWhiteList(dtoList, ExchangeDifferenceDto.class, ExchangeDifferenceEntity.class, HashSet.class, ArrayList.class);
    this.exchangeDifferenceRepository.saveBatch(entities);
  }

  @Override
  @Transactional
  public void updateBatch(List<ExchangeDifferenceDto> dtoList) {
    Validate.notEmpty(dtoList, "换货单差额集合不能为空！");
    List<ExchangeDifferenceEntity> differenceEntities = new ArrayList<>();
    dtoList.forEach(dto->{
      this.updateValidation(dto);
      ExchangeDifferenceEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, ExchangeDifferenceEntity.class, HashSet.class, ArrayList.class);
      differenceEntities.add(entity);
    });
    // 先删除
    List<String> exchangeCodes = dtoList.stream().map(ExchangeDifferenceDto::getExchangeCode).distinct().collect(Collectors.toList());
    this.exchangeDifferenceRepository.removeByExchangeCodes(exchangeCodes);
    // 再创建
    this.exchangeDifferenceRepository.saveBatch(differenceEntities);
  }

  private void createValidation(ExchangeDifferenceDto dto) {
    this.validation(dto);
  }

  private void updateValidation(ExchangeDifferenceDto dto) {
    this.validation(dto);
    Validate.notBlank(dto.getId(), "ID不能为空");
  }

  private void validation(ExchangeDifferenceDto dto) {
    Validate.notNull(dto, "换货单差额对象不能为空");
    Validate.notNull(dto.getExchangeCode(), "换货单编码不能为空");
    Validate.notNull(dto.getExchangeAmount(), "差额不能为空");
    Validate.notNull(dto.getPoolType(), "池子类型不能为空");
    //Validate.notNull(dto.getSourceCode(), "来源编码不能为空");
    if (BigDecimal.ZERO.compareTo(dto.getExchangeAmount()) > 0) {
      //dto.getExchangeAmount() 负数
      dto.setPoolGuide(OperationGuideEnum.USE.getDictCode());
    } else if (BigDecimal.ZERO.compareTo(dto.getExchangeAmount()) < 0) {
      //dto.getExchangeAmount() 正数
      dto.setPoolGuide(OperationGuideEnum.ON.getDictCode());
    }
  }
}
