package com.biz.crm.dms.business.exchange.local.service.internal;

import com.biz.crm.dms.business.exchange.sdk.dto.ExchangeFileDto;
import com.biz.crm.dms.business.exchange.sdk.enums.ExchangeStatusEnum;
import com.biz.crm.dms.business.exchange.sdk.service.ExchangeFlowService;
import com.biz.crm.dms.business.exchange.local.service.helper.OrderHelper;
import com.biz.crm.dms.business.exchange.sdk.constant.ExchangeConstant;
import com.biz.crm.dms.business.exchange.sdk.dto.ExchangeDto;
import com.biz.crm.dms.business.exchange.sdk.service.ExchangeVoService;
import com.biz.crm.dms.business.exchange.sdk.vo.ExchangeVo;
import com.biz.crm.dms.business.order.sdk.service.OrderVoService;
import com.biz.crm.dms.business.order.sdk.vo.OrderVo;
import com.biz.crm.workflow.sdk.constant.WorkFlowGlobals;
import com.biz.crm.workflow.sdk.dto.StartProcessDto;
import com.biz.crm.workflow.sdk.listener.ProcessListener;
import com.biz.crm.workflow.sdk.vo.AttachmentVo;
import com.biz.crm.workflow.sdk.vo.response.CommitWorkflowResponse;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

/**
 * 换货单主表实体(DmsOrderExchange)表服务实现类
 *
 * @author xi.peng
 * @since 2022-07-06 19:00:33
 */
@Slf4j
@Service
public class ExchangeFlowServiceImpl implements ExchangeFlowService {

  @Autowired(required = false)
  private ExchangeVoService exchangeVoService;

  @Autowired(required = false)
  private OrderHelper orderHelper;

  @Autowired(required = false)
  private OrderVoService orderVoService;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Transactional
  @Override
  public void submit(ExchangeDto dto) {
    /**
     * - 创建退货单&订单
     * - 查询出订单信息
     * - 应当时处于某种状态的订单才可以去支付
     * - 支付费用
     */
    ExchangeVo exchangeVo = this.exchangeVoService.createOrUpdate(dto);
    Validate.notBlank(dto.getProduceOrderId(), "提交换货单时，订单未创建成功");
    OrderVo orderVo = this.orderVoService.findById(dto.getProduceOrderId());
    Validate.notNull(orderVo, "提交换货单时，订单未创建成功");
    this.orderHelper.occupyExchange(exchangeVo);
    //订单提交接口已调库存占用，所以这里不再重复调用 orderConfirmService.handleConfirmWithoutResource
    //this.orderExchangeHelper.occupyStock(orderVo);
    this.commitProcess(dto);
  }

  /**
   * 编辑时校验项
   */
  private void commitValidation(ExchangeDto dto) {
    Validate.notBlank(dto.getId(), "换货单ID不能为空");
    Validate.notBlank(dto.getProcessKey(), "审批流程编码不能为空");
    Validate.notBlank(dto.getExchangeCode(), "换货单编码不能为空");
    Validate.notBlank(dto.getCompetenceCode(), "菜单编码不能为空");
  }

  /**
   * 提交流程
   */
  public void commitProcess(ExchangeDto dto) {
    this.commitValidation(dto);
    StartProcessDto start = new StartProcessDto();
    start.setBusinessId(dto.getId());
    start.setProcessKey(dto.getProcessKey());
    start.setBusinessNo(dto.getExchangeCode());
    start.setRemark(dto.getProcessRemark());
    start.setMenuCode(dto.getCompetenceCode());
    start.setFormType(ExchangeConstant.EXCHANGE_FORM_TYPE);
    start.setProcessTitle(StringUtils.join("换货单审批", "：", dto.getExchangeCode()));
    if (CollectionUtils.isNotEmpty(dto.getFileList())) {
      List<AttachmentVo> attachmentVos = (List<AttachmentVo>) this.nebulaToolkitService.copyCollectionByWhiteList(dto.getFileList(), ExchangeFileDto.class, AttachmentVo.class, HashSet.class, ArrayList.class);
      start.setAttachmentVos(attachmentVos);
    }
    Map<String, Object> map = Maps.newHashMap();
    map.put(WorkFlowGlobals.COMMIT_CUS,dto.getCustomerCode());
    start.setVariables(map);
    SerializableBiConsumer<ProcessListener, StartProcessDto> sf = ProcessListener::onStartProcess;
    CommitWorkflowResponse response = (CommitWorkflowResponse) nebulaNetEventClient.directPublish(start, ProcessListener.class, sf);
    Validate.notNull(response, "在发起订单工作流时，没有发现事件监听实现");
    if (StringUtils.isNotEmpty(response.getProcessInstanceId())) {
      this.exchangeVoService.updateOrderStatusAndProcessNumberById(ExchangeStatusEnum.AWAIT_APPROVE.getDictCode(), response.getProcessInstanceId(), dto.getId());
    } else {
      throw new RuntimeException("发起流程失败！");
    }
  }

}
