package com.biz.crm.dms.business.exchange.local.strategy;


import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditPoolTypeEnum;
import com.biz.crm.dms.business.exchange.local.service.helper.ReturnGoodsHelper;
import com.biz.crm.dms.business.exchange.sdk.dto.ExchangeDto;
import com.biz.crm.dms.business.exchange.sdk.strategy.ExchangeVerificationStrategy;
import com.biz.crm.dms.business.order.sdk.service.OrderVoService;
import com.biz.crm.dms.business.order.sdk.vo.OrderPayVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

/**
 * 授信金额校验
 * c. 需校验退还的授信不可超过最大可退授信。也即最大可退授信=原订单使用授信-本订单关联的
 * （审批中+待入库+部分入库+完全入库）的退货单退还的授信金额。其余池子无需校验必须低于使用金额。
 * @author pengxi
 * @date 2022/7/12
 */
@Slf4j
@Component
public class CreditAmountVerificationStrategyImpl implements ExchangeVerificationStrategy {

  @Autowired(required = false)
  private ReturnGoodsHelper returnGoodsHelper;

  @Autowired(required = false)
  private OrderVoService orderVoService;

  @Override
  public void execute(ExchangeDto exchangeDto) {
    Validate.notNull(exchangeDto.getOrder(), "订单信息不能为空！");
    Validate.notNull(exchangeDto.getReturnedGoods(), "退货信息不能为空！");
    List<OrderVo> orderVoList = this.orderVoService.findByOrderCodes(Collections.singletonList(exchangeDto.getOriginalOrderCode()));
    Validate.notEmpty(orderVoList, "原订单不存在或已删除！");
    List<OrderPayVo> orderPays = orderVoList.get(0).getOrderPays();
    Validate.notEmpty(orderPays, "原订单总的记账项目为空！");
    Optional<OrderPayVo> optionalOrderPayVo = orderPays.stream()
        .filter(o-> CreditPoolTypeEnum.CREDIT.getDictCode().equals(o.getItemKey()))
        .findFirst();
    if (!optionalOrderPayVo.isPresent()) {
      return;
    }
    // 订单使用授信金额
    BigDecimal orderUseCreditAmount = optionalOrderPayVo.get().getItemAmount();
    // 最大可退授信金额
    BigDecimal totalCreditAmount = this.returnGoodsHelper.getReturnedAmount(exchangeDto.getOriginalOrderCode(), exchangeDto.getProduceReturnCode());
    BigDecimal balance = orderUseCreditAmount.subtract(totalCreditAmount);
    Validate.isTrue(exchangeDto.getReturnedGoods().getCreditAmount().compareTo(balance) <= 0, "授信金额超出，当前退还的授信金额为：" + exchangeDto.getReturnedGoods().getCreditAmount()+",最大可退授信金额为:"+balance);
  }

  @Override
  public int order() {
    return 3;
  }
}
