package com.biz.crm.dms.business.exchange.local.service.internal;

import com.biz.crm.dms.business.exchange.local.service.helper.OrderHelper;
import com.biz.crm.dms.business.exchange.sdk.constant.ExchangeConstant;
import com.biz.crm.dms.business.exchange.sdk.dto.ExchangeDto;
import com.biz.crm.dms.business.exchange.sdk.enums.ExchangeStatusEnum;
import com.biz.crm.dms.business.exchange.sdk.service.ExchangeFlowService;
import com.biz.crm.dms.business.exchange.sdk.service.ExchangeVoService;
import com.biz.crm.dms.business.exchange.sdk.vo.ExchangeVo;
import com.biz.crm.dms.business.order.sdk.service.OrderVoService;
import com.biz.crm.dms.business.order.sdk.vo.OrderVo;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 换货单主表实体(DmsOrderExchange)表服务实现类
 *
 * @author xi.peng
 * @since 2022-07-06 19:00:33
 */
@Slf4j
@Service
public class ExchangeFlowServiceImpl implements ExchangeFlowService {

  @Autowired(required = false)
  private ExchangeVoService exchangeVoService;

  @Autowired(required = false)
  private OrderHelper orderHelper;

  @Autowired(required = false)
  private OrderVoService orderVoService;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Transactional
  @Override
  public void submit(ExchangeDto dto) {
    /**
     * - 创建退货单&订单
     * - 查询出订单信息
     * - 应当时处于某种状态的订单才可以去支付
     * - 支付费用
     */
    ExchangeVo exchangeVo = this.exchangeVoService.createOrUpdate(dto);
    Validate.notBlank(dto.getProduceOrderId(), "提交换货单时，订单未创建成功");
    OrderVo orderVo = this.orderVoService.findById(dto.getProduceOrderId());
    Validate.notNull(orderVo, "提交换货单时，订单未创建成功");
    this.orderHelper.occupyExchange(exchangeVo);
    //订单提交接口已调库存占用，所以这里不再重复调用 orderConfirmService.handleConfirmWithoutResource
    //this.orderExchangeHelper.occupyStock(orderVo);
    this.commitProcess(dto);
  }

  /**
   * 编辑时校验项
   */
  private void commitValidation(ExchangeDto dto) {
    Validate.notBlank(dto.getId(), "换货单ID不能为空");
    Validate.notBlank(dto.getProcessBusiness().getProcessKey(), "审批流程编码不能为空");
    Validate.notBlank(dto.getExchangeCode(), "换货单编码不能为空");
  }

  @Autowired(required = false)
  private ProcessBusinessService processBusinessService;

  /**
   * 提交流程
   */
  public void commitProcess(ExchangeDto dto) {
    this.commitValidation(dto);
    ProcessBusinessDto processBusiness = dto.getProcessBusiness();
    processBusiness.setProcessTitle(StringUtils.join("换货单审批", "：", dto.getExchangeCode()));
    processBusiness.setBusinessNo(dto.getExchangeCode());
    processBusiness.setBusinessFormJson(JsonUtils.obj2JsonString(dto));
    processBusiness.setBusinessCode(ExchangeConstant.EXCHANGE_FORM_TYPE);
    ProcessBusinessVo processBusinessVo = processBusinessService.processStart(processBusiness);
    this.exchangeVoService.updateOrderStatusAndProcessNumberById(ExchangeStatusEnum.AWAIT_APPROVE.getDictCode(), processBusinessVo.getProcessNo(), dto.getId());
  }

}
