package com.biz.crm.dms.business.exchange.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditPoolTypeEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.vo.CostPoolReplenishmentOrderProductVo;
import com.biz.crm.dms.business.costpool.replenishment.sdk.vo.CostPoolReplenishmentVo;
import com.biz.crm.dms.business.costpool.sdk.dto.CostPoolDto;
import com.biz.crm.dms.business.costpool.sdk.service.CostPoolVoService;
import com.biz.crm.dms.business.costpool.sdk.vo.CostPoolVo;
import com.biz.crm.dms.business.exchange.local.entity.ExchangeEntity;
import com.biz.crm.dms.business.exchange.local.repository.ExchangeRepository;
import com.biz.crm.dms.business.exchange.sdk.constant.ExchangeConstant;
import com.biz.crm.dms.business.exchange.sdk.dto.ExchangeDifferenceDto;
import com.biz.crm.dms.business.exchange.sdk.dto.ExchangeDto;
import com.biz.crm.dms.business.exchange.sdk.dto.ExchangePageDto;
import com.biz.crm.dms.business.exchange.sdk.enums.ExchangeStatusEnum;
import com.biz.crm.dms.business.exchange.sdk.enums.OperationGuideEnum;
import com.biz.crm.dms.business.exchange.sdk.event.ExchangeEventListener;
import com.biz.crm.dms.business.exchange.sdk.service.ExchangeDifferenceVoService;
import com.biz.crm.dms.business.exchange.sdk.service.ExchangeFileVoService;
import com.biz.crm.dms.business.exchange.sdk.service.ExchangeVoService;
import com.biz.crm.dms.business.exchange.sdk.strategy.ExchangeVerificationStrategy;
import com.biz.crm.dms.business.exchange.sdk.vo.ExchangeDifferenceVo;
import com.biz.crm.dms.business.exchange.sdk.vo.ExchangeFileVo;
import com.biz.crm.dms.business.exchange.sdk.vo.ExchangeVo;
import com.biz.crm.dms.business.order.common.sdk.enums.ItemTypeEnum;
import com.biz.crm.dms.business.order.sdk.dto.OrderConfirmDto;
import com.biz.crm.dms.business.order.sdk.dto.OrderDetailPreviewDto;
import com.biz.crm.dms.business.order.sdk.dto.OrderPayConfirmDto;
import com.biz.crm.dms.business.order.sdk.service.OrderConfirmService;
import com.biz.crm.dms.business.order.sdk.service.OrderVoService;
import com.biz.crm.dms.business.order.sdk.vo.OrderConfirmVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderDetailPreviewVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderVo;
import com.biz.crm.dms.business.returned.goods.sdk.dto.goods.ReturnedGoodsDto;
import com.biz.crm.dms.business.returned.goods.sdk.dto.goods.ReturnedGoodsReplenishmentDto;
import com.biz.crm.dms.business.returned.goods.sdk.enums.goods.PayTypeEnum;
import com.biz.crm.dms.business.returned.goods.sdk.enums.goods.ReturnedGoodsStatusEnum;
import com.biz.crm.dms.business.returned.goods.sdk.service.goods.ReturnedGoodsVoService;
import com.biz.crm.dms.business.returned.goods.sdk.vo.goods.ReturnedGoodsVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;

/**
 * 换货单主表实体(DmsOrderExchange)表服务实现类
 *
 * @author xi.peng
 * @since 2022-07-06 19:00:33
 */
@Slf4j
@Service
public class ExchangeVoServiceImpl implements ExchangeVoService {

  @Autowired(required = false)
  private ExchangeRepository exchangeRepository;

  @Autowired(required = false)
  private ExchangeDifferenceVoService exchangeDifferenceVoService;

  @Autowired(required = false)
  private ExchangeFileVoService exchangeFileVoService;

  @Autowired(required = false)
  private List<ExchangeEventListener> exchangeEventListeners;

  @Autowired(required = false)
  private List<ExchangeVerificationStrategy> exchangeVerificationStrategies;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private ReturnedGoodsVoService returnedGoodsVoService;

  @Autowired(required = false)
  private OrderConfirmService orderConfirmService;

  @Autowired(required = false)
  private CostPoolVoService costPoolVoService;

  @Autowired(required = false)
  private OrderVoService orderVoService;

  @Override
  public Page<ExchangeVo> findByConditions(Pageable pageable, ExchangePageDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new ExchangePageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<ExchangePageDto> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.exchangeRepository.findByConditions(page, dto);
  }

  @Override
  public ExchangeVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    ExchangeEntity entity = this.exchangeRepository.findById(id);
    if (entity == null) {
      return null;
    }
    ExchangeVo exchangeVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, ExchangeVo.class, HashSet.class, ArrayList.class);
    //退货信息
    ReturnedGoodsVo returnedGoodsVo = this.returnedGoodsVoService.findDetailById(exchangeVo.getProduceReturnId());
    exchangeVo.setReturnedGoods(returnedGoodsVo);
    //订单信息
    OrderVo orderVo = this.orderVoService.findById(exchangeVo.getProduceOrderId());
    exchangeVo.setOrder(orderVo);
    //差额信息
    List<ExchangeDifferenceVo> differenceVoList = this.exchangeDifferenceVoService.findByExchangeCode(entity.getExchangeCode());
    exchangeVo.setDifferenceList(differenceVoList);
    //附件信息
    List<ExchangeFileVo> fileVoList = this.exchangeFileVoService.findByExchangeCode(entity.getExchangeCode());
    exchangeVo.setFileList(fileVoList);
    return exchangeVo;
  }

  /**
   * 汇总：退货数量，换货数量，退货金额，换货金额，差额（换货减退货）
   */
  private void fieldSummary(ExchangeDto dto, ReturnedGoodsVo returnedGoodsVo, OrderConfirmVo orderVo){
    dto.setReturnQuantity(returnedGoodsVo.getQuantity());
    dto.setReturnAmount(returnedGoodsVo.getTotalAmount());
    if(orderVo != null) {
      BigDecimal totalQuantity = orderVo.getOrderDetails().stream().map(OrderDetailPreviewVo::getQuantity).reduce(BigDecimal.ZERO, BigDecimal::add);
      dto.setExchangeQuantity(totalQuantity);
      dto.setExchangeAmount(orderVo.getTotalOrderAmount());
      dto.setDifferenceAmount(dto.getExchangeAmount().subtract(dto.getReturnAmount()));
    }
  }

  @Override
  @Transactional
  public void draft(ExchangeDto dto) {
    // 参数校验
    this.createValidation(dto);
    Validate.isTrue(dto.getDraft(), "此接口为暂存新增或编辑，请确认是否接口调用是否正确");
    dto.setExchangeStatus(ExchangeStatusEnum.STAGING.getDictCode());

    // 创建或更新暂存状态的退货单
    dto.getReturnedGoods().setReturnedGoodsStatus(ReturnedGoodsStatusEnum.STAGING.getDictCode());
    dto.getReturnedGoods().setIsShow(BooleanEnum.FALSE.getCapital());
    dto.getReturnedGoods().setExternalSources(ExchangeConstant.EXCHANGE_FORM_TYPE);
    dto.getReturnedGoods().setPayType(PayTypeEnum.BALANCE.getDictCode());
    ReturnedGoodsVo returnedGoodsVo = null;
    if (StringUtils.isNotBlank(dto.getReturnedGoods().getId())) {
      returnedGoodsVo = this.returnedGoodsVoService.updateByExchange(dto.getReturnedGoods());
    } else {
      returnedGoodsVo = this.returnedGoodsVoService.createByExchange(dto.getReturnedGoods());
    }
    Validate.notNull(returnedGoodsVo, "退货单创建失败！");
    Validate.notNull(returnedGoodsVo.getId(), "退货单创建后退货单ID获取异常！");
    Validate.notNull(returnedGoodsVo.getReturnedGoodsCode(), "退货单创建后退货单编码获取异常！");
    dto.setProduceReturnId(returnedGoodsVo.getId());
    dto.setProduceReturnCode(returnedGoodsVo.getReturnedGoodsCode());

    // 创建或更新订单，暂存逻辑可能页面还没填订单信息
    OrderConfirmVo orderVo = null;
    if (dto.getOrder() != null) {
      dto.getOrder().setIsShow(Boolean.FALSE);
      dto.getOrder().setExternalSources(ExchangeConstant.EXCHANGE_FORM_TYPE);
      orderVo = this.orderConfirmService.createOrUpdate(dto.getOrder());
      Validate.notNull(orderVo, "订单创建失败！");
      Validate.notNull(orderVo.getId(), "订单创建后订单ID获取异常！");
      Validate.notNull(orderVo.getOrderCode(), "订单创建后订单编码获取异常！");
      dto.setProduceOrderId(orderVo.getId());
      dto.setProduceOrderCode(orderVo.getOrderCode());
    }

    // 换货单编码
    if (StringUtils.isBlank(dto.getExchangeCode())) {
      // redis生成发货单编码，编码规则为FHD+年月日+5位顺序数。每天都从00001开始编
      String ruleCode = StringUtils.join(ExchangeConstant.EXCHANGE_CODE, DateFormatUtils.format(new Date(), "yyyyMMdd"));
      List<String> codeList = this.generateCodeService.generateCode(ruleCode, 1, 5, 2, TimeUnit.DAYS);
      Validate.isTrue(CollectionUtils.isNotEmpty(codeList), "添加信息时，生成换货单编码失败！");
      dto.setExchangeCode(codeList.get(0));
    }

    // 汇总：退货数量，换货数量，退货金额，换货金额，差额（换货减退货）
    this.fieldSummary(dto, returnedGoodsVo, orderVo);

    // 创建换货单
    ExchangeEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, ExchangeEntity.class, HashSet.class, ArrayList.class);
    this.exchangeRepository.saveOrUpdate(entity);

    // 创建换货单差额表
    this.exchangeDifferenceVoService.createBatch(entity.getExchangeCode(), dto.getDifferenceList());
    // 创建换货单附件
    this.exchangeFileVoService.createBatch(entity.getExchangeCode(), dto.getFileList());

    // 暂存逻辑不需要发送通知
  }


  /**
   * 构建除货补外的各池子差额信息
   */
  private void buildDifferenceList(List<ExchangeDifferenceVo> differenceVoList, ExchangeDto dto, String poolType, BigDecimal diffAmount){

    if (BigDecimal.ZERO.compareTo(diffAmount) == 0) {
      //如果当前差额为0，表示没用到，直接跳过
      return;
    }

    //按客户编码和费用池类型获取费用池
    CostPoolDto costPoolDto = new CostPoolDto();
    costPoolDto.setCustomerCode(dto.getCustomerCode());
    costPoolDto.setPoolType(poolType);
    List<CostPoolVo> costPoolVoList = this.costPoolVoService.handleRequestCostPoolVos(costPoolDto);
    String poolCode = null;
    if (CollectionUtils.isNotEmpty(costPoolVoList)) {
      CostPoolVo costPoolVo = costPoolVoList.get(0);
      poolCode = costPoolVo.getPoolCode();
    }
    //授信
    if (CreditPoolTypeEnum.CREDIT.getDictCode().equals(poolType)) {
      ExchangeDifferenceVo credit = new ExchangeDifferenceVo();
      credit.setExchangeCode(dto.getExchangeCode());
      credit.setPoolType(CreditPoolTypeEnum.CREDIT.getDictCode());
      credit.setPoolGuide(BigDecimal.ZERO.compareTo(diffAmount)<1?OperationGuideEnum.ON.getValue():OperationGuideEnum.USE.getValue());
      credit.setExchangeAmount(diffAmount);
      credit.setSourceCode(poolCode);
      differenceVoList.add(credit);
    }
    //资金
    if (com.biz.crm.dms.business.costpool.capital.sdk.enums.PoolTypeEnum.CAPITAL.getDictCode().equals(poolType)) {
      ExchangeDifferenceVo capital = new ExchangeDifferenceVo();
      capital.setExchangeCode(dto.getExchangeCode());
      capital.setPoolType(com.biz.crm.dms.business.costpool.capital.sdk.enums.PoolTypeEnum.CAPITAL.getDictCode());
      capital.setPoolGuide(BigDecimal.ZERO.compareTo(diffAmount)<1?OperationGuideEnum.ON.getValue():OperationGuideEnum.USE.getValue());
      capital.setSourceCode(poolCode);
      capital.setExchangeAmount(diffAmount);
      differenceVoList.add(capital);
    }
    //折扣
    if (com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolTypeEnum.DISCOUNT.getDictCode().equals(poolType)) {
      ExchangeDifferenceVo discount = new ExchangeDifferenceVo();
      discount.setExchangeCode(dto.getExchangeCode());
      discount.setPoolType(com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolTypeEnum.DISCOUNT.getDictCode());
      discount.setPoolGuide(BigDecimal.ZERO.compareTo(diffAmount)<1?OperationGuideEnum.ON.getValue():OperationGuideEnum.USE.getValue());
      discount.setSourceCode(poolCode);
      discount.setExchangeAmount(diffAmount);
      differenceVoList.add(discount);
    }
  }

  @Override
  public ExchangeVo difference(ExchangeDto dto) {

    this.submitValidate(dto);

    //退货相关金额
    ReturnedGoodsDto returnedGoodsDto = dto.getReturnedGoods();
    BigDecimal creditAmountReturned = Objects.isNull(returnedGoodsDto.getCreditAmount())?BigDecimal.ZERO:returnedGoodsDto.getCreditAmount();
    BigDecimal capitalAmountReturned = Objects.isNull(returnedGoodsDto.getCapitalAmount())?BigDecimal.ZERO:returnedGoodsDto.getCapitalAmount();
    BigDecimal discountAmountReturned = Objects.isNull(returnedGoodsDto.getDiscountAmount())?BigDecimal.ZERO:returnedGoodsDto.getDiscountAmount();

    //订单相关金额
    List<OrderPayConfirmDto> orderConfirmPays = dto.getOrder().getOrderPays();
    Validate.notNull(orderConfirmPays, "订单总的记账项目不能为空");
    Map<String, OrderPayConfirmDto> orderPayConfirmDtoMap = orderConfirmPays.stream().collect(Collectors.toMap(OrderPayConfirmDto::getItemKey, Function.identity()));
    BigDecimal creditAmountOrder = Objects.isNull(orderPayConfirmDtoMap.get(CreditPoolTypeEnum.CREDIT.getDictCode()))?BigDecimal.ZERO:orderPayConfirmDtoMap.get(CreditPoolTypeEnum.CREDIT.getDictCode()).getItemAmount();
    BigDecimal capitalAmountOrder = Objects.isNull(orderPayConfirmDtoMap.get(com.biz.crm.dms.business.costpool.capital.sdk.enums.PoolTypeEnum.CAPITAL.getDictCode()))?BigDecimal.ZERO:orderPayConfirmDtoMap.get(com.biz.crm.dms.business.costpool.capital.sdk.enums.PoolTypeEnum.CAPITAL.getDictCode()).getItemAmount();
    BigDecimal discountAmountOrder = Objects.isNull(orderPayConfirmDtoMap.get(com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolTypeEnum.DISCOUNT.getDictCode()))?BigDecimal.ZERO:orderPayConfirmDtoMap.get(com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolTypeEnum.DISCOUNT.getDictCode()).getItemAmount();

    //除货补之外其他池子的差额（保留4位小数，向上取整）
    BigDecimal creditAmount = creditAmountReturned.subtract(creditAmountOrder).setScale(4, BigDecimal.ROUND_UP);
    BigDecimal capitalAmount = capitalAmountReturned.subtract(capitalAmountOrder).setScale(4, BigDecimal.ROUND_UP);
    BigDecimal discountAmount = discountAmountReturned.subtract(discountAmountOrder).setScale(4, BigDecimal.ROUND_UP);

    // 构建差额列表
    List<ExchangeDifferenceVo> differenceVoList = new ArrayList<>();
    this.buildDifferenceList(differenceVoList, dto, CreditPoolTypeEnum.CREDIT.getDictCode(), creditAmount);
    this.buildDifferenceList(differenceVoList, dto, com.biz.crm.dms.business.costpool.capital.sdk.enums.PoolTypeEnum.CAPITAL.getDictCode(), capitalAmount);
    this.buildDifferenceList(differenceVoList, dto, com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolTypeEnum.DISCOUNT.getDictCode(), discountAmount);

    //下面是对货补差额的构建

    //货补费用池
    CostPoolDto costPoolDto3 = new CostPoolDto();
    costPoolDto3.setCustomerCode(dto.getCustomerCode());
    costPoolDto3.setPoolType(com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolTypeEnum.Replenishment.getDictCode());
    List<CostPoolVo> costPoolReplenishmentVoList = this.costPoolVoService.handleRequestCostPoolVos(costPoolDto3);
    Map<String, CostPoolVo> costPoolReplenishmentMap = Optional.ofNullable(costPoolReplenishmentVoList).orElse(new ArrayList<>()).stream().collect(Collectors.toMap(CostPoolVo::getPoolCode,Function.identity()));

    //退货中的货补信息
    List<ReturnedGoodsReplenishmentDto> replenishmentList = Optional.ofNullable(returnedGoodsDto.getReplenishmentList()).orElse(new ArrayList<>());
    //退货中的货补信息按货补池编码分组
    Map<String, List<ReturnedGoodsReplenishmentDto>> returnedGoodsReplenishmentMap = replenishmentList.stream()
        .collect(Collectors.groupingBy(ReturnedGoodsReplenishmentDto::getReplenishmentCode));
    Set<String> returnedGoodsReplenishmentCodes = replenishmentList.stream()
        .map(ReturnedGoodsReplenishmentDto::getReplenishmentCode)
        .collect(Collectors.toSet());

    //订单中的货补信息
    List<OrderDetailPreviewDto> orderDetails = Optional.ofNullable(dto.getOrder().getOrderDetails()).orElse(new ArrayList<>());
    //订单中的货补信息按货补池编码分组
    Map<String, List<OrderDetailPreviewDto>> orderReplenishmentMap = orderDetails.stream()
        .filter(o-> ItemTypeEnum.COMPENSATED_GOODS.getDictCode().equals(o.getItemType()))
        .collect(Collectors.groupingBy(OrderDetailPreviewDto::getItemCode));
    Set<String> orderReplenishmentCodes = orderDetails.stream()
        .filter(o->ItemTypeEnum.COMPENSATED_GOODS.getDictCode().equals(o.getItemType()))
        .map(OrderDetailPreviewDto::getItemCode)
        .collect(Collectors.toSet());

    //退货及订单所涉及到的货补池编码合并起来
    returnedGoodsReplenishmentCodes.addAll(orderReplenishmentCodes);
    returnedGoodsReplenishmentCodes.forEach(replenishmentCode->{
      List<ReturnedGoodsReplenishmentDto> returnedGoodsReplenishmentDtoList = returnedGoodsReplenishmentMap.get(replenishmentCode);
      BigDecimal replenishmentAmountReturned = Optional.ofNullable(returnedGoodsReplenishmentDtoList).orElse(new ArrayList<>()).stream().map(ReturnedGoodsReplenishmentDto::getAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
      List<OrderDetailPreviewDto> orderDetailPreviewDtoList = orderReplenishmentMap.get(replenishmentCode);
      BigDecimal replenishmentAmountOrder = Optional.ofNullable(orderDetailPreviewDtoList).orElse(new ArrayList<>()).stream().map(OrderDetailPreviewDto::getSalesAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
      //退货减订单，得到当前货补池的差额（保留4位小数，向上取整）
      BigDecimal replenishmentAmount = replenishmentAmountReturned.subtract(replenishmentAmountOrder).setScale(4, BigDecimal.ROUND_UP);
      if (BigDecimal.ZERO.compareTo(replenishmentAmount) == 0) {
        //如果当前货补池差额为0，表示没用到，直接跳过
        return;
      }
      CostPoolReplenishmentVo costPoolReplenishmentVo = (CostPoolReplenishmentVo) costPoolReplenishmentMap.get(replenishmentCode);
      ExchangeDifferenceVo replenishment = new ExchangeDifferenceVo();
      replenishment.setExchangeCode(dto.getExchangeCode());
      replenishment.setPoolType(com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolTypeEnum.Replenishment.getDictCode());
      replenishment.setPoolGuide(BigDecimal.ZERO.compareTo(replenishmentAmount)<1?OperationGuideEnum.ON.getValue():OperationGuideEnum.USE.getValue());
      replenishment.setExchangeAmount(replenishmentAmount);
      if (costPoolReplenishmentVo != null) {
        replenishment.setSourceCode(costPoolReplenishmentVo.getPoolCode());
        List<CostPoolReplenishmentOrderProductVo> productVos = costPoolReplenishmentVo.getCostPoolReplenishmentProduct();
        if (CollectionUtils.isNotEmpty(productVos)) {
          replenishment.setSourceBind(StringUtils.join(productVos.stream().map(CostPoolReplenishmentOrderProductVo::getGoodsProductName).collect(Collectors.toList()), "|"));
        } else {
          replenishment.setSourceBind(costPoolReplenishmentVo.getGoodsProductLevelName());
        }
      }
      differenceVoList.add(replenishment);
    });

    ExchangeVo exchangeVo = this.nebulaToolkitService.copyObjectByBlankList(dto, ExchangeVo.class, HashSet.class, LinkedList.class);
    exchangeVo.setDifferenceList(differenceVoList);
    return exchangeVo;
  }

  @Override
  @Transactional
  public ExchangeVo createOrUpdate(ExchangeDto dto) {
    // 参数校验
    this.createValidation(dto);
    //不是暂存，需要对参数进行严格校验
    this.submitValidate(dto);
    Validate.notNull(dto.getDifferenceList(), "换货单差额集合信息不能为空");
    Validate.isTrue(!dto.getDraft(), "此接口为提交审批不是暂存信息，请确认是否接口调用是否正确");
    dto.setExchangeStatus(ExchangeStatusEnum.AWAIT_APPROVE.getDictCode());

    // 创建或更新，生成审批中状态的退货单
    dto.getReturnedGoods().setReturnedGoodsStatus(ReturnedGoodsStatusEnum.UNDER_REVIEW.getDictCode());
    dto.getReturnedGoods().setIsShow(BooleanEnum.FALSE.getCapital());
    dto.getReturnedGoods().setExternalSources(ExchangeConstant.EXCHANGE_FORM_TYPE);
    ReturnedGoodsVo returnedGoodsVo = null;
    if (StringUtils.isNotBlank(dto.getReturnedGoods().getId())) {
      returnedGoodsVo = this.returnedGoodsVoService.updateByExchange(dto.getReturnedGoods());
    } else {
      returnedGoodsVo = this.returnedGoodsVoService.createByExchange(dto.getReturnedGoods());
    }
    Validate.notNull(returnedGoodsVo, "退货单创建失败！");
    Validate.notNull(returnedGoodsVo.getId(), "退货单创建后退货单ID获取异常！");
    Validate.notNull(returnedGoodsVo.getReturnedGoodsCode(), "退货单创建后退货单编码获取异常！");
    dto.setProduceReturnId(returnedGoodsVo.getId());
    dto.setProduceReturnCode(returnedGoodsVo.getReturnedGoodsCode());

    // 创建或更新订单，非暂存逻辑页面必须填完订单信息
    dto.getOrder().setIsShow(Boolean.FALSE);
    dto.getOrder().setExternalSources(ExchangeConstant.EXCHANGE_FORM_TYPE);
    OrderConfirmVo orderVo = this.orderConfirmService.handleConfirmWithoutResource(dto.getOrder());
    Validate.notNull(orderVo, "订单创建失败！");
    Validate.notNull(orderVo.getId(), "订单创建后订单ID获取异常！");
    Validate.notNull(orderVo.getOrderCode(), "订单创建后订单编码获取异常！");
    dto.setProduceOrderId(orderVo.getId());
    dto.setProduceOrderCode(orderVo.getOrderCode());

    // 换货单编码
    if (StringUtils.isBlank(dto.getExchangeCode())) {
      // redis生成发货单编码，编码规则为FHD+年月日+5位顺序数。每天都从00001开始编
      String ruleCode = StringUtils.join(ExchangeConstant.EXCHANGE_CODE, DateFormatUtils.format(new Date(), "yyyyMMdd"));
      List<String> codeList = this.generateCodeService.generateCode(ruleCode, 1, 5, 2, TimeUnit.DAYS);
      Validate.isTrue(CollectionUtils.isNotEmpty(codeList), "添加信息时，生成换货单编码失败！");
      dto.setExchangeCode(codeList.get(0));
    }

    // 汇总：退货数量，换货数量，退货金额，换货金额，差额（换货减退货）
    this.fieldSummary(dto, returnedGoodsVo, orderVo);

    // 按优先级执行核心逻辑校验
    if (CollectionUtils.isNotEmpty(exchangeVerificationStrategies)) {
      this.exchangeVerificationStrategies = this.exchangeVerificationStrategies.stream().sorted(Comparator.comparing(ExchangeVerificationStrategy::order)).collect(Collectors.toList());
      this.exchangeVerificationStrategies.forEach(t-> t.execute(dto));
    }

    // 创建换货单
    ExchangeEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, ExchangeEntity.class, HashSet.class, ArrayList.class);
    this.exchangeRepository.saveOrUpdate(entity);
    dto.setId(entity.getId());

    // 订单创建后重新计算真实的差额
    OrderConfirmDto orderConfirmDto = this.nebulaToolkitService.copyObjectByWhiteList(orderVo, OrderConfirmDto.class, HashSet.class, ArrayList.class, "orderPays","orderDetails","orderDetails.orderDetailPays");
    dto.setOrder(orderConfirmDto);
    ExchangeVo exchangeVoReal = this.difference(dto);
    exchangeVoReal.setId(entity.getId());

    // 创建换货单差额表
    List<ExchangeDifferenceDto> differenceDtoList = (List<ExchangeDifferenceDto>) this.nebulaToolkitService.copyCollectionByWhiteList(exchangeVoReal.getDifferenceList(), ExchangeDifferenceVo.class, ExchangeDifferenceDto.class, HashSet.class, ArrayList.class);
    this.exchangeDifferenceVoService.createBatch(entity.getExchangeCode(), differenceDtoList);
    // 创建换货单附件
    this.exchangeFileVoService.createBatch(entity.getExchangeCode(), dto.getFileList());

    // 发送通知
    if (CollectionUtils.isNotEmpty(exchangeEventListeners)) {
      this.exchangeEventListeners.forEach(event -> event.onCreate(exchangeVoReal));
    }
    return exchangeVoReal;
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<ExchangeEntity> entities = this.exchangeRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    // 可删除审批驳回、暂存、流程追回状态的换货单
    Set<String> set = Sets.newHashSet(
        ExchangeStatusEnum.STAGING.getDictCode(),
        ExchangeStatusEnum.REJECTED.getDictCode(),
        ExchangeStatusEnum.CANCELED.getDictCode()
    );
    entities.forEach(e-> Validate.isTrue(set.contains(e.getExchangeStatus()), String.format("可删除审批驳回、暂存、流程追回状态的换货单，编码[%s]当前状态不能删除", e.getExchangeCode())));
    this.exchangeRepository.updateDelFlagByIds(ids);
    // 发送删除通知
    if (CollectionUtils.isNotEmpty(exchangeEventListeners)) {
      List<ExchangeVo> voList = (List<ExchangeVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, ExchangeEntity.class, ExchangeVo.class, HashSet.class, ArrayList.class);
      this.exchangeEventListeners.forEach(event -> event.onDelete(voList));
    }
  }

  @Override
  public ExchangeVo findByExchangeCode(String exchangeCode) {
    if (StringUtils.isBlank(exchangeCode)) {
      return null;
    }
    ExchangeEntity entity = this.exchangeRepository.findByExchangeCode(exchangeCode);
    if (entity == null) {
      return null;
    }
    ExchangeVo exchangeVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, ExchangeVo.class, HashSet.class, ArrayList.class);
    List<ExchangeDifferenceVo> differenceVoList = this.exchangeDifferenceVoService.findByExchangeCode(exchangeCode);
    exchangeVo.setDifferenceList(differenceVoList);
    return exchangeVo;
  }

  @Override
  public void updateExchangeStatusById(String exchangeStatus, String id) {
    Validate.notBlank(exchangeStatus, "换货单状态不能为空");
    Validate.notBlank(id, "换货单ID不能为空");
    this.exchangeRepository.updateExchangeStatusById(exchangeStatus, id);
  }

  @Override
  public void updateOrderStatusAndProcessNumberById(String exchangeStatus, String processNumber, String id) {
    this.exchangeRepository.updateOrderStatusAndProcessNumberById(exchangeStatus, processNumber, id);
  }

  @Override
  public ExchangeVo findDetailByCode(String code) {
    if (StringUtils.isBlank(code)) {
      return null;
    }
    ExchangeEntity entity = this.exchangeRepository.findByExchangeCode(code);
    if (entity == null) {
      return null;
    }
    ExchangeVo exchangeVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, ExchangeVo.class, HashSet.class, ArrayList.class);
    //退货信息
    ReturnedGoodsVo returnedGoodsVo = this.returnedGoodsVoService.findDetailById(exchangeVo.getProduceReturnId());
    exchangeVo.setReturnedGoods(returnedGoodsVo);
    //订单信息
    OrderVo orderVo = this.orderVoService.findById(exchangeVo.getProduceOrderId());
    exchangeVo.setOrder(orderVo);
    //差额信息
    List<ExchangeDifferenceVo> differenceVoList = this.exchangeDifferenceVoService.findByExchangeCode(entity.getExchangeCode());
    exchangeVo.setDifferenceList(differenceVoList);
    //附件信息
    List<ExchangeFileVo> fileVoList = this.exchangeFileVoService.findByExchangeCode(entity.getExchangeCode());
    exchangeVo.setFileList(fileVoList);
    return exchangeVo;
  }

  /**
   * 创建时校验项
   */
  private void createValidation(ExchangeDto dto) {
    this.validation(dto);
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
  }

  /**
   * 编辑时校验项
   */
  private void updateValidation(ExchangeDto dto) {
    this.validation(dto);
    Validate.notBlank(dto.getId(), "ID不能为空");
    Validate.notNull(dto.getDifferenceList(), "换货单差额集合信息不能为空");
  }

  /**
   * 公共校验项
   */
  private void validation(ExchangeDto dto) {
    Validate.notNull(dto, "换货单对象不能为空");
    Validate.notNull(dto.getDraft(), "暂存标识，不能为空！");
    Validate.notNull(dto.getExchangeTime(), "换货时间不能为空");
    Validate.notBlank(dto.getOriginalOrderCode(), "原订单编号不能为空");
    Validate.notBlank(dto.getCustomerCode(), "客户编码不能为空");
    Validate.notBlank(dto.getCustomerName(), "客户名称不能为空");
    Validate.notBlank(dto.getReason(), "换货原因不能为空");
    //因为第一步填写的退货信息，所以必须有它
    Validate.notNull(dto.getReturnedGoods(), "退货信息不能为空");
    //可编辑状态检查
    if (StringUtils.isNotBlank(dto.getId())) {
      ExchangeEntity oldEntity = this.exchangeRepository.findById(dto.getId());
      Validate.notNull(oldEntity, "换货单对象不存在或已删除");
      // 可编辑审批驳回、暂存、流程追回状态的换货单
      Set<String> set = Sets.newHashSet(
          ExchangeStatusEnum.STAGING.getDictCode(),
          ExchangeStatusEnum.REJECTED.getDictCode(),
          ExchangeStatusEnum.CANCELED.getDictCode()
      );
      Validate.isTrue(set.contains(oldEntity.getExchangeStatus()), "当前换货单状态不能编辑，可编辑审批驳回、暂存、流程追回状态的换货单");
    }
  }

  /**
   * 为换货单提交业务提供的数据完整性校验方法
   */
  private void submitValidate(ExchangeDto dto) {
    Validate.notNull(dto.getReturnedGoods(), "退货信息不能为空");
    Validate.notNull(dto.getOrder(), "订单信息不能为空");
  }
}
