package com.biz.crm.dms.business.exchange.local.service.notifier;

import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.dms.business.exchange.local.service.helper.ExchangeHelper;
import com.biz.crm.dms.business.exchange.local.service.helper.OrderHelper;
import com.biz.crm.dms.business.exchange.sdk.constant.ExchangeConstant;
import com.biz.crm.dms.business.exchange.sdk.enums.ExchangeStatusEnum;
import com.biz.crm.dms.business.exchange.sdk.service.ExchangeVoService;
import com.biz.crm.dms.business.exchange.sdk.vo.ExchangeVo;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderStatusEnum;
import com.biz.crm.dms.business.order.sdk.service.OrderVoService;
import com.biz.crm.dms.business.order.sdk.vo.OrderVo;
import com.biz.crm.dms.business.returned.goods.sdk.enums.goods.ReturnedGoodsStatusEnum;
import com.biz.crm.dms.business.returned.goods.sdk.service.goods.ReturnedGoodsVoService;
import com.biz.crm.workflow.sdk.dto.ProcessStatusDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.listener.ProcessCompleteListener;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * 换货单的工作流的回调处理
 *
 * @author hefan
 * @date 2022/08/31
 */
@Component
public class ProcessCompleteListenerOfExchange implements ProcessCompleteListener {
  @Autowired(required = false)
  private ExchangeVoService exchangeVoService;

  @Autowired(required = false)
  private OrderHelper orderHelper;

  @Autowired(required = false)
  private ExchangeHelper exchangeHelper;

  @Autowired(required = false)
  private OrderVoService orderVoService;

  @Autowired(required = false)
  private ReturnedGoodsVoService returnedGoodsVoService;

  @Override
  public String getBusinessCode() {
    return ExchangeConstant.EXCHANGE_FORM_TYPE;
  }

  @Override
  public void onProcessComplete(ProcessStatusDto dto) {
    if (!dto.getBusinessCode().equals(ExchangeConstant.EXCHANGE_FORM_TYPE)) {
      return;
    }
    // 查询换货单是否存在
    String formNo = dto.getBusinessNo();
    ExchangeVo exchangeVo = this.exchangeVoService.findByExchangeCode(formNo);
    if (Objects.isNull(exchangeVo)) {
      return;
    }
    // 原换货单数据的状态是否可以被回调修改
    String exchangeStatus = exchangeVo.getExchangeStatus();
    String awaitApprove = ExchangeStatusEnum.AWAIT_APPROVE.getDictCode();
    boolean equals = awaitApprove.equals(exchangeStatus);
    Validate.isTrue(equals, "此换货单非审批中状态，无法接受工作流的操作！");
    // 修改换货单的状态
    String processState = dto.getProcessStatus();
    // - 把工作流状态转换为业务状态
    exchangeStatus = this.getExchangeStatus(processState);
    this.exchangeVoService.updateExchangeStatusById(exchangeStatus, exchangeVo.getId());
    OrderVo orderVo = this.orderVoService.findById(exchangeVo.getProduceOrderId());
    Validate.notNull(orderVo, "审批时未获取到订单信息！");
    // 工作流操作延申的业务操作
    boolean isCompleted = ExchangeStatusEnum.COMPLETED.getDictCode().equals(exchangeStatus);
    if (isCompleted) {
      // 审批通过，把资金占用变成扣减
      this.orderHelper.paymentExchange(exchangeVo);
      // 审批通过，差额为正的进行上账
      this.exchangeHelper.upAccount(exchangeVo);
      //同步订单状态
      this.orderVoService.updateOrderStatusAndIsShowByOrderCode(exchangeVo.getProduceOrderCode(), OrderStatusEnum.WAIT_SHIPPED, Boolean.TRUE);
      //同步退货单状态
      this.returnedGoodsVoService.updateStatusAndIsShowByReturnedGoodsCode(exchangeVo.getProduceReturnCode(), ReturnedGoodsStatusEnum.PENDING_STOCK, BooleanEnum.TRUE);
      // 因为提交审批时是用退货单与订单差额为负数的绝对值做的占用，所以审批通过不需要执行退货单的回退逻辑
      return;
    }
    /**
     * 释放费用
     * 归还促销政策额度
     * 归还库存
     * 更新退货单状态
     * 更新订单状态
     */
    this.orderHelper.releaseExchange(exchangeVo);
    this.orderHelper.releaseSalePolicy(orderVo);
    this.orderHelper.returnStock(orderVo);

    // 默认状态 - 审批驳回：回退扣减的费用；释放占用的库存
    ReturnedGoodsStatusEnum returnedGoodsStatusEnum = ReturnedGoodsStatusEnum.REJECTED;
    OrderStatusEnum orderStatusEnum = OrderStatusEnum.REJECTED;
    // - 流程撤回：回退扣减的费用；释放占用的库存
    if (ProcessStatusEnum.RECOVER.getDictCode().equals(processState.toString())) {
      returnedGoodsStatusEnum = ReturnedGoodsStatusEnum.RECALL;
      orderStatusEnum = OrderStatusEnum.CANCELED;
    }
    this.returnedGoodsVoService.updateStatusAndIsShowByReturnedGoodsCode(exchangeVo.getProduceReturnCode(), returnedGoodsStatusEnum, BooleanEnum.FALSE);
    this.orderVoService.updateOrderStatusAndIsShowByOrderCode(exchangeVo.getProduceOrderCode(), orderStatusEnum, Boolean.FALSE);
  }

  /**
   * 把工作流状态转换为业务状态
   * @param processState
   * @return
   */
  private String getExchangeStatus(String processState) {
    Map<String, String> map = new HashMap<>(ProcessStatusEnum.values().length);
    map.put(ProcessStatusEnum.PASS.getDictCode(), ExchangeStatusEnum.COMPLETED.getDictCode());
    map.put(ProcessStatusEnum.REJECT.getDictCode(), ExchangeStatusEnum.REJECTED.getDictCode());
    map.put(ProcessStatusEnum.RECOVER.getDictCode(), ExchangeStatusEnum.CANCELED.getDictCode());
    return map.get(processState);
  }

}
