package com.biz.crm.dms.business.exchange.local.strategy;


import com.biz.crm.dms.business.costpool.sdk.dto.CostPoolDto;
import com.biz.crm.dms.business.costpool.sdk.service.CostPoolVoService;
import com.biz.crm.dms.business.costpool.sdk.vo.CostPoolVo;
import com.biz.crm.dms.business.exchange.sdk.dto.ExchangeDifferenceDto;
import com.biz.crm.dms.business.exchange.sdk.dto.ExchangeDto;
import com.biz.crm.dms.business.exchange.sdk.strategy.ExchangeVerificationStrategy;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.List;

/**
 * 差额校验
 * a. 差额为负时，校验各项金额的绝对值是否超各资金池、授信池、折扣池、货补池当前可用余额。
 * b. 校验通过，需要按照为负的差额的绝对值做相关池子的占用。
 *
 * @author pengxi
 * @date 2022/7/12
 */
@Slf4j
@Component
public class DifferenceVerificationStrategyImpl implements ExchangeVerificationStrategy {

  @Autowired(required = false)
  private CostPoolVoService costPoolVoService;

  @Override
  public void execute(ExchangeDto exchangeDto) {
    Validate.notEmpty(exchangeDto.getDifferenceList(), "换货单差额集合信息不能为空！");
    List<ExchangeDifferenceDto> differenceDtoList = exchangeDto.getDifferenceList();
    differenceDtoList.forEach(diff->{
      if (BigDecimal.ZERO.compareTo(diff.getExchangeAmount()) < 1) {
        // 差额为正，不需要校验
        return;
      }
      //按客户编码和费用池类型获取费用池
      CostPoolDto costPoolDto = new CostPoolDto();
      costPoolDto.setPoolType(diff.getPoolType());
      costPoolDto.setCustomerCode(exchangeDto.getCustomerCode());
      List<CostPoolVo> costPoolVoList = this.costPoolVoService.handleRequestCostPoolVos(costPoolDto);
      Validate.notEmpty(costPoolVoList, String.format("未获取到类型为[%s]的费用池数据！", diff.getPoolType()));
      CostPoolVo costPoolVo = costPoolVoList.get(0);
      Validate.isTrue(diff.getExchangeAmount().abs().compareTo(costPoolVo.getUsableAmount()) < 1, String.format("当前差额绝对值%s，超出费用池[%s]当前可用余额%s！",diff.getExchangeAmount().abs(), diff.getSourceCode(), costPoolVo.getUsableAmount()));
    });
  }

  @Override
  public int order() {
    return 1;
  }
}
