package com.biz.crm.dms.business.exchange.local.strategy;


import com.biz.crm.dms.business.exchange.sdk.dto.ExchangeDto;
import com.biz.crm.dms.business.exchange.sdk.strategy.ExchangeVerificationStrategy;
import com.biz.crm.dms.business.returned.goods.sdk.dto.goods.ReturnedGoodsDto;
import com.biz.crm.dms.business.returned.goods.sdk.dto.goods.ReturnedGoodsReplenishmentDto;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.List;

/**
 * 货补校验
 * a. 货补明细列表中填写的“货补金额”的加和=本单退还货补费用。
 *
 * @author pengxi
 * @date 2022/7/12
 */
@Slf4j
@Component
public class ReplenishmentVerificationStrategyImpl implements ExchangeVerificationStrategy {

  @Override
  public void execute(ExchangeDto exchangeDto) {
    Validate.notNull(exchangeDto.getReturnedGoods(), "退货信息不能为空！");
    ReturnedGoodsDto returnedGoodsDto = exchangeDto.getReturnedGoods();
    if (returnedGoodsDto.getReplenishmentAmount() == null) {
      //没有用到货补
      return;
    }
    //退还货补明细集合
    List<ReturnedGoodsReplenishmentDto> replenishmentList = returnedGoodsDto.getReplenishmentList();
    if (CollectionUtils.isEmpty(replenishmentList)) {
      return;
    }
    BigDecimal replenishmentAmount = replenishmentList.stream()
        .map(ReturnedGoodsReplenishmentDto::getAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
    Validate.isTrue(returnedGoodsDto.getReplenishmentAmount().compareTo(replenishmentAmount) == 0, "必须满足：货补明细列表中填写的“货补金额”的加和=本单退还货补费用");
  }

  @Override
  public int order() {
    return 1;
  }
}
