package com.biz.crm.dms.business.interaction.local.service.carouselPicture.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.interaction.local.entity.carouselPicture.CarouselPictureEntity;
import com.biz.crm.dms.business.interaction.local.repository.carouselPicture.CarouselPictureRepository;
import com.biz.crm.dms.business.interaction.local.service.base.AbstractRelationDataService;
import com.biz.crm.dms.business.interaction.local.service.carouselPicture.CarouselPictureScopeService;
import com.biz.crm.dms.business.interaction.local.service.carouselPicture.CarouselPictureService;
import com.biz.crm.dms.business.interaction.sdk.dto.carouselPicture.CarouselPictureCustomerPageDto;
import com.biz.crm.dms.business.interaction.sdk.dto.carouselPicture.CarouselPictureDto;
import com.biz.crm.dms.business.interaction.sdk.dto.carouselPicture.CarouselPicturePageDto;
import com.biz.crm.dms.business.interaction.sdk.enums.ScopeType;
import com.biz.crm.dms.business.interaction.sdk.event.CarouselPictureEventListener;
import com.biz.crm.dms.business.interaction.sdk.vo.carouselPicture.CarouselPictureEventVo;
import com.biz.crm.dms.business.interaction.sdk.vo.carouselPicture.CarouselPictureVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 图片信息Service实现
 *
 * @author huangzhi
 */
@Service
public class CarouselPictureServiceImpl extends AbstractRelationDataService implements CarouselPictureService {

  @Autowired(required = false)
  CarouselPictureRepository carouselPictureRepository;

  @Autowired(required = false)
  CarouselPictureScopeService carouselPictureScopeService;

  @Autowired(required = false)
  NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  List<CarouselPictureEventListener> listeners;

  private void createValidation(CarouselPictureDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getPictureDescription(), "图片描述不能为空!");
    Validate.notNull(dto.getPictureSort(), "图片顺序不能为空!");
    Validate.notBlank(dto.getPictureName(), "图片名称不能为空!");
    Validate.notNull(dto.getStartTime(), "有效开始时间不能为空!");
    Validate.notNull(dto.getEndTime(), "有效结束时间不能为空!");
    Validate.isTrue(dto.getStartTime().before(dto.getEndTime()), "生效开始时间必须小于生效结束时间!");
    if (CollectionUtils.isNotEmpty(dto.getScopeList())) {
      dto.getScopeList().forEach(pictureScopeDto -> {
        List<String> scopeTypeList = Arrays.stream(ScopeType.values()).map(ScopeType::name)
            .filter(code -> !code.equals("CHANNEL")).collect(Collectors.toList());
        Validate.isTrue(scopeTypeList.contains(pictureScopeDto.getScopeType()), "不支持的范围类型!");
      });
    }
  }

  private void updateValidation(CarouselPictureDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入！");
    Validate.notBlank(dto.getId(), "图片id不能为空！");
    Validate.notBlank(dto.getPictureDescription(), "图片描述不能为空！");
    Validate.notNull(dto.getPictureSort(), "图片顺序不能为空！");
    Validate.notBlank(dto.getPictureName(), "图片名称不能为空！");
    Validate.notNull(dto.getStartTime(), "有效开始时间不能为空！");
    Validate.notNull(dto.getEndTime(), "有效结束时间不能为空！");
    Validate.isTrue(dto.getStartTime().before(dto.getEndTime()), "生效开始时间必须小于生效结束时间!");
    if (CollectionUtils.isNotEmpty(dto.getScopeList())) {
      dto.getScopeList().forEach(scopeDto -> {
        List<String> scopeTypeList = Arrays.stream(ScopeType.values()).map(ScopeType::name)
            .filter(code -> !code.equals(ScopeType.CHANNEL.name())).collect(Collectors.toList());
        Validate.isTrue(scopeTypeList.contains(scopeDto.getScopeType()), "不支持的范围类型");
      });
    }
  }

  @Override
  @Transactional
  public CarouselPictureEntity create(CarouselPictureDto dto) {
    createValidation(dto);
    CarouselPictureEntity entity = nebulaToolkitService.copyObjectByWhiteList(dto, CarouselPictureEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    String tenant = TenantUtils.getTenantCode();
    entity.setTenantCode(tenant);
    carouselPictureRepository.save(entity);

    //绑定关联数据
    carouselPictureScopeService.update(dto.getScopeList(), entity.getId());

    //图片新增通知
    if (CollectionUtils.isNotEmpty(listeners)) {
      CarouselPictureEventVo picEventVo = nebulaToolkitService.copyObjectByWhiteList(dto, CarouselPictureEventVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(picEventListener -> picEventListener.onCreate(picEventVo));
    }

    return entity;
  }

  @Override
  @Transactional
  public CarouselPictureEntity update(CarouselPictureDto dto) {
    updateValidation(dto);
    CarouselPictureEntity entity = this.carouselPictureRepository.getById(dto.getId());
    Validate.notNull(entity, "图片信息不存在");
    CarouselPictureEntity newEntity = nebulaToolkitService.copyObjectByWhiteList(dto, CarouselPictureEntity.class, HashSet.class, ArrayList.class);
    carouselPictureRepository.updateById(newEntity);

    //绑定关联数据
    carouselPictureScopeService.update(dto.getScopeList(), newEntity.getId());

    //图片更新通知
    if (CollectionUtils.isNotEmpty(listeners)) {
      CarouselPictureEventVo oldVo = nebulaToolkitService.copyObjectByWhiteList(entity, CarouselPictureEventVo.class, HashSet.class, ArrayList.class);
      CarouselPictureEventVo newVo = nebulaToolkitService.copyObjectByWhiteList(dto, CarouselPictureEventVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(picEventListener -> picEventListener.onUpdate(oldVo, newVo));
    }

    return entity;
  }


  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "缺失id");
    List<CarouselPictureEntity> entities = carouselPictureRepository.listByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "删除个数不匹配");
    carouselPictureRepository.updateDelFlagByIds(ids);

    //轮播图删除通知
    if (CollectionUtils.isNotEmpty(listeners)) {
      List<CarouselPictureEventVo> voList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(entities, CarouselPictureEntity.class, CarouselPictureEventVo.class, HashSet.class, ArrayList.class));
      listeners.forEach(picEventListener -> picEventListener.onDelete(voList)
      );
    }
  }

  @Transactional
  public void enableBatch(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) return;
    List<CarouselPictureEntity> entities = carouselPictureRepository.listByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "数据启用个数不匹配");
    carouselPictureRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);

    //轮播图启用通知
    if (CollectionUtils.isNotEmpty(listeners)) {
      List<CarouselPictureEventVo> voList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(entities, CarouselPictureEntity.class, CarouselPictureEventVo.class, HashSet.class, ArrayList.class));
      listeners.forEach(picEventListener -> picEventListener.onEnable(voList)
      );
    }
  }

  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue((CollectionUtils.isNotEmpty(ids)), "缺失id");
    List<CarouselPictureEntity> entities = carouselPictureRepository.listByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "数据禁用个数不匹配");
    this.carouselPictureRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);

    //图片禁用事件通知
    if (CollectionUtils.isNotEmpty(listeners)) {
      List<CarouselPictureEventVo> voList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(entities, CarouselPictureEntity.class, CarouselPictureEventVo.class, HashSet.class, ArrayList.class));
      listeners.forEach(picEventListener -> picEventListener.onDisable(voList)
      );
    }
  }

  @Override
  public Page<CarouselPictureEntity> findByCarouselPictureCustomerPageDto(Pageable pageable, CarouselPictureCustomerPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new CarouselPictureCustomerPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    this.buildCustomerPageDto(dto);
    List<String> scopeCodeList = Lists.newArrayList();
    dto.getOrgCodeList().forEach(o -> {
      scopeCodeList.add(o);
    });
    scopeCodeList.add(dto.getCustomerCode());
    dto.setScopeCodeList(scopeCodeList);
    return carouselPictureRepository.findByCarouselPictureCustomerPageDto(pageable, dto);
  }
}
