package com.biz.crm.dms.business.interaction.local.service.complaint.internal;

import com.biz.crm.dms.business.interaction.local.entity.complaint.ComplaintEntity;
import com.biz.crm.dms.business.interaction.local.entity.complaint.ComplaintReplyEntity;
import com.biz.crm.dms.business.interaction.local.repository.complaint.ComplaintReplyRepository;
import com.biz.crm.dms.business.interaction.local.repository.complaint.ComplaintRepository;
import com.biz.crm.dms.business.interaction.local.service.complaint.ComplaintReplyFileService;
import com.biz.crm.dms.business.interaction.local.service.complaint.ComplaintReplyService;
import com.biz.crm.dms.business.interaction.sdk.dto.complaint.ComplaintReplyDto;
import com.biz.crm.dms.business.interaction.sdk.enums.ComplaintState;
import com.biz.crm.dms.business.interaction.sdk.enums.ReplierType;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 投诉回复表服务实现类
 *
 * @author ning.zhang
 * @date 2021-11-24 16:40:08
 */
@Slf4j
@Service("complaintReplyService")
public class ComplaintReplyServiceImpl implements ComplaintReplyService {

  @Autowired(required = false)
  private ComplaintReplyRepository complaintReplyRepository;
  @Autowired(required = false)
  private ComplaintRepository complaintRepository;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private ComplaintReplyFileService complaintReplyFileService;

  @Override
  @Transactional
  public ComplaintReplyEntity create(ComplaintReplyDto dto) {
    this.createValidation(dto);
    ComplaintEntity complaintEntity = complaintRepository.getById(dto.getComplaintId());
    Validate.notNull(complaintEntity, "投诉建议不存在,无法回复!");
    String complaintState;
    if (ReplierType.FACTORY.getCode().equals(dto.getReplierType())) {
      complaintState = ComplaintState.WAIT_CUSTOMER_RETURN_VISIT.getCode();
    } else {
      //只有投诉状态为待厂家处理或者待客户回复,客户才能进行回复
      Validate.isTrue(ComplaintState.WAIT_FACTORY_DEAL.getCode().equals(complaintEntity.getComplaintState())
              || ComplaintState.WAIT_CUSTOMER_RETURN_VISIT.getCode().equals(complaintEntity.getComplaintState())
          , String.format("当前投诉建议状态为:%s,无法回复!", ComplaintState.getDesc(complaintEntity.getComplaintState())));
      complaintState = ComplaintState.WAIT_FACTORY_DEAL.getCode();
    }
    ComplaintReplyEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, ComplaintReplyEntity.class, HashSet.class, ArrayList.class);
    entity.setCreateTime(new Date());
    this.complaintReplyRepository.save(entity);
    //更新投诉状态
    this.complaintRepository.updateComplaintStateById(complaintState, dto.getComplaintId());
    //保存文件信息
    this.complaintReplyFileService.create(dto.getFileList(), entity.getId());
    return entity;
  }

  /**
   * 在创建ComplaintReply模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(ComplaintReplyDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getComplaintId(), "缺失投诉id");
    Validate.notBlank(dto.getReplierType(), "缺失回复人类型");
    Validate.notBlank(dto.getReplyContent(), "缺失回复内容");
    List<String> replierTypeList = Arrays.stream(ReplierType.values()).map(ReplierType::getCode).collect(Collectors.toList());
    Validate.isTrue(replierTypeList.contains(dto.getReplierType()),"不支持的回复人类型!");
    Validate.isTrue(dto.getReplyContent().length() < 500, "回复内容，在进行编辑时填入值超过了限定长度(500)，请检查!");
  }
}
