package com.biz.crm.dms.business.interaction.local.service.information.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.interaction.local.entity.information.CompanyInformationEntity;
import com.biz.crm.dms.business.interaction.local.repository.information.CompanyInformationRepository;
import com.biz.crm.dms.business.interaction.local.service.base.AbstractRelationDataService;
import com.biz.crm.dms.business.interaction.local.service.information.CompanyInformationFileService;
import com.biz.crm.dms.business.interaction.local.service.information.CompanyInformationScopeService;
import com.biz.crm.dms.business.interaction.local.service.information.CompanyInformationService;
import com.biz.crm.dms.business.interaction.sdk.dto.information.CompanyInformationCustomerPageDto;
import com.biz.crm.dms.business.interaction.sdk.dto.information.CompanyInformationDto;
import com.biz.crm.dms.business.interaction.sdk.dto.information.CompanyInformationPageDto;
import com.biz.crm.dms.business.interaction.sdk.enums.ScopeType;
import com.biz.crm.dms.business.interaction.sdk.event.CompanyInformationEventListener;
import com.biz.crm.dms.business.interaction.sdk.vo.information.CompanyInformationEventVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 公司资料表服务实现类
 *
 * @author ning.zhang
 * @date 2021-11-24 13:34:46
 */
@Slf4j
@Service("companyInformationService")
public class CompanyInformationServiceImpl extends AbstractRelationDataService implements CompanyInformationService {

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private CompanyInformationRepository noticeRepository;
  @Autowired(required = false)
  private CompanyInformationFileService noticeFileService;
  @Autowired(required = false)
  private CompanyInformationScopeService noticeScopeService;
  @Autowired(required = false)
  private List<CompanyInformationEventListener> listeners;

  @Override
  @Transactional
  public CompanyInformationEntity create(CompanyInformationDto dto) {
    this.createValidation(dto);
    CompanyInformationEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, CompanyInformationEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    //TODO 设置用户的组织信息
    this.noticeRepository.save(entity);
    //绑定关联数据
    this.noticeFileService.update(dto.getFileList(), entity.getId());
    this.noticeScopeService.update(dto.getScopeList(), entity.getId());
    //新增公司资料事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      CompanyInformationEventVo vo = this.nebulaToolkitService.copyObjectByBlankList(entity, CompanyInformationEventVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onCreate(vo);
      });
    }
    return entity;
  }

  @Override
  @Transactional
  public CompanyInformationEntity update(CompanyInformationDto dto) {
    this.updateValidation(dto);
    CompanyInformationEntity entity = this.noticeRepository.getById(dto.getId());
    Validate.notNull(entity, "公司资料信息不存在");
    CompanyInformationEntity updateEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, CompanyInformationEntity.class, HashSet.class, ArrayList.class);
    this.noticeRepository.updateById(updateEntity);
    //绑定关联数据
    this.noticeFileService.update(dto.getFileList(), entity.getId());
    this.noticeScopeService.update(dto.getScopeList(), entity.getId());
    //更新公司资料事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      CompanyInformationEventVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, CompanyInformationEventVo.class, HashSet.class, ArrayList.class);
      CompanyInformationEventVo newVo = this.nebulaToolkitService.copyObjectByBlankList(dto, CompanyInformationEventVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onUpdate(oldVo, newVo);
      });
    }
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<CompanyInformationEntity> entities = this.noticeRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    this.noticeRepository.updateDelFlagByIds(ids);
    //删除公司资料事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<CompanyInformationEventVo> voList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(entities, CompanyInformationEntity.class
          , CompanyInformationEventVo.class, HashSet.class, ArrayList.class));
      listeners.forEach(listener -> {
        listener.onDelete(voList);
      });
    }
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<CompanyInformationEntity> entities = this.noticeRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据启用个数不匹配");
    this.noticeRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    //启用公司资料事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<CompanyInformationEventVo> voList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(entities, CompanyInformationEntity.class
          , CompanyInformationEventVo.class, HashSet.class, ArrayList.class));
      listeners.forEach(listener -> {
        listener.onEnable(voList);
      });
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<CompanyInformationEntity> entities = this.noticeRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据禁用个数不匹配");
    this.noticeRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    //禁用公司资料事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<CompanyInformationEventVo> voList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(entities, CompanyInformationEntity.class
          , CompanyInformationEventVo.class, HashSet.class, ArrayList.class));
      listeners.forEach(listener -> {
        listener.onDisable(voList);
      });
    }
  }

  @Override
  public Page<CompanyInformationEntity> findByConditions(Pageable pageable, CompanyInformationPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new CompanyInformationPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    return this.noticeRepository.findByConditions(pageable, dto);
  }

  @Override
  public Page<CompanyInformationEntity> findByCompanyInformationCustomerPageDto(Pageable pageable, CompanyInformationCustomerPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new CompanyInformationCustomerPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    this.buildCustomerPageDto(dto);
    return this.noticeRepository.findByCompanyInformationCustomerPageDto(pageable, dto);
  }

  /**
   * 在创建CompanyInformation模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(CompanyInformationDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getName(), "缺失资料名称");
    Validate.notBlank(dto.getType(), "缺失资料类型");
    Validate.notNull(dto.getStartTime(), "缺失生效开始时间");
    Validate.notNull(dto.getEndTime(), "缺失生效结束时间");
    Validate.isTrue(dto.getStartTime().before(dto.getEndTime()), "生效开始时间必须小于生效结束时间");
    if (!CollectionUtils.isEmpty(dto.getScopeList())) {
      dto.getScopeList().forEach(noticeScopeDto -> {
        List<String> scopeTypeList = Arrays.stream(ScopeType.values()).map(ScopeType::name).collect(Collectors.toList());
        Validate.isTrue(scopeTypeList.contains(noticeScopeDto.getScopeType()), "不支持的范围类型!");
      });
    }
    Validate.isTrue(dto.getName().length() < 128, "资料名称，在进行编辑时填入值超过了限定长度(128)，请检查!");
  }

  /**
   * 在修改CompanyInformation模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(CompanyInformationDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getName(), "缺失资料名称");
    Validate.notBlank(dto.getType(), "缺失资料类型");
    Validate.notNull(dto.getStartTime(), "缺失生效开始时间");
    Validate.notNull(dto.getEndTime(), "缺失生效结束时间");
    Validate.isTrue(dto.getStartTime().before(dto.getEndTime()), "生效开始时间必须小于生效结束时间");
    if (!CollectionUtils.isEmpty(dto.getScopeList())) {
      dto.getScopeList().forEach(noticeScopeDto -> {
        List<String> scopeTypeList = Arrays.stream(ScopeType.values()).map(ScopeType::name).collect(Collectors.toList());
        Validate.isTrue(scopeTypeList.contains(noticeScopeDto.getScopeType()), "不支持的范围类型!");
      });
    }
    Validate.isTrue(dto.getName().length() < 128, "资料名称，在进行编辑时填入值超过了限定长度(128)，请检查!");
  }
}
