package com.biz.crm.dms.business.interaction.local.service.notice.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.LoginUserDetails;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.interaction.local.entity.notice.NoticeEntity;
import com.biz.crm.dms.business.interaction.local.repository.notice.NoticeRepository;
import com.biz.crm.dms.business.interaction.local.service.notice.NoticeFileService;
import com.biz.crm.dms.business.interaction.local.service.notice.NoticeScopeService;
import com.biz.crm.dms.business.interaction.local.service.notice.NoticeService;
import com.biz.crm.dms.business.interaction.sdk.dto.notice.NoticeDto;
import com.biz.crm.dms.business.interaction.sdk.enums.NoticePopupType;
import com.biz.crm.dms.business.interaction.sdk.enums.ScopeType;
import com.biz.crm.dms.business.interaction.sdk.event.NoticeEventListener;
import com.biz.crm.dms.business.interaction.sdk.vo.notice.NoticeEventVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 公告表服务实现类
 *
 * @author ning.zhang
 * @date 2021-11-22 16:10:03
 */
@Slf4j
@Service("noticeService")
public class NoticeServiceImpl implements NoticeService {

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private NoticeRepository noticeRepository;
  @Autowired(required = false)
  private NoticeFileService noticeFileService;
  @Autowired(required = false)
  private NoticeScopeService noticeScopeService;
  @Autowired(required = false)
  private List<NoticeEventListener> listeners;
  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Override
  @Transactional
  public NoticeEntity create(NoticeDto dto) {
    this.createValidation(dto);
    LoginUserDetails loginDetails = loginUserService.getLoginDetails(LoginUserDetails.class);
    NoticeEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, NoticeEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setPublishOrgCode(loginDetails.getOrgCode());
    entity.setPublishOrgName(loginDetails.getOrgName());
    this.noticeRepository.save(entity);
    //绑定关联数据
    this.noticeFileService.update(dto.getFileList(), entity.getId());
    this.noticeScopeService.update(dto.getScopeList(), entity.getId());
    //新增公告事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      NoticeEventVo vo = this.nebulaToolkitService.copyObjectByBlankList(entity, NoticeEventVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onCreate(vo);
      });
    }
    return entity;
  }


  @Override
  @Transactional
  public NoticeEntity update(NoticeDto dto) {
    this.updateValidation(dto);
    NoticeEntity entity = this.noticeRepository.getById(dto.getId());
    Validate.notNull(entity, "公告信息不存在");
    NoticeEntity updateEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, NoticeEntity.class, HashSet.class, ArrayList.class);
    this.noticeRepository.updateById(updateEntity);
    //绑定关联数据
    this.noticeFileService.update(dto.getFileList(), entity.getId());
    this.noticeScopeService.update(dto.getScopeList(), entity.getId());
    //更新公告事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      NoticeEventVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, NoticeEventVo.class, HashSet.class, ArrayList.class);
      NoticeEventVo newVo = this.nebulaToolkitService.copyObjectByBlankList(dto, NoticeEventVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onUpdate(oldVo, newVo);
      });
    }
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<NoticeEntity> entities = this.noticeRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    this.noticeRepository.updateDelFlagByIds(ids);
    //删除公告事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<NoticeEventVo> voList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(entities, NoticeEntity.class
          , NoticeEventVo.class, HashSet.class, ArrayList.class));
      listeners.forEach(listener -> {
        listener.onDelete(voList);
      });
    }
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<NoticeEntity> entities = this.noticeRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据启用个数不匹配");
    this.noticeRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    //启用公告事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<NoticeEventVo> voList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(entities, NoticeEntity.class
          , NoticeEventVo.class, HashSet.class, ArrayList.class));
      listeners.forEach(listener -> {
        listener.onEnable(voList);
      });
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<NoticeEntity> entities = this.noticeRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据禁用个数不匹配");
    this.noticeRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    //禁用公告事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<NoticeEventVo> voList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(entities, NoticeEntity.class
          , NoticeEventVo.class, HashSet.class, ArrayList.class));
      listeners.forEach(listener -> {
        listener.onDisable(voList);
      });
    }
  }

  /**
   * 在创建notice模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(NoticeDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getTitle(), "缺失标题");
    Validate.notBlank(dto.getType(), "缺失类型");
    Validate.notNull(dto.getStartTime(), "缺失生效开始时间");
    Validate.notNull(dto.getEndTime(), "缺失生效结束时间");
    Validate.notNull(dto.getIndexPopup(), "缺失首页是否弹出");
    if (Boolean.TRUE.equals(dto.getIndexPopup())) {
      Validate.notBlank(dto.getPopupType(), "缺失弹出类型");
      List<String> popupTypeList = Arrays.stream(NoticePopupType.values()).map(NoticePopupType::getDictCode).collect(Collectors.toList());
      Validate.isTrue(popupTypeList.contains(dto.getPopupType()), "不支持的弹出类型!");
    }
    Validate.isTrue(dto.getStartTime().before(dto.getEndTime()), "生效开始时间必须小于生效结束时间");
    if (!CollectionUtils.isEmpty(dto.getScopeList())) {
      dto.getScopeList().forEach(noticeScopeDto -> {
        List<String> scopeTypeList = Arrays.stream(ScopeType.values()).map(ScopeType::name).collect(Collectors.toList());
        Validate.isTrue(scopeTypeList.contains(noticeScopeDto.getScopeType()), "不支持的范围类型!");
      });
    }
    Validate.isTrue(dto.getTitle().length() < 128, "标题，在进行编辑时填入值超过了限定长度(128)，请检查!");
  }

  /**
   * 在修改notice模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(NoticeDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getTitle(), "缺失标题");
    Validate.notBlank(dto.getType(), "缺失类型");
    Validate.notNull(dto.getStartTime(), "缺失生效开始时间");
    Validate.notNull(dto.getEndTime(), "缺失生效结束时间");
    Validate.notNull(dto.getIndexPopup(), "缺失首页是否弹出");
    if (Boolean.TRUE.equals(dto.getIndexPopup())) {
      Validate.notBlank(dto.getPopupType(), "缺失弹出类型");
      List<String> popupTypeList = Arrays.stream(NoticePopupType.values()).map(NoticePopupType::getDictCode).collect(Collectors.toList());
      Validate.isTrue(popupTypeList.contains(dto.getPopupType()), "不支持的弹出类型!");
    }
    Validate.isTrue(dto.getStartTime().before(dto.getEndTime()), "生效开始时间必须小于生效结束时间");
    if (!CollectionUtils.isEmpty(dto.getScopeList())) {
      dto.getScopeList().forEach(noticeScopeDto -> {
        List<String> scopeTypeList = Arrays.stream(ScopeType.values()).map(ScopeType::name).collect(Collectors.toList());
        Validate.isTrue(scopeTypeList.contains(noticeScopeDto.getScopeType()), "不支持的范围类型!");
      });
    }
    Validate.isTrue(dto.getTitle().length() < 128, "标题，在进行编辑时填入值超过了限定长度(128)，请检查!");
  }
}
