package com.biz.crm.dms.business.interaction.local.service.notice.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.LoginUserDetails;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.interaction.local.constant.NoticeConstant;
import com.biz.crm.dms.business.interaction.local.entity.notice.NoticeEntity;
import com.biz.crm.dms.business.interaction.local.entity.notice.NoticeFileEntity;
import com.biz.crm.dms.business.interaction.local.entity.notice.NoticeReadRecordEntity;
import com.biz.crm.dms.business.interaction.local.entity.notice.NoticeScopeEntity;
import com.biz.crm.dms.business.interaction.local.model.NoticeModelVo;
import com.biz.crm.dms.business.interaction.local.repository.notice.NoticeFileRepository;
import com.biz.crm.dms.business.interaction.local.repository.notice.NoticeModelRepository;
import com.biz.crm.dms.business.interaction.local.repository.notice.NoticeReadRecordRepository;
import com.biz.crm.dms.business.interaction.local.repository.notice.NoticeRepository;
import com.biz.crm.dms.business.interaction.local.repository.notice.NoticeScopeRepository;
import com.biz.crm.dms.business.interaction.local.service.base.AbstractRelationDataService;
import com.biz.crm.dms.business.interaction.sdk.dto.notice.NoticeCustomerPageDto;
import com.biz.crm.dms.business.interaction.sdk.dto.notice.NoticeDto;
import com.biz.crm.dms.business.interaction.sdk.dto.notice.NoticePageDto;
import com.biz.crm.dms.business.interaction.sdk.enums.NoticePopupType;
import com.biz.crm.dms.business.interaction.sdk.enums.ScopeType;
import com.biz.crm.dms.business.interaction.sdk.service.notice.NoticeVoService;
import com.biz.crm.dms.business.interaction.sdk.vo.notice.NoticeVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 公告VO服务接口实现类
 *
 * @author ning.zhang
 * @date 2021/11/22
 */
@Service
public class NoticeVoServiceImpl extends AbstractRelationDataService implements NoticeVoService {

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private NoticeRepository noticeRepository;
  @Autowired(required = false)
  private NoticeModelRepository noticeModelRepository;
  @Autowired(required = false)
  private NoticeReadRecordRepository noticeReadRecordRepository;
  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired(required = false)
  private NoticeFileRepository noticeFileRepository;
  @Autowired(required = false)
  private NoticeScopeRepository noticeScopeRepository;

  @Override
  public Page<NoticeVo> findByConditions(Pageable pageable, NoticePageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new NoticePageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<NoticeModelVo> modelPage = this.noticeModelRepository.findByConditions(pageable, dto);
    Page<NoticeVo> pageResult = new Page<>(modelPage.getCurrent(), modelPage.getSize(), modelPage.getTotal());
    if (CollectionUtils.isEmpty(modelPage.getRecords())) {
      return pageResult;
    }
    List<NoticeVo> list = (List<NoticeVo>) this.nebulaToolkitService.copyCollectionByBlankList(modelPage.getRecords()
        , NoticeModelVo.class, NoticeVo.class, HashSet.class, ArrayList.class);
    Date now = new Date();
    list.forEach(e -> {
      if (e.getStartTime().after(now)) {
        e.setNoticeState(NoticeConstant.NOTICE_STATE_NOT_START);
      }
      if (!e.getStartTime().after(now) && !e.getEndTime().before(now)) {
        e.setNoticeState(NoticeConstant.NOTICE_STATE_START);
      }
      if (e.getEndTime().before(now)) {
        e.setNoticeState(NoticeConstant.NOTICE_STATE_ENDED);
      }
    });
    pageResult.setRecords(list);
    return pageResult;
  }

  /**
   * 客户-公告分页列表
   * 1.根据当前登录客户信息,封装请求参数dto中相应的查询参数
   * 2.分页查询出客户公告分页列表
   * 3.封装公告已读标志
   *
   * @param pageable 分页信息
   * @param dto      请求参数dto
   * @return Page<NoticeVo> 公告分页列表
   */
  @Override
  public Page<NoticeVo> findByNoticeCustomerPageDto(Pageable pageable, NoticeCustomerPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new NoticeCustomerPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    this.buildCustomerPageDto(dto);
    Page<NoticeEntity> entityPage = this.noticeRepository.findByNoticeCustomerPageDto(pageable, dto);
    Page<NoticeVo> pageResult = new Page<>(entityPage.getCurrent(), entityPage.getSize(), entityPage.getTotal());
    if (CollectionUtils.isEmpty(entityPage.getRecords())) {
      return pageResult;
    }
    //转换并封装公告相关信息
    List<NoticeVo> list = Lists.newArrayList(this.nebulaToolkitService
        .copyCollectionByBlankList(entityPage.getRecords(), NoticeEntity.class, NoticeVo.class, HashSet.class, ArrayList.class));
    List<String> noticeIds = list.stream().map(NoticeVo::getId).collect(Collectors.toList());
    List<NoticeReadRecordEntity> recordEntities = this.noticeReadRecordRepository.findByNoticeIdsAndCustomerCode(noticeIds, dto.getCustomerCode());
    Map<String, NoticeReadRecordEntity> recordEntityMap = Maps.newHashMap();
    if (!CollectionUtils.isEmpty(recordEntities)) {
      recordEntityMap = recordEntities.stream().collect(Collectors.toMap(NoticeReadRecordEntity::getNoticeId, t -> t, (key1, key2) -> key2));
    }
    for (NoticeVo noticeVo : list) {
      noticeVo.setHaveRead(Objects.nonNull(recordEntityMap.get(noticeVo.getId())));
    }
    Date now = new Date();
    list.forEach(e -> {
      if (e.getStartTime().after(now)) {
        e.setNoticeState(NoticeConstant.NOTICE_STATE_NOT_START);
      }
      if (!e.getStartTime().after(now) && !e.getEndTime().before(now)) {
        e.setNoticeState(NoticeConstant.NOTICE_STATE_START);
      }
      if (e.getEndTime().before(now)) {
        e.setNoticeState(NoticeConstant.NOTICE_STATE_ENDED);
      }
    });
    pageResult.setRecords(list);
    if(ObjectUtils.isNotEmpty(dto.getHaveRead())){
      List<NoticeVo> lists;
      if(dto.getHaveRead()){
        lists = list.stream().filter(NoticeVo::getHaveRead).collect(Collectors.toList());
      }else{
        lists = list.stream().filter(e -> !e.getHaveRead()).collect(Collectors.toList());
      }
      pageResult.setRecords(lists);
    }
    return pageResult;
  }

  @Override
  public NoticeVo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    NoticeEntity entity = this.noticeRepository.findById(id);
    if (Objects.isNull(entity)) {
      return null;
    }
    NoticeVo noticeVo = this.nebulaToolkitService.copyObjectByBlankList(entity, NoticeVo.class, HashSet.class, ArrayList.class);
    this.perfectScopeInfo(noticeVo.getScopeList());
    return noticeVo;
  }

  @Override
  @Transactional
  public void create(List<NoticeDto> dtoList) {
    this.createValidation(dtoList);
    LoginUserDetails loginDetails = loginUserService.getLoginDetails(LoginUserDetails.class);
    List<NoticeEntity> noticeEntities = dtoList.stream().map(dto -> {
      NoticeEntity entity = this.nebulaToolkitService.copyObjectByBlankList(dto, NoticeEntity.class, HashSet.class, ArrayList.class);
      entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      entity.setPublishOrgCode(loginDetails.getOrgCode());
      entity.setPublishOrgName(loginDetails.getOrgName());
      return entity;
    }).collect(Collectors.toList());
    this.noticeRepository.saveBatch(noticeEntities);
    this.saveExtInfo(noticeEntities);

  }

  /**
   * 保存扩展信息
   *
   * @param noticeEntities 公告实体列表
   */
  private void saveExtInfo(List<NoticeEntity> noticeEntities) {
    List<NoticeScopeEntity> noticeScopeEntities = Lists.newArrayList();
    List<NoticeFileEntity> noticeFileEntities = Lists.newArrayList();
    noticeEntities.forEach(noticeEntity -> {
      if (!CollectionUtils.isEmpty(noticeEntity.getScopeList())) {
        noticeEntity.getScopeList().forEach(noticeScopeEntity -> {
          noticeScopeEntity.setTenantCode(noticeEntity.getTenantCode());
          noticeScopeEntity.setNoticeId(noticeEntity.getId());
        });
        noticeScopeEntities.addAll(noticeEntity.getScopeList());
      }
      if (!CollectionUtils.isEmpty(noticeEntity.getFileList())) {
        noticeEntity.getFileList().forEach(noticeFileEntity -> {
          noticeFileEntity.setTenantCode(noticeEntity.getTenantCode());
          noticeFileEntity.setNoticeId(noticeEntity.getId());
        });
        noticeFileEntities.addAll(noticeEntity.getFileList());
      }
    });
    if (!CollectionUtils.isEmpty(noticeScopeEntities)) {
      this.noticeScopeRepository.saveBatch(noticeScopeEntities);
    }
    if (!CollectionUtils.isEmpty(noticeFileEntities)) {
      this.noticeFileRepository.saveBatch(noticeFileEntities);
    }
  }

  /**
   * 在创建notice模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dtoList 检查对象
   */
  private void createValidation(List<NoticeDto> dtoList) {
    Validate.isTrue(!CollectionUtils.isEmpty(dtoList), "进行当前操作时，信息对象必须传入!");
    dtoList.forEach(dto -> {
      dto.setId(null);
      dto.setTenantCode(TenantUtils.getTenantCode());
      Validate.notBlank(dto.getTitle(), "缺失标题");
      Validate.notBlank(dto.getType(), "缺失类型");
      Validate.notNull(dto.getStartTime(), "缺失生效开始时间");
      Validate.notNull(dto.getEndTime(), "缺失生效结束时间");
      Validate.notNull(dto.getIndexPopup(), "缺失首页是否弹出");
      if (Boolean.TRUE.equals(dto.getIndexPopup())) {
        Validate.notBlank(dto.getPopupType(), "缺失弹出类型");
        List<String> popupTypeList = Arrays.stream(NoticePopupType.values()).map(NoticePopupType::getDictCode).collect(Collectors.toList());
        Validate.isTrue(popupTypeList.contains(dto.getPopupType()), "不支持的弹出类型!");
      }
      Validate.isTrue(dto.getStartTime().before(dto.getEndTime()), "生效开始时间必须小于生效结束时间");
      if (!CollectionUtils.isEmpty(dto.getScopeList())) {
        dto.getScopeList().forEach(noticeScopeDto -> {
          List<String> scopeTypeList = Arrays.stream(ScopeType.values()).map(ScopeType::name).collect(Collectors.toList());
          Validate.isTrue(scopeTypeList.contains(noticeScopeDto.getScopeType()), "不支持的范围类型!");
        });
      }
      Validate.isTrue(dto.getTitle().length() < 128, "标题，在进行编辑时填入值超过了限定长度(128)，请检查!");
    });
  }
}
