package com.biz.crm.dms.business.interaction.local.service.complaint.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.interaction.local.entity.complaint.ComplaintEntity;
import com.biz.crm.dms.business.interaction.local.repository.complaint.ComplaintRepository;
import com.biz.crm.dms.business.interaction.local.service.base.AbstractRelationDataService;
import com.biz.crm.dms.business.interaction.local.service.complaint.ComplaintFileService;
import com.biz.crm.dms.business.interaction.local.service.complaint.ComplaintService;
import com.biz.crm.dms.business.interaction.sdk.dto.complaint.ComplaintCustomerPageDto;
import com.biz.crm.dms.business.interaction.sdk.dto.complaint.ComplaintDto;
import com.biz.crm.dms.business.interaction.sdk.dto.complaint.ComplaintPageDto;
import com.biz.crm.dms.business.interaction.sdk.dto.complaint.ComplaintStateDto;
import com.biz.crm.dms.business.interaction.sdk.enums.ComplaintState;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;

/**
 * 投诉表服务实现类
 *
 * @author ning.zhang
 * @date 2021-11-24 16:38:48
 */
@Service("complaintService")
public class ComplaintServiceImpl extends AbstractRelationDataService implements ComplaintService {

  @Autowired(required = false)
  private ComplaintRepository complaintRepository;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired(required = false)
  private ComplaintFileService complaintFileService;
  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Override
  @Transactional
  public ComplaintEntity create(ComplaintDto dto) {
    if(dto == null || StringUtils.isBlank(dto.getCustomerCode()) || StringUtils.isBlank(dto.getCustomerName())) {
      throw new IllegalArgumentException("错误的参数传递，请指定客户信息");
    }
    String customerCode = dto.getCustomerCode();
    String customerName = dto.getCustomerName();
    dto.setCustomerCode(customerCode);
    dto.setCustomerName(customerName);
    this.createValidation(dto);
    dto.setComplaintCode(this.generateCodeService.generateCode("TSJY", 1).get(0));
    ComplaintEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, ComplaintEntity.class, HashSet.class, ArrayList.class);
    entity.setComplaintState(ComplaintState.WAIT_FACTORY_DEAL.getCode());
    this.complaintRepository.save(entity);
    //保存文件信息
    this.complaintFileService.create(dto.getFileList(), entity.getId());
    return entity;
  }

  @Override
  @Transactional
  public void updateComplaintState(ComplaintStateDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    Validate.notBlank(dto.getComplaintId(), "缺失投诉id");
    Validate.notBlank(dto.getComplaintState(), "缺失投诉状态");
    Validate.isTrue(ComplaintState.RESOLVED.getCode().equals(dto.getComplaintState())
        || ComplaintState.UNDONE.getCode().equals(dto.getComplaintState()), "当前只支持解决和撤销操作!");
    ComplaintEntity entity = complaintRepository.getById(dto.getComplaintId());
    Validate.notNull(entity, "投诉建议不存在!");
    Validate.isTrue(ComplaintState.WAIT_FACTORY_DEAL.getCode().equals(entity.getComplaintState())
            || ComplaintState.WAIT_CUSTOMER_RETURN_VISIT.getCode().equals(entity.getComplaintState())
        , String.format("当前投诉建议状态为:%s,不支持当前操作!", ComplaintState.getDesc(entity.getComplaintState())));
    this.complaintRepository.updateComplaintStateById(dto.getComplaintState(), dto.getComplaintId());
  }

  @Override
  public Page<ComplaintEntity> findByComplaintCustomerPageDto(Pageable pageable, ComplaintCustomerPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new ComplaintCustomerPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    this.buildCustomerPageDto(dto);
    if (StringUtils.isBlank(dto.getCustomerCode())) {
      return null;
    }
    return this.complaintRepository.findByComplaintCustomerPageDto(pageable, dto);
  }

  @Override
  public Page<ComplaintEntity> findByConditions(Pageable pageable, ComplaintPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new ComplaintPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    return this.complaintRepository.findByConditions(pageable, dto);
  }

  /**
   * 在创建Complaint模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(ComplaintDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getComplaintType(), "缺失投诉类型");
    Validate.notBlank(dto.getComplaintTitle(), "缺失投诉标题");
    Validate.notBlank(dto.getComplaintContent(), "缺失投诉内容");
    Validate.notBlank(dto.getCustomerCode(), "缺失客户编码");
    Validate.notBlank(dto.getCustomerName(), "缺失客户名称");
    Validate.isTrue(dto.getComplaintTitle().length() <= 128, "投诉标题，在进行编辑时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(dto.getComplaintContent().length() <= 500, "投诉内容，在进行编辑时填入值超过了限定长度(500)，请检查!");
    Validate.isTrue(StringUtils.isBlank(dto.getComplainantContact()) || dto.getComplainantContact().length() < 30,
        "联系人，在进行编辑时填入值超过了限定长度(30)，请检查!");
  }
}
