package com.biz.crm.dms.business.itextpdf.local.service.internal;

import com.biz.crm.dms.business.itextpdf.local.service.FileUploadVoService;
import com.biz.crm.dms.business.itextpdf.local.service.ITextPdfVoService;
import com.biz.crm.dms.business.itextpdf.local.utils.MultipartFileUtils;
import com.biz.crm.vo.UploadVo;
import com.itextpdf.html2pdf.ConverterProperties;
import com.itextpdf.html2pdf.HtmlConverter;
import com.itextpdf.html2pdf.resolver.font.DefaultFontProvider;
import com.itextpdf.io.font.PdfEncodings;
import com.itextpdf.kernel.font.PdfFont;
import com.itextpdf.kernel.font.PdfFontFactory;
import com.itextpdf.kernel.geom.PageSize;
import com.itextpdf.kernel.pdf.PdfDocument;
import com.itextpdf.kernel.pdf.PdfWriter;
import com.itextpdf.layout.Document;
import com.itextpdf.layout.element.AreaBreak;
import com.itextpdf.layout.property.AreaBreakType;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.multipart.MultipartFile;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;

/**
 * @description: 文件转pdf
 * @author: rentao
 * @date: 2022/4/25 15:39
 */
@Slf4j
@Service
public class ITextPdfVoServiceImpl implements ITextPdfVoService {

  @Autowired(required = false)
  private FileUploadVoService fileUploadVoService;

  /**
   * file 转 pdf (直转)
   *
   * @param html
   */
  @Override
  public UploadVo createPdfByFile(MultipartFile html) {
    Validate.notNull(html, "请选择文件！");
    return this.fileUploadVoService.upload(html);
  }

  /**
   * html 转 pdf (直转)
   *
   * @param html
   * @return
   */
  @Override
  public UploadVo createPdfByHtml(String html) {
    Validate.notNull(html, "入参不能为空！");
    try {
      ByteArrayOutputStream stream = this.html2Pdf(html);
      MultipartFile multipartFile = MultipartFileUtils
          .getMultipartFile(new ByteArrayInputStream(stream.toByteArray()),
              System.currentTimeMillis() + ".pdf");
      UploadVo uploadVo = this.fileUploadVoService.upload(multipartFile);
      return uploadVo;
    } catch (IOException e) {
      log.error(e.getMessage(), e);
    }
    return null;
  }

  /**
   * @param html html页面字符串
   * @return
   * @throws FileNotFoundException
   * @throws IOException
   * @Description 将html转换为pdf文件
   */
  private ByteArrayOutputStream html2Pdf(String html) throws IOException {
    ConverterProperties props = new ConverterProperties();
    //设置字体
    this.setFont(props);
    //设置输出流
    ByteArrayOutputStream bao = new ByteArrayOutputStream();
    PdfWriter writer = new PdfWriter(bao);
    PdfDocument pdf = new PdfDocument(writer);
    pdf.setDefaultPageSize(PageSize.A4);
    //转成document进行操作
    Document document = HtmlConverter.convertToDocument(html, pdf, props);
    document.add(new AreaBreak(AreaBreakType.NEXT_PAGE));
    document.getRenderer().close();
    document.close();
    return bao;
  }

  /**
   * 设置中文编码
   *
   * @param props
   * @throws IOException
   */
  private void setFont(ConverterProperties props) throws IOException {
    DefaultFontProvider defaultFontProvider = new DefaultFontProvider(false, false, false);
    //设置正常字体
    InputStream inputStream = this.getClass().getClassLoader()
        .getResourceAsStream("font/SourceHanSansCN-Regular.ttf");
    byte[] bytes = MultipartFileUtils.toByteArray(inputStream);
    PdfFont regularFont = PdfFontFactory.createFont(bytes, PdfEncodings.IDENTITY_H, false);
    defaultFontProvider.addFont(regularFont.getFontProgram());
    //设置加粗字体
    InputStream boldInputStream = this.getClass().getClassLoader()
        .getResourceAsStream("font/SourceHanSansCN-Bold.ttf");
    byte[] boldBytes = MultipartFileUtils.toByteArray(boldInputStream);
    PdfFont boldFont = PdfFontFactory.createFont(boldBytes, PdfEncodings.IDENTITY_H, false);
    defaultFontProvider.addFont(boldFont.getFontProgram());
    props.setFontProvider(defaultFontProvider);
  }


}
