package com.biz.crm.dms.business.itextpdf.local.utils;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.springframework.http.MediaType;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.commons.CommonsMultipartFile;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Objects;

/**
 * @description: 转file工具类
 * @author: rentao
 * @date: 2022/4/25 15:40
 */
@Slf4j
public class MultipartFileUtils {
  /**
   * MultipartFile 转 File
   *
   * @param file
   * @throws Exception
   */
  public static File multipartFileToFile(MultipartFile file) {
    if (Objects.isNull(file)) {
      return null;
    }
    try (InputStream ins = file.getInputStream()) {
      File File = new File(file.getOriginalFilename());
      inputStreamToFile(ins, File);
      return File;
    } catch (IOException e) {
      log.error(e.getMessage(), e);
    }
    return null;
  }

  /**
   * 获取流文件
   * @param ins
   * @param file
   */
  private static void inputStreamToFile(InputStream ins, File file) {
    try (OutputStream os = new FileOutputStream(file)){
      int bytesRead = 0;
      byte[] buffer = new byte[8192];
      while ((bytesRead = ins.read(buffer, 0, 8192)) != -1) {
        os.write(buffer, 0, bytesRead);
      }
    } catch (IOException e) {
      log.error(e.getMessage(), e);
    } finally {
      try {
        ins.close();
      } catch (IOException e) {
        e.printStackTrace();
      }
    }
  }

  /**
   * 删除本地临时文件
   * @param file
   */
  public static void deleteTempFile(File file) {
    if (file != null) {
      File del = new File(file.toURI());
      if (!del.delete()) {
        throw new IllegalArgumentException("文件删除失败");
      }
    }
  }

  /**
   * 获取封装得MultipartFile
   *
   * @param inputStream inputStream
   * @param fileName    fileName
   * @return MultipartFile
   */
  public static MultipartFile getMultipartFile(InputStream inputStream, String fileName) {
    FileItem fileItem = createFileItem(inputStream, fileName);
    //CommonsMultipartFile是feign对multipartFile的封装，但是要FileItem类对象
    return new CommonsMultipartFile(fileItem);
  }


  /**
   * FileItem类对象创建
   *
   * @param inputStream inputStream
   * @param fileName    fileName
   * @return FileItem
   */
  public static FileItem createFileItem(InputStream inputStream, String fileName) {
    FileItemFactory factory = new DiskFileItemFactory(16, null);
    String textFieldName = "file";
    FileItem item = factory.createItem(textFieldName, MediaType.MULTIPART_FORM_DATA_VALUE, true, fileName);
    int bytesRead = 0;
    byte[] buffer = new byte[8192];
    //使用输出流输出输入流的字节
    try (OutputStream os = item.getOutputStream()) {
      while ((bytesRead = inputStream.read(buffer, 0, 8192)) != -1) {
        os.write(buffer, 0, bytesRead);
      }
    } catch (IOException e) {
      log.error("Stream copy exception", e);
      throw new IllegalArgumentException("文件上传失败");
    } finally {
      try {
        inputStream.close();
      } catch (IOException e) {
        e.printStackTrace();
      }
    }
    return item;
  }

  /**
   * 输入流转成byte
   * @param input
   * @return
   */
  public static byte[] toByteArray(InputStream input) {
    try (ByteArrayOutputStream output = new ByteArrayOutputStream();) {
      byte[] buffer = new byte[1024 * 4];
      int n = 0;
      while (-1 != (n = input.read(buffer))) {
        output.write(buffer, 0, n);
      }
      return output.toByteArray();
    } catch (IOException e) {
      e.printStackTrace();
    } finally {
      try {
        input.close();
      } catch (IOException e) {
        e.printStackTrace();
      }
    }
    return null;
  }
}
