package com.biz.crm.dms.business.order.cart.local.controller;

import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.dms.business.order.cart.local.entity.OrderCartEntity;
import com.biz.crm.dms.business.order.cart.local.helper.OrderCartHelper;
import com.biz.crm.dms.business.order.cart.local.service.OrderCartService;
import com.biz.crm.dms.business.order.cart.sdk.constant.OrderCartConstant;
import com.biz.crm.dms.business.order.cart.sdk.dto.OrderCartDto;
import com.biz.crm.dms.business.order.cart.sdk.dto.OrderCartUpdateDto;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @Project crm-dms
 * @PackageName com.biz.crm.dms.business.order.cart.local.controller
 * @ClassName OrderCartController
 * @Author YangWei
 * @Date 2022/4/1 下午3:48
 * @Description 购物车控制器
 */
@Slf4j
@RestController
@RequestMapping("/v1/order/orderCart")
@Api(tags = "商城订单: OrderCart: 购物车")
public class OrderCartController {

  @Autowired(required = false)
  private OrderCartService orderCartService;

  @Autowired(required = false)
  private RedisMutexService redisMutexService;

  @Autowired(required = false)
  private OrderCartHelper orderCartHelper;

  /**
   * 添加商品到购物车
   *
   * @param dto 参数dto
   * @return 添加商品到购物车
   */
  @ApiOperation(value = "添加商品到购物车")
  @PostMapping("/addBatch")
  public Result addBatch(@RequestBody OrderCartDto dto) {
    String lockKey = this.orderCartHelper.buildCashLockKeyByCustomerCode();
    Validate.isTrue(this.redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, OrderCartConstant.TRY_LOCK_OUT_TIME)
        , OrderCartConstant.LOCK_FAIL_MESSAGE);
    try {
      this.orderCartService.addBatch(dto);
      return Result.ok("添加成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }finally {
      this.redisMutexService.unlock(lockKey);
    }
  }

  /**
   * 删除购物车中的商品
   *
   * @param ids 公告id集合
   * @return 操作结果
   */
  @ApiOperation(value = "删除购物车中的商品")
  @DeleteMapping("/deleteBatch")
  public Result deleteBatch(@RequestParam("ids") List<String> ids) {
    String lockKey = this.orderCartHelper.buildCashLockKeyByCustomerCode();
    Validate.isTrue(this.redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, OrderCartConstant.TRY_LOCK_OUT_TIME)
        , OrderCartConstant.LOCK_FAIL_MESSAGE);
    try {
      this.orderCartService.deleteBatch(ids);
      return Result.ok("删除成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }finally {
      this.redisMutexService.unlock(lockKey);
    }
  }
}