package com.biz.crm.dms.business.order.cart.local.controller;

import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.dms.business.order.cart.local.helper.OrderCartHelper;
import com.biz.crm.dms.business.order.cart.sdk.constant.OrderCartConstant;
import com.biz.crm.dms.business.order.cart.sdk.dto.OrderCartCustomerDto;
import com.biz.crm.dms.business.order.cart.sdk.dto.OrderCartUpdateDto;
import com.biz.crm.dms.business.order.cart.sdk.service.OrderCartVoService;
import com.biz.crm.dms.business.order.cart.sdk.vo.OrderCartVo;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @Project crm-dms
 * @PackageName com.biz.crm.dms.business.order.cart.local.controller
 * @ClassName OrderCartVoController
 * @Author YangWei
 * @Date 2022/4/2 上午11:07
 * @Description 购物车Vo控制器
 */
@Slf4j
@RestController
@RequestMapping("/v1/order/orderCart")
@Api(tags = "商城订单: OrderCartVo: 购物车")
public class OrderCartVoController {

  @Autowired(required = false)
  private OrderCartVoService orderCartVoService;

  @Autowired(required = false)
  private RedisMutexService redisMutexService;

  @Autowired(required = false)
  private OrderCartHelper orderCartHelper;

  /**
   * 商城购物车列表
   * @param dto
   * @return
   */
  @ApiOperation(value = "商城购物车列表")
  @GetMapping("/findByOrderCartCustomerDto")
  public Result<List<OrderCartVo>> findByOrderCartCustomerDto(@ApiParam(name = "OrderCartCustomerDto", value = "参数Dto") OrderCartCustomerDto dto) {
    try {
      return Result.ok(this.orderCartVoService.findByOrderCartCustomerDto(dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 修改购物车中的商品数量
   * @param dto
   * @return {@link Result< OrderCartVo>}
   */
  @ApiOperation(value = "修改购物车中的商品数量")
  @PatchMapping("/updateQuantity")
  public Result<OrderCartVo> updateQuantity(@RequestBody OrderCartUpdateDto dto) {
    String lockKey = this.orderCartHelper.buildCashLockKeyByCustomerCode(dto.getCustomerCode());
    Validate.isTrue(this.redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, OrderCartConstant.TRY_LOCK_OUT_TIME)
        , OrderCartConstant.LOCK_FAIL_MESSAGE);
    try {
      OrderCartVo orderCartVo = this.orderCartVoService.updateQuantity(dto);
      return Result.ok(orderCartVo);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }finally {
      this.redisMutexService.unlock(lockKey);
    }
  }

  /**
   * 修改购物车中的商品数量(小程序专用)
   * @param dto
   * @return {@link Result< OrderCartVo>}
   */
  @ApiOperation(value = "修改购物车中的商品数量")
  @PostMapping("/updateQuantityTurnPatch")
  public Result<OrderCartVo> updateQuantityTurnPatch(@RequestBody OrderCartUpdateDto dto) {
    String lockKey = this.orderCartHelper.buildCashLockKeyByCustomerCode(dto.getCustomerCode());
    Validate.isTrue(this.redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, OrderCartConstant.TRY_LOCK_OUT_TIME)
        , OrderCartConstant.LOCK_FAIL_MESSAGE);
    try {
      OrderCartVo orderCartVo = this.orderCartVoService.updateQuantity(dto);
      return Result.ok(orderCartVo);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }finally {
      this.redisMutexService.unlock(lockKey);
    }
  }
}