package com.biz.crm.dms.business.order.cart.local.helper;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleListTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.list.dto.ValidateAllowSaleProductDto;
import com.biz.crm.dms.business.allow.sale.sdk.list.service.AllowSaleListVoService;
import com.biz.crm.dms.business.order.cart.local.entity.OrderCartEntity;
import com.biz.crm.dms.business.order.cart.sdk.constant.OrderCartConstant;
import com.biz.crm.dms.business.order.cart.sdk.dto.OrderCartDto;
import com.biz.crm.dms.business.order.cart.sdk.dto.OrderCartProductDto;
import com.biz.crm.dms.business.order.cart.sdk.vo.OrderCartVo;
import com.biz.crm.dms.business.order.common.sdk.enums.ItemTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCategoryEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.sdk.dto.OrderDetailPreviewDto;
import com.biz.crm.dms.business.order.sdk.dto.OrderPreviewDto;
import com.biz.crm.dms.business.order.sdk.service.OrderPreviewService;
import com.biz.crm.dms.business.order.sdk.vo.OrderDetailPreviewVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderPreviewVo;
import com.biz.crm.mdm.business.price.sdk.dto.FindPriceDto;
import com.biz.crm.mdm.business.price.sdk.enums.FindPriceUserTypeEnum;
import com.biz.crm.mdm.business.price.sdk.service.PriceModelVoService;
import com.biz.crm.mdm.business.price.sdk.vo.PriceModelVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Project crm-dms
 * @PackageName com.biz.crm.dms.business.order.cart.local.helper
 * @ClassName OrderCartHelper
 * @Author YangWei
 * @Date 2022/4/1 下午5:37
 * @Description 购物车助手
 */
@Slf4j
@Component
public class OrderCartHelper {

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private OrderPreviewService orderPreviewService;

  @Autowired(required = false)
  private ProductVoService productVoService;

  @Autowired(required = false)
  private PriceModelVoService priceModelVoService;

  /**
   * 通过客户编码租户编码构建购物车锁
   * @return
   */
  public String buildCashLockKeyByCustomerCode(String customerCode) {
    AbstractCrmUserIdentity loginUser = loginUserService.getAbstractLoginUser();
    Validate.isTrue(!ObjectUtils.isEmpty(loginUser),"登录信息错误");
    String tenantCode = loginUser.getTenantCode();
    Validate.notBlank(customerCode,"客户编码不能为空");
    Validate.notBlank(tenantCode,"租户编码不能为空");
    return String.format(OrderCartConstant.CREATE_LOCK_KEY, customerCode,tenantCode);
  }

  /**
   * 添加商品校验
   * @param dto
   */
  public void createValidation(OrderCartDto dto) {
    String customerCode = dto.getCustomerCode();
    Validate.isTrue(!ObjectUtils.isEmpty(dto),"传入数据对象不能为空");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getOrderCartProductDtoList()),"商品集合数据不能为空");
    AbstractCrmUserIdentity loginUser = loginUserService.getAbstractLoginUser();
    Validate.isTrue(!ObjectUtils.isEmpty(loginUser),"登录信息错误");
    dto.setCustomerCode(customerCode);
    String account = loginUser.getAccount();
    dto.setAccount(account);
    Validate.notBlank(customerCode,"客户编码不能为空");
    Validate.notBlank(account,"登录账号不能为空");
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.getOrderCartProductDtoList().forEach(o -> {
      Validate.notBlank(o.getSpuCode(),"SPU商品编码不能为空");
      Validate.notBlank(o.getProductCode(),"SKU商品编码不能为空");
      Validate.isTrue(!ObjectUtils.isEmpty(o.getQuantity()),"数量不能为空");
      Validate.isTrue(o.getQuantity().compareTo(0) > 0,"数量必须大于零");
    });
  }

  /**
   * 构建新增数据
   * @param addMap
   * @param dto
   * @param cartProductDto
   */
  public void buildAddEntity(Map<String, OrderCartEntity> addMap, OrderCartDto dto, OrderCartProductDto cartProductDto) {
    OrderCartEntity entity = new OrderCartEntity();
    entity.setCustomerCode(dto.getCustomerCode());
    entity.setAccount(dto.getAccount());
    entity.setSpuCode(cartProductDto.getSpuCode());
    entity.setTenantCode(dto.getTenantCode());
    entity.setProductCode(cartProductDto.getProductCode());
    entity.setQuantity(cartProductDto.getQuantity());
    addMap.put(cartProductDto.getProductCode(),entity);
  }

  /**
   * 构建修改对象
   * @param updateMap
   * @param existEntity
   * @param cartProductDto
   */
  public void buildUpdateQuantity(Map<String, OrderCartEntity> updateMap,OrderCartEntity existEntity, OrderCartProductDto cartProductDto) {
    Integer quantity = existEntity.getQuantity();
    Integer addQuantity = cartProductDto.getQuantity();
    existEntity.setQuantity(quantity + addQuantity);
    updateMap.put(cartProductDto.getProductCode(),existEntity);
  }

  @Autowired(required = false)
  private AllowSaleListVoService allowSaleListVoService;

  /**
   * 找到允许出售产品Code
   *
   * @param customerCode 客户代码
   * @param productCodes 产品代码
   * @return {@link List}<{@link String}>
   */
  private Set<String> findAllowSaleProductCodes(String customerCode, List<String> productCodes) {
    ValidateAllowSaleProductDto dto = new ValidateAllowSaleProductDto();
    dto.setBusinessCode(customerCode);
    dto.setListType(AllowSaleListTypeEnums.CUSTOMER.getCode());
    dto.setProductCodeList(productCodes);
    List<String> list = this.allowSaleListVoService.findAllowSaleProductCodes(dto);
    return Sets.newHashSet(list);
  }

  /**
   * 构建商品信息
   * @param cartVos
   */
  public void buildProductInfo(List<OrderCartVo> cartVos, String customerCode) {
    if(CollectionUtils.isEmpty(cartVos)){
      return;
    }
    JSONObject loginUserJson = loginUserService.getLoginUserJson();
    // 获取sku的信
    List<String> productCodes = cartVos.stream().map(OrderCartVo::getProductCode).collect(Collectors.toList());
    List<ProductVo> detailsByIdsOrProductCodes = productVoService.findDetailsByIdsOrProductCodes(null, productCodes);
    Validate.isTrue(!CollectionUtils.isEmpty(detailsByIdsOrProductCodes), "没有查询到商品sku的详细信息");
    Map<String, ProductVo> productVoMap = detailsByIdsOrProductCodes.stream()
        .collect(Collectors.toMap(ProductVo::getProductCode, Function.identity()));
    // 获取单价
    FindPriceDto findPriceDto = new FindPriceDto();
    findPriceDto.setUserType(FindPriceUserTypeEnum.CUSTOMER.getDictCode());
    findPriceDto.setUserCode(customerCode);
    findPriceDto.setProductCodeSet(Sets.newHashSet(productCodes));
    Map<String, PriceModelVo> price = this.priceModelVoService.findPrice(findPriceDto);
    // 查询可购的商品
    Set<String> allowSaleProductCodes = this.findAllowSaleProductCodes(customerCode, productCodes);
    cartVos.forEach(item -> {
      ProductVo productVo = productVoMap.get(item.getProductCode());
      item.setProductName(productVo.getProductName());
      item.setUnite(productVo.getSaleUnit());
      item.setSpec(productVo.getSpec());
      item.setPrimaryPictureUrl(productVo.getPrimaryPictureUrl());
      item.setPictureMediaList(productVo.getPictureMediaList());
      item.setVideoMediaList(productVo.getVideoMediaList());
      item.setIsShelf(productVo.getIsShelf());
      item.setIsAllowsale(allowSaleProductCodes.contains(item.getProductCode()));
      PriceModelVo priceModelVo = price.get(item.getProductCode());
      item.setPresetUnitPrice(Optional.ofNullable(priceModelVo).orElse(new PriceModelVo()).getPrice());
    });
    Map<String, OrderCartVo> map = new HashMap<>(cartVos.size());
    for (OrderCartVo cartVo : cartVos) {
      String productCode = cartVo.getProductCode();
      OrderCartVo orderCartVo = map.get(productCode);
      if (ObjectUtils.isEmpty(orderCartVo)) {
        map.put(productCode, cartVo);
      } else {
        Integer quantity = orderCartVo.getQuantity();
        Integer quantity1 = cartVo.getQuantity();
        cartVo.setQuantity(quantity + quantity1);
        map.put(productCode, cartVo);
      }
    }
    cartVos = Lists.newArrayList(map.values());
  }

  /**
   * 构建促销政策
   * @param cartVos
   */
  public void buildPolicyInfo(List<OrderCartVo> cartVos) {
    if(CollectionUtils.isEmpty(cartVos)){
      return;
    }
    OrderCartVo orderCartVo = cartVos.get(0);
    //执行接口参数
    OrderPreviewDto orderPreviewDto = new OrderPreviewDto();
    orderPreviewDto.setRelateCode(orderCartVo.getCustomerCode());
    orderPreviewDto.setOrderCategory(OrderCategoryEnum.SALES_ORDER.getDictCode());
    orderPreviewDto.setOrderType(OrderTypeEnum.STANDARD.getDictCode());
    //商品信息
    List<OrderDetailPreviewDto> orderDetails = Lists.newArrayList();
    cartVos.forEach(o -> {
      OrderDetailPreviewDto previewDto = new OrderDetailPreviewDto();
      previewDto.setGoodsCode(o.getProductCode());
      previewDto.setQuantity(new BigDecimal(o.getQuantity()));
      previewDto.setItemType(ItemTypeEnum.NORMAL_GOODS.getDictCode());
      previewDto.setIsAutoHit(true);
      previewDto.setSalesAmount(new BigDecimal(1200));
      orderDetails.add(previewDto);
    });
    orderPreviewDto.setOrderDetails(orderDetails);
    OrderPreviewVo orderPreviewVo = orderPreviewService.handlePreview(orderPreviewDto);
    String cityCode = orderPreviewVo.getCityCode();
    Map<String, OrderDetailPreviewVo> itemMap = orderPreviewVo.getOrderDetails().stream()
        .collect(Collectors.toMap(OrderDetailPreviewVo::getGoodsCode, Function.identity()));
    cartVos.forEach(item -> {
      OrderDetailPreviewVo previewVo = itemMap.get(item.getProductCode());
      buildItem(item,previewVo);
    });
  }

  /**
   * 构建行信息
   * @param item
   * @param previewVo
   */
  private void buildItem(OrderCartVo item, OrderDetailPreviewVo previewVo) {
    item.setOrderDetailPays(previewVo.getOrderDetailPays());
    item.setSalesAmount(previewVo.getSalesAmount());
    item.setShouldPaymentAmount(previewVo.getShouldPaymentAmount());
    item.setPresetUnitPrice(previewVo.getPresetUnitPrice());
    item.setActualUnitPrice(previewVo.getActualUnitPrice());
    item.setAverageUnitPrice(previewVo.getAverageUnitPrice());
  }

}