package com.biz.crm.dms.business.order.cart.local.service.internal;

import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.order.cart.local.entity.OrderCartEntity;
import com.biz.crm.dms.business.order.cart.local.helper.OrderCartHelper;
import com.biz.crm.dms.business.order.cart.local.repository.OrderCartRepository;
import com.biz.crm.dms.business.order.cart.local.service.OrderCartService;
import com.biz.crm.dms.business.order.cart.sdk.dto.OrderCartDto;
import com.biz.crm.dms.business.order.cart.sdk.dto.OrderCartProductDto;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Project crm-dms
 * @PackageName com.biz.crm.dms.business.order.cart.local.service.internal
 * @ClassName OrderCartServiceImpl
 * @Author YangWei
 * @Date 2022/4/1 下午4:07
 * @Description 购物车内部接口实现
 */
@Slf4j
@Service("orderCartService")
public class OrderCartServiceImpl implements OrderCartService {

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private OrderCartRepository orderCartRepository;

  @Autowired(required = false)
  private OrderCartHelper orderCartHelper;

  /**
   * 添加商品到购物车
   * @param dto
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public void addBatch(OrderCartDto dto) {
    //校验
    this.orderCartHelper.createValidation(dto);
    //根据客户以及商品编码查询是否存在
    List<String> productCodes = dto.getOrderCartProductDtoList().stream()
        .map(OrderCartProductDto::getProductCode).collect(Collectors.toList());
    List<String> spuCodes = dto.getOrderCartProductDtoList().stream()
        .map(OrderCartProductDto::getSpuCode).collect(Collectors.toList());
    List<OrderCartEntity> existEntities = this.orderCartRepository
        .findByCustomerCodeAndSpuCodeAndProductCodes(dto.getCustomerCode(),productCodes);
    Map<String, OrderCartEntity> existMap = existEntities.stream()
        .collect(Collectors.toMap(OrderCartEntity::getProductCode, Function.identity()));
    //新增集合
    Map<String, OrderCartEntity> addMap = Maps.newHashMap();
    //更新集合
    Map<String, OrderCartEntity> updateMap = Maps.newHashMap();
    dto.getOrderCartProductDtoList().forEach(o -> {
      String key =o.getProductCode();
      OrderCartEntity existEntity = existMap.get(key);
      if(ObjectUtils.isEmpty(existEntity)){
        this.orderCartHelper.buildAddEntity(addMap,dto,o);
      }else {
        //商品已经在购物车里面就调整数量
        this.orderCartHelper.buildUpdateQuantity(updateMap,existEntity,o);
      }
    });
    if (!addMap.isEmpty()) {
      this.orderCartRepository.saveBatch(addMap.values());
    }
    if (!updateMap.isEmpty()) {
      this.orderCartRepository.saveOrUpdateBatch(updateMap.values());
    }
  }

  /**
   * 删除购物车中的商品
   * @param ids
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids),"购物车ID集合不能为空");
    this.orderCartRepository.deleteBatch(ids);
  }
}