package com.biz.crm.dms.business.order.config.local.repository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.order.config.local.entity.OrderConfigEntity;
import com.biz.crm.dms.business.order.config.local.mapper.OrderConfigMapper;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * 订单配置表的数据库访问类 {@link OrderConfigEntity}
 *
 * @author pengxi
 * @date 2021/12/30
 */
@Component
public class OrderConfigRepository extends ServiceImpl<OrderConfigMapper, OrderConfigEntity> {

  /**
   * 根据订单配置编码获取详情
   *
   * @param tenantCode 租户编码
   * @param configCode 订单配置编码
   * @return OrderConfigEntity
   */
  public OrderConfigEntity findByConfigCode(String tenantCode, String configCode) {
    LambdaQueryWrapper<OrderConfigEntity> wrapper = Wrappers.lambdaQuery();
    wrapper.eq(OrderConfigEntity::getTenantCode, tenantCode);
    wrapper.eq(OrderConfigEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
    wrapper.eq(OrderConfigEntity::getEnableStatus, EnableStatusEnum.ENABLE.getCode());
    wrapper.eq(OrderConfigEntity::getConfigCode, configCode);
    wrapper.last("limit 1");
    return this.baseMapper.selectOne(wrapper);
  }

  /**
   * 根据订单配置编码获取详情
   *
   * @param tenantCode 租户编码
   * @param orderType 订单类型
   * @return OrderConfigEntity
   */
  public OrderConfigEntity findByDefaultAndOrderType(String tenantCode, String orderType) {
    LambdaQueryWrapper<OrderConfigEntity> wrapper = Wrappers.lambdaQuery();
    wrapper.eq(OrderConfigEntity::getTenantCode, tenantCode);
    wrapper.eq(OrderConfigEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
    wrapper.eq(OrderConfigEntity::getEnableStatus, EnableStatusEnum.ENABLE.getCode());
    wrapper.eq(OrderConfigEntity::getIsDefault, Boolean.TRUE);
    wrapper.eq(OrderConfigEntity::getOrderType, orderType);
    wrapper.last("limit 1");
    return this.baseMapper.selectOne(wrapper);
  }
  
  /**
   * 根据id集合获取详情集合
   *
   * @param ids ID集合
   * @return List<OrderConfigEntity>
   */
  public List<OrderConfigEntity> findByIds(List<String> ids) {
    return this.lambdaQuery()
        .eq(OrderConfigEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .in(OrderConfigEntity::getId, ids)
        .list();
  }
  
  /**
   * 批量更新启用禁用状态
   *
   * @param ids          ID集合
   * @param enableStatus 启用禁用状态
   */
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enableStatus) {
    this.lambdaUpdate()
        .in(OrderConfigEntity::getId, ids)
        .set(OrderConfigEntity::getEnableStatus, enableStatus.getCode())
        .update();
  }

  /**
   * 逻辑删除
   *
   * @param ids ID集合
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate().in(OrderConfigEntity::getId, ids)
        .set(OrderConfigEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

}
