package com.biz.crm.dms.business.order.feerate.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.order.feerate.fee.statistics.sdk.constant.OrderStatisticsElementConstant;
import com.biz.crm.dms.business.order.feerate.local.entity.DmsFeeRate;
import com.biz.crm.dms.business.order.feerate.local.repository.DmsFeeRateRepository;
import com.biz.crm.dms.business.order.feerate.local.service.DmsFeeRateService;
import com.biz.crm.dms.business.order.feerate.sdk.dto.DmsFeeRateDto;
import com.biz.crm.dms.business.order.feerate.sdk.dto.DmsOptimalFeePoolRateDto;
import com.biz.crm.dms.business.order.feerate.sdk.enums.FeeTypeEnum;
import com.biz.crm.dms.business.order.feerate.sdk.event.FeeRateEventListener;
import com.biz.crm.dms.business.order.feerate.sdk.strategy.DemensionStrategy;
import com.biz.crm.dms.business.order.feerate.sdk.vo.DmsFeeRateVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 费用比例(DmsFeeRate)
 *
 * @author rentao
 * @since 2021-12-29 17:53:57
 */
@Service("dmsFeeRateService")
@Slf4j
public class DmsFeeRateServiceImpl implements DmsFeeRateService {

  @Autowired(required = false) private DmsFeeRateRepository dmsFeeRateRepository;

  @Autowired(required = false) private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private List<FeeRateEventListener> feeRateEventListeners;

  @Autowired(required = false)
  private List<DemensionStrategy> demensionStrategies;

  /**
   * 费用池默认（费比使用）抿成
   */
  public static final String  FEE_POOL_DEFAULT_ELEMENT_NAME = "费用池默认";

  /**
   * 分页查询所有数据
   *
   * @param pageable
   * @param dmsFeeRateDto
   * @author rentao
   * @date
   */
  @Override
  public Page<DmsFeeRate> findByConditions(Pageable pageable, DmsFeeRateDto dmsFeeRateDto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(dmsFeeRateDto)) {
      dmsFeeRateDto = new DmsFeeRateDto();
    }
    return this.dmsFeeRateRepository.findByConditions(pageable, dmsFeeRateDto);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id
   * @author rentao
   * @date
   */
  @Override
  public DmsFeeRate findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.dmsFeeRateRepository.getById(id);
  }

  /**
   * 新增数据
   *
   * @param dmsFeeRateDto
   * @author rentao
   * @date
   */
  @Transactional
  @Override
  public DmsFeeRate create(DmsFeeRateDto dmsFeeRateDto) {
    this.createValidate(dmsFeeRateDto);
    DmsFeeRate dmsFeeRate =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dmsFeeRateDto, DmsFeeRate.class, HashSet.class, ArrayList.class);
    this.dmsFeeRateRepository.save(dmsFeeRate);
    dmsFeeRateDto.setId(dmsFeeRate.getId());
    if (!CollectionUtils.isEmpty(feeRateEventListeners)) {
      DmsFeeRateVo dmsFeeRateVo =
          this.nebulaToolkitService.copyObjectByWhiteList(
              dmsFeeRateDto, DmsFeeRateVo.class, HashSet.class, ArrayList.class);
      feeRateEventListeners.forEach(
          listener -> {
            listener.onCreate(dmsFeeRateVo);
          });
    }
    return dmsFeeRate;
  }

  /**
   * 编辑数据
   *
   * @param dmsFeeRateDto
   * @author rentao
   * @date
   */
  @Transactional
  @Override
  public DmsFeeRate update(DmsFeeRateDto dmsFeeRateDto) {
    this.updateValidate(dmsFeeRateDto);
    DmsFeeRate dmsFeeRate =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dmsFeeRateDto, DmsFeeRate.class, HashSet.class, ArrayList.class);
    this.dmsFeeRateRepository.updateById(dmsFeeRate);
    if (!CollectionUtils.isEmpty(feeRateEventListeners)) {
      DmsFeeRateVo dmsFeeRateVo =
          this.nebulaToolkitService.copyObjectByWhiteList(
              dmsFeeRateDto, DmsFeeRateVo.class, HashSet.class, ArrayList.class);
      feeRateEventListeners.forEach(
          listener -> {
            listener.onUpdate(dmsFeeRateVo);
          });
    }
    return dmsFeeRate;
  }

  /**
   * 批量删除
   *
   * @param ids
   * @author rentao
   */
  @Transactional
  @Override
  public void delete(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "主键集合不能为空");
    this.dmsFeeRateRepository.removeByIds(ids);
  }

  @Transactional
  @Override
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    List<DmsFeeRate> entities = this.dmsFeeRateRepository.listByIds(ids);
    Validate.isTrue(
        !CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据启用个数不匹配");
    this.dmsFeeRateRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
    // 启用费比事件通知
    if (!CollectionUtils.isEmpty(feeRateEventListeners)) {
      List<DmsFeeRateVo> voList =
          (List<DmsFeeRateVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  entities, DmsFeeRate.class, DmsFeeRateVo.class, HashSet.class, ArrayList.class);
      feeRateEventListeners.forEach(
          listener -> {
            listener.onEnable(voList);
          });
    }
  }

  @Transactional
  @Override
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    List<DmsFeeRate> entities = this.dmsFeeRateRepository.listByIds(ids);
    Validate.isTrue(
        !CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据禁用个数不匹配");
    this.dmsFeeRateRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
    // 禁用费比事件通知
    if (!CollectionUtils.isEmpty(feeRateEventListeners)) {
      List<DmsFeeRateVo> voList =
          (List<DmsFeeRateVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  entities, DmsFeeRate.class, DmsFeeRateVo.class, HashSet.class, ArrayList.class);
      feeRateEventListeners.forEach(
          listener -> {
            listener.onDisable(voList);
          });
    }
  }

  /**
   * 新增验证
   *
   * @param dmsFeeRateDto
   */
  private void createValidate(DmsFeeRateDto dmsFeeRateDto) {
    this.baseValidate(dmsFeeRateDto);
    dmsFeeRateDto.setId(null);
    dmsFeeRateDto.setTenantCode(TenantUtils.getTenantCode());
    dmsFeeRateDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dmsFeeRateDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    List<DmsFeeRateVo> dmsFeeRates = this.getDmsFeeRate(dmsFeeRateDto);
    Validate.isTrue(CollectionUtils.isEmpty(dmsFeeRates), "费用维度已存在数据。");
  }

  /**
   * 编辑验证֤
   *
   * @param dmsFeeRateDto
   */
  private void updateValidate(DmsFeeRateDto dmsFeeRateDto) {
    this.baseValidate(dmsFeeRateDto);
    Validate.notBlank(dmsFeeRateDto.getId(), "费用比例id不能为空");
    List<DmsFeeRateVo> dmsFeeRates = this.getDmsFeeRate(dmsFeeRateDto);
    if (!CollectionUtils.isEmpty(dmsFeeRates)) {
      Validate.isTrue(dmsFeeRates.size() < 2, "费用维度已存在数据");
      Validate.isTrue(
          dmsFeeRates.size() == 1 && dmsFeeRateDto.getId().equals(dmsFeeRates.get(0).getId()),
          "费用维度已存在数据");
    }
  }

  /**
   * 基础验证֤
   *
   * @param dmsFeeRate
   */
  private void baseValidate(DmsFeeRateDto dmsFeeRate) {
    Validate.notNull(dmsFeeRate, "费用比例传入参数不能为空");
    Validate.notNull(dmsFeeRate.getDemension(), "费用比例生效维度不能为空");
    Validate.notBlank(dmsFeeRate.getFeeStatisticsCode(), "费用比例费用统计范围编码不能为空");
    Validate.notBlank(dmsFeeRate.getFeeStatisticsName(), "费用比例费用统计范围名称不能为空");
    Validate.notNull(dmsFeeRate.getFeeType(), "费用比例费比类型");
    Validate.notNull(FeeTypeEnum.getByKey(dmsFeeRate.getFeeType()), "费比类型传入有误");
    Validate.notNull(dmsFeeRate.getRate(), "费用比例 （%）不能为空");
    Validate.isTrue(!CollectionUtils.isEmpty(demensionStrategies),"生效维度没有策略实现！");
    List<DemensionStrategy> strategies = demensionStrategies.stream().filter(
        demensionStrategy -> demensionStrategy.getDemensionCode().equals(dmsFeeRate.getDemension())).collect(Collectors.toList());
    Validate.isTrue(!CollectionUtils.isEmpty(strategies),"生效维度传入数据有误！");
    if(FeeTypeEnum.ENTIRE_ORDER.getKey().equals(dmsFeeRate.getFeeType())){
      Validate.notBlank(dmsFeeRate.getSaleStatisticsCode(), "费用比例销售额统计范围编码不能为空");
      Validate.notBlank(dmsFeeRate.getSaleStatisticsName(), "费用比例销售额统计范围名称不能为空");
    }else {
      dmsFeeRate.setSaleStatisticsCode(OrderStatisticsElementConstant.FEE_POOL_DEFAULT_ELEMENT);
      dmsFeeRate.setSaleStatisticsName(FEE_POOL_DEFAULT_ELEMENT_NAME);
    }
    DemensionStrategy demensionStrategy = strategies.get(0);
    dmsFeeRate.setDemensionName(demensionStrategy.getDemensionName());
    demensionStrategy.validateDmsFeeRate(dmsFeeRate);
  }

  /**
   * 查询唯一性数据
   *
   * @param dmsFeeRateDto
   */
  private List<DmsFeeRateVo> getDmsFeeRate(DmsFeeRateDto dmsFeeRateDto) {
    List<DemensionStrategy> strategies = demensionStrategies.stream().filter(
        demensionStrategy -> demensionStrategy.getDemensionCode().equals(dmsFeeRateDto.getDemension())).collect(Collectors.toList());
    DemensionStrategy demensionStrategy = strategies.get(0);
    DmsOptimalFeePoolRateDto dmsOptimalFeePoolRateDto = new DmsOptimalFeePoolRateDto();
    dmsOptimalFeePoolRateDto.setSaleStatisticsCode(dmsFeeRateDto.getSaleStatisticsCode());
    dmsOptimalFeePoolRateDto.setFeeStatisticsCode(dmsFeeRateDto.getFeeStatisticsCode());
    dmsOptimalFeePoolRateDto.setBusinessCode(dmsFeeRateDto.getBusinessCode());
    FeeTypeEnum feeTypeEnum = FeeTypeEnum.getByKey(dmsFeeRateDto.getFeeType());
    return demensionStrategy.getByDemension(dmsOptimalFeePoolRateDto,feeTypeEnum);
  }
}
