package com.biz.crm.dms.business.order.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.dms.business.order.local.entity.Order;
import com.biz.crm.dms.business.order.local.service.OrderService;
import com.biz.crm.dms.business.order.sdk.dto.CustomerOrderPageDto;
import com.biz.crm.dms.business.order.sdk.dto.OrderPaginationDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * 订单主信息(Order)表相关的http接口
 *
 * @author makejava
 * @since 2022-01-08 15:13:04
 */
@RestController
@RequestMapping("/v1/order/order")
@Slf4j
@Api(tags = "订单模块：Order：订单主信息")
public class OrderController {
  /**
   * 服务对象
   */
  @Autowired(required = false)
  private OrderService orderService;

  /**
   * 分页查询所有数据
   *
   * @param pageable 分页对象
   * @param order    查询实体
   * @return 所有数据
   */
  @ApiOperation(value = "分页查询所有数据")
  @GetMapping("findByConditions")
  public Result<Page<Order>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                              @ApiParam(name = "order", value = "订单主信息") OrderPaginationDto order) {
    try {
      Page<Order> page = this.orderService.findByConditions(pageable, order);
      return Result.ok(page);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 分页查询当前客户的数据
   *
   * @param pageable             分页对象
   * @param customerOrderPageDto 查询实体
   * @return 所有数据
   */
  @ApiOperation(value = "分页查询当前客户的数据")
  @GetMapping("findByCustomerOrderPageDto")
  public Result<Page<Order>> findByCustomerOrderPageDto(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                        @ApiParam(name = "order", value = "订单主信息") CustomerOrderPageDto customerOrderPageDto) {
    try {
      Page<Order> page = this.orderService.findByCustomerOrderPageDto(pageable, customerOrderPageDto);
      return Result.ok(page);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * hf:
   * 新增数据
   * --同时作为暂存接口使用
   *
   * @param order 实体对象
   * @return 新增结果
   */
  @ApiOperation(value = "新增数据--暂存功能使用")
  @PostMapping
  public Result<Order> create(@ApiParam(name = "order", value = "订单主信息") @RequestBody Order order) {
    try {
      Order result = this.orderService.create(order);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 修改数据
   *
   * @param order 实体对象
   * @return 修改结果
   */
  @ApiOperation(value = "修改数据")
  @PatchMapping
  public Result<Order> update(@ApiParam(name = "order", value = "订单主信息") @RequestBody Order order) {
    try {
      Order result = this.orderService.update(order);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @DeleteMapping
  public Result<?> delete(@ApiParam(name = "ids", value = "主键集合") @RequestParam("ids") List<String> idList) {
    try {
      this.orderService.delete(idList);
      return Result.ok();
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 查询可退的订单（完全发货、部分发货）
   *
   * @param relateCode 关联编码 【客户编码】
   * @return 可退的订单
   */
  @ApiOperation(value = "查询可退的订单（完全发货、部分发货）")
  @GetMapping("findRefundableByCustomerCode")
  public Result<List<Order>> findRefundableByCustomerCode(@ApiParam(name = "relateCode", value = "关联编码 【客户编码】") String relateCode) {
    try {
      List<Order> list = this.orderService.findRefundableByCustomerCode(relateCode);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

}
