package com.biz.crm.dms.business.order.local.notifier;

import com.biz.crm.common.message.sdk.dto.SysMessageDto;
import com.biz.crm.common.message.sdk.dto.SysMessageItemDto;
import com.biz.crm.common.message.sdk.enums.MessageBusinessType;
import com.biz.crm.common.message.sdk.enums.MessageHandlerEnum;
import com.biz.crm.common.message.sdk.enums.MessageRelateType;
import com.biz.crm.common.message.sdk.enums.MessageTypeEnum;
import com.biz.crm.common.message.sdk.service.SendMessageVoService;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderStatusEnum;
import com.biz.crm.dms.business.order.local.entity.Order;
import com.biz.crm.dms.business.order.local.repository.OrderRepository;
import com.biz.crm.dms.business.order.sdk.dto.OrderEventDto;
import com.biz.crm.dms.business.order.sdk.event.OrderLogEventListener;
import com.biz.crm.dms.business.order.sdk.vo.OrderVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerDockingVo;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.customer.user.sdk.service.CustomerUserRelateCustomerVoService;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserRelateCustomerVo;
import com.biz.crm.mdm.business.user.sdk.service.UserInfoVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserInfoVo;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 系统消息之订单事件监听器
 *
 * @author hefan
 * @date 2022/07/12
 */
@Component
public class OrderLogEventListenerOfSystemMessage implements OrderLogEventListener {

  @Autowired(required = false)
  private SendMessageVoService sendMessageVoService;

  @Autowired(required = false)
  private CustomerUserRelateCustomerVoService customerUserRelateCustomerVoService;

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  @Autowired(required = false)
  private UserInfoVoService userInfoVoService;

  @Autowired(required = false)
  private OrderRepository orderRepository;

  @Override
  public void onCreate(OrderEventDto eventDto) {
    // 不需要实现
  }

  @Override
  public void onClose(OrderEventDto orderEventDto) {
    OrderVo newest = orderEventDto.getNewest();
    String relateCode = newest.getRelateCode();
    //
    SysMessageDto msg = new SysMessageDto();
    msg.setCode(MessageTypeEnum.CLOSE_ORDER.getDictCode());
    Map<String, List<SysMessageItemDto>> map = Maps.newHashMap();
    // 业务员信息
    CustomerVo customerVo = customerVoService.findDetailsByIdOrCode(null, relateCode);
    Validate.notNull(customerVo, "关闭订单时，发送系统消息，没有查到客户信息");
    List<CustomerDockingVo> dockingList = customerVo.getDockingList();
    if (!CollectionUtils.isEmpty(dockingList)) {
      Set<String> positionCodes = new HashSet<>(dockingList.size());
      for (CustomerDockingVo customerDockingVo : dockingList) {
        String positionCode = customerDockingVo.getPositionCode();
        positionCodes.add(positionCode);
      }
      List<UserInfoVo> userList = userInfoVoService.findByPositionCodes(positionCodes);
      if (!CollectionUtils.isEmpty(userList)) {
        List<SysMessageItemDto> userItems = Lists.newLinkedList();
        for (UserInfoVo userInfoVo : userList) {
          SysMessageItemDto dto1 = new SysMessageItemDto();
          dto1.setBusinessCode(userInfoVo.getUserName());
          dto1.setBusinessType(MessageBusinessType.BACKGROUND.getDictCode());
          dto1.setRelateCode(newest.getId());
          dto1.setRelateType(MessageRelateType.ORDER.getDictCode());
          Map<String, String> param1Map = Maps.newHashMap();
          param1Map.put("orderCode", newest.getOrderCode());
          dto1.setParamsMap(param1Map);
          userItems.add(dto1);
        }
        // k- CloseOrderUserMessageHandlerRegisterImpl->code
        map.put(MessageHandlerEnum.CLOSE_ORDER_USER.getDictCode(), userItems);
      }
    }
    // 客户用户
    String createAccount = newest.getCreateAccount();
    // - 查询经销商下的所有用户
    List<CustomerUserRelateCustomerVo> customerUserRelateCustomerVos = customerUserRelateCustomerVoService.findByCustomerCodes(Lists.newArrayList(relateCode));
    if (CollectionUtils.isEmpty(customerUserRelateCustomerVos)) {
      return;
    }
    Set<String> userNames = customerUserRelateCustomerVos.stream().map(CustomerUserRelateCustomerVo::getUserName).collect(Collectors.toSet());
    // 搜集消息
    List<SysMessageItemDto> customerItems = Lists.newLinkedList();
    // - 判断是否为客户下单
    boolean isCustomerOrder = userNames.contains(createAccount);
    if (isCustomerOrder) {
      // - 只给该客户用户发消息
      SysMessageItemDto dto2 = new SysMessageItemDto();
      dto2.setBusinessCode(createAccount);
      dto2.setBusinessType(MessageBusinessType.CUSTOMER.getDictCode());
      dto2.setRelateCode(newest.getId());
      dto2.setRelateType(MessageRelateType.ORDER.getDictCode());
      Map<String, String> param2Map = Maps.newHashMap();
      param2Map.put("orderCode", newest.getOrderCode());
      dto2.setParamsMap(param2Map);
      customerItems.add(dto2);
    } else {
      // - 给所有客户用户发消息
      for (String userName : userNames) {
        SysMessageItemDto dto2 = new SysMessageItemDto();
        dto2.setBusinessCode(userName);
        dto2.setBusinessType(MessageBusinessType.CUSTOMER.getDictCode());
        dto2.setRelateCode(newest.getId());
        dto2.setRelateType(MessageRelateType.ORDER.getDictCode());
        Map<String, String> param2Map = Maps.newHashMap();
        param2Map.put("orderCode", newest.getOrderCode());
        dto2.setParamsMap(param2Map);
        customerItems.add(dto2);
      }
    }
    // k- CloseOrderCustomerMessageHandlerRegisterImpl->code
    map.put(MessageHandlerEnum.CLOSE_ORDER_CUSTOMER.getDictCode(), customerItems);
    msg.setMap(map);
    // 发送系统消息
    this.sendMessageVoService.send(msg);
  }


  /**
   * 订单返货延迟10天，就每天生产一条消息
   * - 先1分钟发一个测试下
   * - 线上：0 0 1 * * ?
   * - 测试：0 0/1 * * * ?
   */
  @DynamicTaskService(cornExpression = "0 0 1 * * ?", taskDesc = "订单延迟发货的系统消息通知")
  public void deliveryDelay() {
    /**
     * - 查询未发货的订单 且 自提交日起已经过去10天的订单
     *
     */
    // 查询10天未发货的订单
    Date last10DaysStartTime = this.last10DaysStartTime();
    List<Order> waitShippedOrders = orderRepository.findByOrderStatusAndCreateTimeLe(OrderStatusEnum.WAIT_SHIPPED, last10DaysStartTime);
    if (CollectionUtils.isEmpty(waitShippedOrders)) {
      return;
    }
    // 查询客户信息
    List<String> customerCodes = waitShippedOrders.stream().map(Order::getRelateCode).collect(Collectors.toList());
    List<CustomerVo> customerVos = customerVoService.findByCustomerCodes(customerCodes);
    if (CollectionUtils.isEmpty(customerVos)) {
      return;
    }
    // 客户的对接人的岗位，客户与对接人岗位的映射，
    Set<String> positionCodes = new HashSet<>();
    Map<String, List<String>> customerCodeToPositionMap = new HashMap<>(customerVos.size());
    for (CustomerVo customerVo : customerVos) {
      List<CustomerDockingVo> dockingList = customerVo.getDockingList();
      if (CollectionUtils.isEmpty(dockingList)) {
        continue;
      }
      List<String> positionCodeList = Lists.newLinkedList();
      for (CustomerDockingVo dockingVo : dockingList) {
        positionCodes.add(dockingVo.getPositionCode());
        positionCodeList.add(dockingVo.getPositionCode());
      }
      customerCodeToPositionMap.put(customerVo.getCustomerCode(), positionCodeList);
    }
    // 用对接人的岗位查询对接人信息
    List<UserInfoVo> userList = userInfoVoService.findByPositionCodes(positionCodes);
    if (!CollectionUtils.isEmpty(userList)) {
      // 把对接人按照岗位分组
      Map<String, List<UserInfoVo>> positionToUserMap = userList.stream().collect(Collectors.groupingBy(UserInfoVo::getPositionCode));
      SysMessageDto msg = new SysMessageDto();
      // 消息类型编码  DeliveryDelayMessageRegisterImpl -》code
      msg.setCode(MessageTypeEnum.DELIVERY_DELAY.getDictCode());
      Map<String, List<SysMessageItemDto>> map = Maps.newHashMap();
      // 业务员信息: 遍历订单-》找到客户的对接人的岗位-》遍历岗位-》用岗位找到对接人-》给负责该订单的对接人发消息
      List<SysMessageItemDto> userItems = Lists.newLinkedList();
      for (Order order : waitShippedOrders) {
        String relateCode = order.getRelateCode();
        List<String> positionCodeList = customerCodeToPositionMap.get(relateCode);
        if (CollectionUtils.isEmpty(positionCodeList)) {
          continue;
        }
        for (String positionCode : positionCodeList) {
          List<UserInfoVo> userInfoVos = positionToUserMap.get(positionCode);
          if (CollectionUtils.isEmpty(userInfoVos)) {
            continue;
          }
          for (UserInfoVo userInfoVo : userInfoVos) {
            SysMessageItemDto dto1 = new SysMessageItemDto();
            dto1.setBusinessCode(userInfoVo.getUserName());
            dto1.setBusinessType(MessageBusinessType.BACKGROUND.getDictCode());
            dto1.setRelateCode(order.getId());
            dto1.setRelateType(MessageRelateType.ORDER.getDictCode());
            Map<String, String> param1Map = Maps.newHashMap();
            param1Map.put("orderCode", order.getOrderCode());
            dto1.setParamsMap(param1Map);
            userItems.add(dto1);
          }
        }
      }
      // k- DeliveryDelayUserMessageHandlerRegisterImpl->code
      map.put(MessageHandlerEnum.DELIVERY_DELAY_USER.getDictCode(), userItems);
      msg.setMap(map);
      // 发送系统消息
      this.sendMessageVoService.send(msg);
    }
  }

  /**
   * 10天前的开始时间
   */
  private Date last10DaysStartTime() {
    LocalDateTime of = LocalDateTime.of(LocalDate.now().minus(10L, ChronoUnit.DAYS), LocalTime.MIN);
    return Date.from(of.atZone(ZoneId.systemDefault()).toInstant());
  }


}
